/* Linux Prism II Stumbler - Utility Scan for 802_11 networks under Linux
 * 
 * File : $Name:  $
 * Project : WifiScanner (c) 2002 Herv Schauer Consultants
 * Usage : This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA).
 * 
 * Base code was from prismstumbler Jan Fernquist <Jan.B.Fernquist@telia.com>
 * and wlanctl from www.linux-wlan.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: scanner.c,v 1.32 2003/07/25 11:23:08 poggij Exp $
 */


#include <include.h>
#include <src/interface.h>
#include <src/driver.h>
#include <src/functions.h>
#include <src/analyse.h>
#include <src/scanner.h>
#include <src/crt_io.h>
#include <src/conversion.h>
#ifdef WITH_WIRETAP
#include <wtap.h>
#endif

/*
*/
void HelpAndBye (void);
void VersionAndBye (void);

static UINT8 ID[] = "$Id: scanner.c,v 1.32 2003/07/25 11:23:08 poggij Exp $";
/*
 * Use globals, ugly ... 
 */
ScanResult_t Res;
UINT8 DebugLevel;
WINDOW *Title_WND, *Panel_WND, *Sum_WND, *RealTime_WND;
Statistics_t Stats;
UINT8 CursesIsInUse = (UINT8) 0;	/* To know if curses environment is in use */
UINT8 TypeOfCard = WLAN_NG_CARD;	/* Type of driver to use */
#ifdef WITH_THREAD
/// MUTEX for lock screen, so only one func write to it at the same time
pthread_mutex_t screen_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_t thread_ChgChan, thread_RefreshRTW;
UINT8 stop_sniffing = (UINT8) 0;
#else
static UINT8 stop_sniffing = (UINT8) 0;
#endif

static UINT8 SingleChannel = (UINT8) 0;
/// number of miliseconds before the next channel is select
static unsigned int TimeToSleepBeforeChangeChannel;
/// Channel hop used
static UINT8 ChannelHop;


/**
 * Signal Handler
 **/
RETSIGTYPE
stop_signal (void)
{
  // We don't need to stop, if we know that we will stop
  if (!stop_sniffing) {
#ifdef WITH_THREAD
    // Clean and Force STOP the thread
    pthread_mutex_unlock (&screen_mutex);
    //pthread_kill( &thread_ChgChan, (int) 1);
    pthread_mutex_destroy (&screen_mutex);
#endif
    warning ("Received CTRL-C - sniffing aborted\n\n");
    stop_sniffing = (UINT8) 1;
  }
}

/*
 * Help
 */
void
HelpAndBye (void)
{
  fprintf (stderr, "%s", HELPTEXT);
  fprintf (stderr, "\nPlease send Bug report to : %s\n\n", WIFISCANNER_BUG);
  exit (EXIT_SUCCESS);
}

#ifdef WITH_THREAD
void
ChangeChannel_th (void *ptr1)
{
  struct timeb binary_now, TimeScann;
  char *devname;
  //struct timespec delay;

  ftime (&TimeScann);
  devname = (char *) ptr1;
  /* For other OS than Linux :
     delay.tv_sec = 0;
     delay.tv_nsec = TimeToSleepBeforeChangeChannel * 100000;
   */

  while (stop_sniffing == 0) {
    /*** Time trigger for change channel ***/
    // Save time, so we change channel only when it's time
    ftime (&binary_now);
    // is it time to change channel ?
    if (((binary_now.time - TimeScann.time) * 1000) +
	((binary_now.millitm - TimeScann.millitm)) >=
	TimeToSleepBeforeChangeChannel) {
      /* it's time to change Channel */
      ftime (&TimeScann);
      if (NextChannel (devname, ChannelHop)) {
	stop_sniffing = 2;
	debug (2, "devname = %s, ChannelHop=%d\n", devname, ChannelHop);
	debug (0, "Can't change Channel\n");
	break;			// Exit while (stop_sniffing == 0)
      }
    }
    //pthread_delay_np (&delay);
    usleep (TimeToSleepBeforeChangeChannel * 100);
  }
  pthread_exit (0);
}
#endif



/*
 * Write Version only and exit
 */
void
VersionAndBye (void)
{
//  fprintf (stderr, "\nPlease send Bug report to : %s\n\n", WIFISCANNER_BUG);
  exit (EXIT_SUCCESS);
}

/******** MAIN ********/

int
main (int argc, char **argv)
{
  int recvlen;			// Size of packet received
  //int MaxPacketPerChannel;    // Number of packet received before the next channel is select
  struct tm *ascii_now = NULL;	// The timestamp in ASCII (human readable)
  struct timeb binary_now;
#ifndef WITH_THREAD
  struct timeb TimeScann;
#endif
  char devname[6];		// Device name
  unsigned char msgbuf[MAX_BUFFER_SIZE];	// packet container
#ifdef WITH_WIRETAP
  struct wtap_pkthdr packet_hdr_info;	// pointer to the header of wtap packet 
  int wtap_error;
  wtap_dumper *dump_file = NULL;	// pointer to the file used to save pcap data
#else
  char *dump_file = NULL;
#endif
  UINT32 CatchPacket = 0;
  UINT32 MaxCatchPacket;
  UINT8 DateFormat;
  char PacketStatus[256];
  char *OutFileName = NULL, *OutDumpFileName = NULL, *OutDotFileName = NULL;
  FILE *OutFile = NULL, *OutDumpFile = NULL, *OutDotFile = NULL;
  UINT8 GotItOne = FALSE;
  UINT8 DoNotDisplay = (UINT8) 0;	// 0000 0000
  //                                            |||\_Ack
  //                                            ||\__Beacon
  //                                            |\___Control
  //                                            \____Data
  UINT8 IDS_is_ON = FALSE;
  UINT8 ids_warning = (UINT8) 0;
  char strChannel[8];
  UINT8 ActifChannel = 0;
  UINT8 CheckScreenSize = TRUE;
  int ret;


  fprintf (stderr, "WifiScanner v%s (c) 2002 Herv Schauer Consultants (",
	   WIFISCANNER_VERSION);
  fprintf (stderr, "Jerome.Poggi@hsc-labs.com)\n");

  /* Check if we are run by UID = 0 */
  if (getuid () != 0) {
    warning ("Hum hum, you seem to be not have the root capabilities...\n"
	     "Try to obtain root capabilities and re-run me :-)\n");
    HelpAndBye ();
  }

  /* Initialise defaut params */
  SingleChannel = (UINT8) 0;
  DebugLevel = (UINT8) 0;
  TimeToSleepBeforeChangeChannel = TIME_TO_SLEEP_BEFORE_CHANGE_CHANNEL;
//  TimeToSleepBeforeChangeChannel = 0;
  ChannelHop = (UINT8) 1;
  DateFormat = (UINT8) 1;	// 1 is human readable
  MaxCatchPacket = 0;		// 0 = unlimited
  strcpy (devname, "wlan0");
  TypeOfCard = WLAN_NG_CARD;

  /* parse command line */
  ParseCommandLine (argc, argv, &SingleChannel, &OutFileName,
		    &OutDumpFileName, &OutDotFileName, &DebugLevel,
		    &TimeToSleepBeforeChangeChannel, devname, &ChannelHop,
		    &MaxCatchPacket, &DateFormat, &DoNotDisplay, &IDS_is_ON,
		    &CheckScreenSize, &TypeOfCard);

  // For information
  debug (2, "Size of ClientInfo_t struct : %6d bytes\n",
	 sizeof (ClientInfo_t));
  debug (2, "Size of ClientInfo_t array  : %6d bytes\n",
	 (sizeof (ClientInfo_t) * MAX_NUMBER_OF_DETECTED_CLIENT));

  // We try to interract with the card, if it's not possible we exit
  if (openCard (devname)) {
    fatal ("The interface can not be found or initialize\n");
  }
  // Sanity check
  if (OutFileName) {
    if ((OutFile = fopen (OutFileName, "a")) == NULL) {
      fatal ("Cant open outfile %s for writing\n", OutFileName);
    }
  }
#ifdef WITH_WIRETAP
  // Sanity check
  if (OutDumpFileName) {
    dump_file = wtap_dump_open (OutDumpFileName, WTAP_FILE_PCAP,
				WTAP_ENCAP_IEEE_802_11, 2344, &wtap_error);
    if (!dump_file) {
      fatal ("Cannot make the dump file!\n");
    }
  }
#endif

  fprintf (stderr, "Beginning scan of the 802.11b networks...\n");

  if (signal (SIGINT, (__sighandler_t) stop_signal) == SIG_ERR) {
    fatal ("AAAARRRGGG ! Can't install the signal handler !\n");
  }
  fprintf (stderr, "Use CTRL-C to stop sniffing\n");

#ifdef LWNG_14
  if (openPacket (devname) < 0) {
    fprintf (stderr, "Can't open pcap device\n");
#ifdef WITH_WIRETAP
    wtap_dump_close (dump_file, &wtap_error);
#endif
    exit (EXIT_FAILURE);
  }
#else
  if (openPacket () < 0) {
    fprintf (stderr, "You usually have to be root to open raw sockets\n");
    wtap_dump_close (dump_file, &wtap_error);
    exit (EXIT_FAILURE);
  }
#endif

#ifdef HAVE_LIBNCURSES
  /* Initialisation of NCURSES */
  CursesIsInUse = TRUE;
  InitScreen (CheckScreenSize);
  if (IDS_is_ON == TRUE)
    mvwprintw (Sum_WND, 17, 3, "IDS is ON ");
#else
  if (IDS_is_ON == TRUE)
    fprintf (stderr, "IDS is ON");
#endif

  /*** 
   * Change my Priority and put all keyboard input to trash
   ***/

  (void) setvbuf (stdin, NULL, _IONBF, 0);	// Input go to trash :-)
  (void) setpriority (PRIO_PROCESS, 0, -15);	// take more CPU than normal

  /***
   *  Initialisation of the selected channel in single channel scan 
   ***/
  if (SingleChannel && !selectChannel (devname, SingleChannel)) {
    //stop_sniffing = 1;
    warning ("Can't initialize Channel number !\n");
    //sleep (5);
  }
#ifndef WITH_THREAD
  ftime (&TimeScann);
#else
  if (!SingleChannel) {
    //pthread_mutex_init (&screen_mutex, PTHREAD_MUTEX_INITIALIZER);
    ret = pthread_create (&thread_ChgChan, NULL,
			  (void *) &ChangeChannel_th, (void *) devname);
    if (ret < 0) {
      fatal ("Boom ! Thread to change channel cannot create (err:%d)\n", ret);
    }
  }
  if (DebugLevel < 2) {
    ret = pthread_create (&thread_RefreshRTW, NULL,
			  (void *) &RefreshRealTime_WND_th, (void *) NULL);
    if (ret < 0) {
      fatal ("Boom ! Thread to refresh screen cannot create (err:%d)\n", ret);
    }
  }
#endif


  /*****
    This is the linear and threaded scheduler
    TODO : To convert in more thread (Work in progress) :-)
  ******/
  while (stop_sniffing == 0) {

#ifndef WITH_THREAD
    /*** Time trigger for change channel ***/
    // Save time, so we change channel only when it's time
    ftime (&binary_now);
    // is it time to change channel ?
    if (((binary_now.time - TimeScann.time) * 1000) +
	(binary_now.millitm - TimeScann.millitm) >=
	(TimeToSleepBeforeChangeChannel)) {
      /* it's time to change Channel */
      if (!SingleChannel) {
	ftime (&TimeScann);
	if (NextChannel (devname, ChannelHop)) {
	  stop_sniffing = 2;
	  debug (0, "Can't change Channel\n");
	  break;		// Exit while (stop_sniffing == 0)
	}
      }
    }
#endif

    /*** Get packet process ***/
    memset (msgbuf, 0, MAX_BUFFER_SIZE);
    recvlen = getPacket (msgbuf, MAX_BUFFER_SIZE, TIMEOUT_TRY_TO_READ_PAQUET);
    if (recvlen > 0) {
      // Get it one ? Yes ! so analyse it
      Stats.Packets++;
      GotItOne = TRUE;
      // Save the hardware channel before the washing machine
      ActifChannel = Res.SChannel;
      memset (&Res, 0, sizeof (Res));
      Res.SChannel = ActifChannel;
      // Print the packet number and size
      debug (2, "Packet number : %ld\n", CatchPacket);
      debug (2, "recvlen = %X\n", recvlen);

      // Just an Hexdump packet
      if (DebugLevel >= 2) {
	DumpHexPaquets (RealTime_WND, msgbuf, recvlen);
	wrefresh (RealTime_WND);
      }
      /* END DEBUG */

#ifdef WITH_WIRETAP
 /*** Write DATA to DumpFile ***/
      if (OutDumpFileName) {
	/*
	 * Setup Wiretap packet header 
	 */
	gettimeofday (&packet_hdr_info.ts, NULL);
	packet_hdr_info.caplen = recvlen;
	packet_hdr_info.len = recvlen;
	packet_hdr_info.pkt_encap = WTAP_ENCAP_IEEE_802_11;

	/*
	 * Now we can save the frame to the capture file 
	 */
	if (!wtap_dump (dump_file, &packet_hdr_info, NULL, &msgbuf,
			&wtap_error)) {
	  stop_sniffing = 4;
	  debug (0, "Can't save DATA\n");
	  goto NoPacketIsCatched;
	}
      }
#endif

      /*
       * Analyse Data and Write Result
       */
      if (processPacket (msgbuf, recvlen)) {
	// Define how and what we display for channel
	ChannelToStr (strChannel, Res.Channel, Res.SChannel);

	// Save the maximum strength of signal for print an hitogram
	if (Stats.MaxSignal[ActifChannel] < Res.Signal)
	  Stats.MaxSignal[ActifChannel] = Res.Signal;

	// Different date format.
	ftime (&binary_now);
	if (DateFormat == 0) {
	  snprintf (PacketStatus, 256,
		    "%d.%03d,\"%s\",%s,%s,%s,%03d,%03d,%s,%s,%s,%s,%s,%s,%s\n",
		    (int) binary_now.time, binary_now.millitm,
		    Res.SSID, strChannel, BoolToWepNoWep (Res.hasWep),
		    BoolToStaAp (Res.isAp), Res.Signal, Res.Noise,
		    Res.DestMac, Res.SrcMac, Res.BssId,
		    RateToString (Res.Rate),
		    TypeOfClientToString (Res.TypeOfClient),
		    SubTypeOfClientToString (Res.TypeOfClient),
		    Res.TypeOfPacket);
	} else {
	  ascii_now = localtime (&binary_now.time);
	  snprintf (PacketStatus, 256,
		    "%02d/%02d/%04d %02d:%02d:%02d.%03d,"
		    "\"%s\",%s,%s,%s,%03d,%03d,%s,%s,%s,%s,%s,%s,%s\n",
		    ascii_now->tm_mon + 1, ascii_now->tm_mday,
		    ascii_now->tm_year + 1900, ascii_now->tm_hour,
		    ascii_now->tm_min,
		    (binary_now.millitm == 1000) ? 
                      (ascii_now->tm_sec + 1) : ascii_now->tm_sec,
		    (binary_now.millitm == 1000) ? 
                      0 : binary_now.millitm,
		    Res.SSID, strChannel, BoolToWepNoWep (Res.hasWep),
		    BoolToStaAp (Res.isAp), Res.Signal, Res.Noise,
		    Res.DestMac, Res.SrcMac, Res.BssId,
		    RateToString (Res.Rate),
		    TypeOfClientToString (Res.TypeOfClient),
		    SubTypeOfClientToString (Res.TypeOfClient),
		    Res.TypeOfPacket);
	}

	// SWITCH : DISPLAY or NOT a sort of Packet
	if (!
	    ((DO_NOT_DISPLAY_ACK) ||
	     (DO_NOT_DISPLAY_BEACON) ||
	     (DO_NOT_DISPLAY_CONTROL) || (DO_NOT_DISPLAY_DATA))) {
#ifdef HAVE_LIBNCURSES
	  wprintw (RealTime_WND, "%s", PacketStatus);
#else
	  fprintf (stdout, "%s", PacketStatus);
#endif //#ifdef HAVE_LIBNCURSES
	}
	/// Write it also on file if wanted
	if (OutFile) {
	  fprintf (OutFile, "%s", PacketStatus);
	  fflush (OutFile);
	}
	/// A little mark for better view :)
	debug (1, "--- END OF PACKET PROCESSING ---\n");
#ifdef HAVE_LIBNCURSES
	WritePanel (TRUE);
	RefreshRealTime_WND (GotItOne);
#else
	fflush (stdout);
#endif // #ifdef HAVE_LIBNCURSES
      } else {			/* if (processPacket */
	/* Packet is not analysed */
	// Init of time
	//ftime (&binary_now);

	if (Res.TypeOfPacket != NULL) {
	  if (DateFormat == 0) {
	    snprintf (PacketStatus, 256,
		      "%d.%03d,%s\n",
		      (int) binary_now.time, binary_now.millitm,
		      Res.TypeOfPacket);
	  } else {
	    ascii_now = localtime (&binary_now.time);
	    snprintf (PacketStatus, 256,
		      "%02d/%02d/%04d %02d:%02d:%02d.%03d,%s\n",
		      ascii_now->tm_mon + 1, ascii_now->tm_mday,
		      ascii_now->tm_year + 1900, ascii_now->tm_hour,
		      ascii_now->tm_min, ascii_now->tm_sec,
		      binary_now.millitm, Res.TypeOfPacket);
	  }

#ifdef HAVE_LIBNCURSES
	  wprintw (RealTime_WND, "%s", PacketStatus);
	  WritePanel (FALSE);
	  RefreshRealTime_WND (GotItOne);
#else
	  fprintf (stdout, "%s\n", PacketStatus);
#endif // #ifdef HAVE_LIBNCURSES

	  // Write it also on file if wanted
	  if (OutFile) {
	    fprintf (OutFile, "%s", PacketStatus);
	  }
	}
      }
      // Exit if we reach the maximum of packets wanted
      ++CatchPacket;
      if ((MaxCatchPacket != 0)
	  && (CatchPacket > (MaxCatchPacket + Stats.INVLD))) {
	stop_sniffing = 5;	// Exit the principal loop
	warning ("Max packets reached\n");
      }
    } else {
#ifdef LWNG_14
      // No packet ? so sleep and wait
      /* An another solution is to used "blocking" IO */
      //if (SingleChannel)
      //usleep (TIMEOUT_TRY_TO_READ_PAQUET);
      usleep (TimeToSleepBeforeChangeChannel * 10);
#endif
    }
  NoPacketIsCatched:

    if (GotItOne) {
      if (IDS_is_ON == TRUE)
	ids_warning = IDS ();
      if (ids_warning != 0) {
	ascii_now = localtime (&binary_now.time);
#ifdef HAVE_LIBNCURSES
	if (DateFormat == 0) {
	  mvwprintw (Panel_WND, ROW_WND_PANEL - 1, 24,
		     "IDS WARNING (%d)", (int) binary_now.time);
	} else {
	  mvwprintw (Panel_WND, ROW_WND_PANEL - 1, 24,
		     "IDS WARNING (%02d/%02d/%04d %02d:%02d:%02d)",
		     ascii_now->tm_mon + 1, ascii_now->tm_mday,
		     ascii_now->tm_year + 1900, ascii_now->tm_hour,
		     ascii_now->tm_min, ascii_now->tm_sec);
	}
#else
	if (DateFormat == 0) {

	  fprintf (stderr, "IDS WARNING (%d)", (int) binary_now.time);
	} else {
	  fprintf (stderr,
		   "IDS WARNING (%02d/%02d/%04d %02d:%02d:%02d)",
		   ascii_now->tm_mon + 1, ascii_now->tm_mday,
		   ascii_now->tm_year + 1900, ascii_now->tm_hour,
		   ascii_now->tm_min, ascii_now->tm_sec);
	}
#endif
      }
    }
#ifdef HAVE_LIBNCURSES
    WriteSummary ();
    RefreshRealTime_WND (GotItOne);
#endif
    // For refreshing screen only one time per second
    GotItOne = FALSE;
  }				/* while still sniffing */

#ifdef WITH_THREAD
  // Destroy MUTEX
  pthread_mutex_destroy (&screen_mutex);
  // Wait thread to exit
  if (!SingleChannel) {
    void *ret;
    warning ("Waiting thread_ChgChan to exit...");
    (void) pthread_join (thread_ChgChan, &ret);
    warning ("OK\n");
  }
  if (DebugLevel < 2) {
    void *ret;
    warning ("Waiting thread_RefreshRTW to exit...");
    (void) pthread_join (thread_RefreshRTW, &ret);
    warning ("OK\n");
  }
#endif
  if (DebugLevel > 0) {
    warning ("Wait 5 seconds before exit...\n");
    sleep (5);
  }

  EndCurses ();			/// Exit from curses environment

  if (OutFile) {		/// Close file if needed, it's better clean
    fclose (OutFile);
  }
#ifdef WITH_WIRETAP
  if (OutDumpFile) {		/// Close file if needed, it's better clean
    wtap_dump_close (dump_file, &wtap_error);
  }
#endif

  closePacket ();		/// Close RaxSocket
  shutCard (devname);		/// put card in a normal state

  LogWriteReport ();		/// Write the summary

  if (OutDotFileName) {		/// And the .dot file also
    if ((OutDotFile = fopen (OutDotFileName, "w")) == NULL) {
      fatal ("Cant open outfile %s for writing\n", OutDotFileName);
    }
    LogWriteDOT (OutDotFile);
    fclose (OutDotFile);
  }

  if (!SingleChannel) {
    LogWriteHisto ();
  }

  debug (1, "Stop Sniffing = %d\n", stop_sniffing);
  exit (EXIT_SUCCESS);
}
