/* Linux Prism II Stumbler - Utility Scan for 802_11 networks under Linux
 * 
 * File : interface.c
 * Project : WifiScanner (c) 2002 Herv Schauer Consultants
 * Usage : This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA).
 * 
 * Base code was from prismstumbler Jan Fernquist <Jan.B.Fernquist@telia.com>
 * and wlanctl from www.linux-wlan.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: interface.c,v 1.14 2003/07/25 11:23:08 poggij Exp $
 */

#include <include.h>
#include <src/interface.h>
#include <src/analyse.h>

extern WINDOW *Title_WND, *Panel_WND, *Sum_WND, *RealTime_WND;
extern Statistics_t Stats;
extern UINT8 DebugLevel;
extern UINT8 CursesIsInUse;
#ifdef WITH_THREAD
extern pthread_mutex_t screen_mutex;
extern UINT8 stop_sniffing;
#endif

/*
 * Function to initialise the screen
 */
void
InitScreen (UINT8 CheckScreenSize)
{
  initscr ();
  cbreak ();
  noecho ();
  nonl ();
  intrflush (stdscr, FALSE);
  keypad (stdscr, TRUE);

  if ((CheckScreenSize)
      && ((LINES < ROW_MINI_NEEDED) || (COLS < COL_MINI_NEEDED))) {
    endwin ();
    fprintf (stderr,
	     "\nThis program requires a screen size of at least %d columns by %d lines\n",
	     COL_MINI_NEEDED, ROW_MINI_NEEDED);
    fprintf (stderr, "Please resize your window\n\n");
    exit (1);
  }

  Title_WND = subwin (stdscr, ROW_WND_TITLE, COL_WND_TITLE, 0, 0);
  Panel_WND = subwin (stdscr, ROW_WND_PANEL, COL_WND_PANEL, ROW_WND_TITLE, 0);
  Sum_WND =
    subwin (stdscr, ROW_WND_SUM, COL_WND_SUM, ROW_WND_TITLE, COL_WND_PANEL);
  RealTime_WND =
    subwin (stdscr, ROW_WND_REALTIME, COL_WND_REALTIME,
	    (ROW_WND_TITLE + ROW_WND_PANEL), 0);

  scrollok (RealTime_WND, TRUE);
  scrollok (Panel_WND, TRUE);
  //nodelay (RealTime_WND, TRUE);
  //werase(Sum_WND);

  box (Sum_WND, '|', '-');
  mvwprintw (Sum_WND, 0, 3, " Summary ");
  mvwprintw (Sum_WND, 1, 2, "AP     :");
  mvwprintw (Sum_WND, 2, 2, "STA    :");
  mvwprintw (Sum_WND, 3, 2, "BEACON :");
  mvwprintw (Sum_WND, 4, 2, "SSID   :");
  mvwprintw (Sum_WND, 5, 2, "Channel:");
  mvwprintw (Sum_WND, 6, 2, "Invalid:");
  mvwprintw (Sum_WND, 7, 2, "Crypted:");
  mvwprintw (Sum_WND, 8, 2, "Weak   :");
  mvwprintw (Sum_WND, 9, 2, "Last IV:");
  mvwprintw (Sum_WND, 10, 2, "Packets:");

  mvwprintw (Sum_WND, 12, 2, "Scan ");
  mvwprintw (Sum_WND, 13, 2, "|_____________  _|");
  mvwprintw (Sum_WND, 14, 3, "0000000001111  1");
  mvwprintw (Sum_WND, 15, 3, "1234567890123  4");
  mvwprintw (Sum_WND, 17, 3, "IDS is OFF");

  box (Panel_WND, '|', '-');

  wprintw (Title_WND,
	   "WifiScanner v%s (c) 2002 Herv Schauer Consultants (%s)\n",
	   WIFISCANNER_VERSION, WIFISCANNER_BUG);
  RefreshAllWND ();
}

void
RefreshAllWND (void)
{
#ifdef WITH_THREAD
  pthread_mutex_lock (&screen_mutex);
#endif
  wrefresh (Sum_WND);
  wrefresh (Panel_WND);
  wrefresh (RealTime_WND);
  wrefresh (Title_WND);
  refresh ();
#ifdef WITH_THREAD
  pthread_mutex_unlock (&screen_mutex);
#endif
}

/* Write Stats in Summary Window */
void
WriteSummary (void)
{
  DoSummary ();

#ifdef WITH_THREAD
  pthread_mutex_lock (&screen_mutex);
#endif
  mvwprintw (Sum_WND, 1, 11, "%d", Stats.AP);
  mvwprintw (Sum_WND, 2, 11, "%d", Stats.STA);
  mvwprintw (Sum_WND, 3, 11, "%d", Stats.Beacon);
  mvwprintw (Sum_WND, 4, 11, "%d", Stats.SSID);
  mvwprintw (Sum_WND, 5, 11, "%d", Stats.Channel);
  mvwprintw (Sum_WND, 6, 11, "%d", Stats.INVLD);
  mvwprintw (Sum_WND, 7, 11, "%d", Stats.CryptedPackets);
  mvwprintw (Sum_WND, 8, 11, "%d", Stats.WeakIV);
//  mvwprintw (Sum_WND, 9, 11, "%02X:%02X:%02X:%02X", Stats.IV[3], Stats.IV[2],
//           Stats.IV[1], Stats.IV[0]);
  mvwprintw (Sum_WND, 9, 11, "%02X:%02X:%02X", Stats.IV[2], Stats.IV[1],
	     Stats.IV[0]);
  mvwprintw (Sum_WND, 10, 11, "%d", Stats.Packets);

  wrefresh (Sum_WND);
#ifdef WITH_THREAD
  pthread_mutex_unlock (&screen_mutex);
#endif
}


/***
 * Function to refresh screen only one time per second
 *  This function is threaded, so it do more sleep than work ;-)
 ***/
#ifdef WITH_THREAD
void
RefreshRealTime_WND_th (void)
{
  struct tm *ascii_now;		// The timestamp in ASCII (human readable)
  struct timeb binary_now;

  while (!stop_sniffing) {
    ftime (&binary_now);
    ascii_now = localtime (&binary_now.time);
    pthread_mutex_lock (&screen_mutex);
    mvwprintw (Panel_WND, ROW_WND_PANEL - 1, 3, "Last Updt %02d:%02d:%02d ",
	       ascii_now->tm_hour, ascii_now->tm_min, ascii_now->tm_sec);
    wrefresh (RealTime_WND);
    pthread_mutex_unlock (&screen_mutex);
    // Sleep for less than 1 second, but wake up at HH:MM:SS:000
    usleep((1000 - binary_now.millitm) * 1000);
  }
  pthread_exit (0);
}

/***
 * This function is to give compatibility
 * and because wben debug is more or egual than 2 we refresh every packets
 ***/
void
RefreshRealTime_WND (UINT8 GotItOne)
{
  if (DebugLevel >= 2) {
    pthread_mutex_lock (&screen_mutex);
    wrefresh (RealTime_WND);
    pthread_mutex_unlock (&screen_mutex);
  }
}

#else
/// Not threaded version
void
RefreshRealTime_WND (UINT8 GotItOne)
{
  static time_t now, last_time = 0;
  struct tm *ascii_now;		// The timestamp in ASCII (human readable)
  struct timeb binary_now;
  static UINT8 NeedToRefresh = FALSE;

  time (&now);

  if (GotItOne)
    NeedToRefresh = TRUE;

  if (((now != last_time) && (NeedToRefresh)) || (DebugLevel >= 2)) {
    ftime (&binary_now);
    last_time = now;
    ascii_now = localtime (&binary_now.time);
    ascii_now = localtime (&now);
    mvwprintw (Panel_WND, ROW_WND_PANEL - 1, 3, "Last Updt %02d:%02d:%02d ",
	       ascii_now->tm_hour, ascii_now->tm_min, ascii_now->tm_sec);

    wrefresh (RealTime_WND);
    NeedToRefresh = FALSE;
  }
}
#endif

// Function to exit from ncurses environment
void
EndCurses (void)
{
  CursesIsInUse = FALSE;
  endwin ();
}
