/* Linux Prism II Stumbler - Utility Scan for 802_11 networks under Linux
 * 
 * File : functions.c
 * Project : WifiScanner (c) 2002 Herv Schauer Consultants
 * Usage : This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA).
 * 
 * Base code was from prismstumbler Jan Fernquist <Jan.B.Fernquist@telia.com>
 * and wlanctl from www.linux-wlan.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: functions.c,v 1.34 2003/07/25 11:23:08 poggij Exp $
 */

#include <include.h>
#include <src/functions.h>
#include <src/analyse.h>
#include <src/crc-32.h>
#include <src/crt_io.h>
#include <src/conversion.h>

// This two function come from scanner.c
void HelpAndBye (void);
void VersionAndBye (void);

/*
 * Some internal forward refs
 */
int analyseBeacon (unsigned char *packet, int len);
int analyseProbeReq (unsigned char *packet);
int analyseProbeRep (unsigned char *packet);
int analyseCTS (unsigned char *packet);
int analyseRTS (unsigned char *packet);
int analyseACK (unsigned char *packet);
int analysePSPOLL (unsigned char *packet);
int analyseData (unsigned char *packet, int len);
int analyseMGMT (unsigned char *packet);	/// Generic analyse og MGMT packets

void ProcessTagBits (unsigned char *packet, int len, int FrameType);
UINT8 dataIsCrypted (unsigned char *packet, int len);

/*
 * Use globals, ugly but efficient
 */
extern struct sockaddr_nl nl_sk_addr;
extern ScanResult_t Res;
extern unsigned int DebugLevel;
extern Statistics_t Stats;
extern WINDOW *Title_WND, *Panel_WND, *Sum_WND, *RealTime_WND;

extern p80211_caphdr_t wlan_header;
//extern UINT8 wlan_payload[];

/************************************/

/*
 * Same as strncpy and snprintf, but always be sure the result is terminated.
 */

char *
safe_strncpy (char *dst, const char *src, int size)
{
  dst[size - 1] = '\0';
  return strncpy (dst, src, size - 1);
}

int
safe_snprintf (char *s, int size, char *fmt, ...)
{
  va_list ap;
  int ret;

  va_start (ap, fmt);
  ret = vsnprintf (s, size, fmt, ap);
  s[size - 1] = '\0';
  va_end (ap);

  return ret;
}

/*************************************/
/* How to detect a crypted packet ?  */
/* Take a look at airsnort ;-)       */
UINT8
dataIsCrypted (unsigned char *packet, int len)
{
  UINT8 ret = 0;
  unsigned char *DataFrame;

  (char *) DataFrame = packet + sizeof (p80211msg_lnxind_wlansniffrm_t);

  //                    SNAP                       NetBIOS
  //if ((DataFrame[24] == 0xAA) || (DataFrame[24] == 0xF0))
  //if (DataFrame[24] == 0xAA)
  if ((DataFrame[24] == 0xAA) && (DataFrame[25] == 0xAA))
    ret = 0;
  else
    {
      ret = 1;
      Stats.CryptedPackets++;
      // is there some crypted data in packet ?
      if ((unsigned int) len >
	  (sizeof (p80211msg_lnxind_wlansniffrm_t) + 24 + 4))
	{
	  // Save IV and key number
	  // IV:IV:IV:
	  Stats.IV[0] = Res.IV[0] = DataFrame[24];
	  Stats.IV[1] = Res.IV[1] = DataFrame[25];
	  Stats.IV[2] = Res.IV[2] = DataFrame[26];
	  Stats.IV[3] = Res.IV[3] = DataFrame[27];

	  // Check if it's a Weak IV
	  // For more Information take a look at :
	  // http://www.dachb0den.com/projects/bsd-airtools/wepexp.txt
	  if ((Res.IV[1] == 255 && Res.IV[0] > 2 && Res.IV[0] < 16) ||
	      ((Res.IV[0] + Res.IV[1]) == 1
	       && (Res.IV[2] <= 0x0A || Res.IV[2] == 0xFF))
	      || ((Res.IV[0] + Res.IV[1]) <= 0x0C
		  && (Res.IV[2] >= 0xF2 && Res.IV[2] <= 0xFE
		      && Res.IV[2] != 0xFD)))
	    Stats.WeakIV++;
	}
    }
  return ret;
}

// Update Res.OtherInformation
void
UpdateOtherInformation (char *Buff)
{
  char Buff2[MAXSIZE_OTHERINFORMATION + 1];

  strncpy (Buff2, Res.OtherInformation, MAXSIZE_OTHERINFORMATION);
  if ((strlen (Buff) + strlen (Buff2)) > (MAXSIZE_OTHERINFORMATION - 3))
    snprintf (Res.OtherInformation, MAXSIZE_OTHERINFORMATION, "%s - %s",
	      Buff2, Buff);
  else
    sprintf (Res.OtherInformation, "%s - %s", Buff2, Buff);

  debug (2, "--- Processing UpdateOtherInformation : %s ---\n", Buff);

}


// Print MAC Add in a correct Format
void
printfMAC (UINT8 AddMac[WLAN_ADDR_LEN])
{
  printf ("%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	  AddMac[0], AddMac[1], AddMac[2], AddMac[3], AddMac[4], AddMac[5]);
}

/*
 * Get the different data from the MGMT header. Fill global struct
 */
int
processPacket (unsigned char *packet, int len)
{
  int ReturnCode = 0;
  UINT32 CRC;

  p80211_hdr_t *d80211b_Header;
  (char *) d80211b_Header = packet;
  Res.TypeSubtype = d80211b_Header->a3.fc;


  Res.Signal = wlan_header.ssi_signal;
  Res.Noise = wlan_header.ssi_noise;
  Res.Rate = wlan_header.datarate;

  switch (WLAN_GET_FC_FTYPE (d80211b_Header->a3.fc))
    {
      /* Frame subtypes */
    case WLAN_FTYPE_MGMT:
      debug (1, "Process Management Frame\n");
      /* Management */
      switch (WLAN_GET_FC_FSTYPE (d80211b_Header->a3.fc))
	{
	case WLAN_FSTYPE_ASSOCREQ:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_ASSOCRESP:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_REASSOCREQ:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_REASSOCRESP:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_PROBEREQ:
	  ReturnCode = analyseProbeReq (packet);
	  break;
	case WLAN_FSTYPE_PROBERESP:
	  ReturnCode = analyseProbeRep (packet);
	  break;
	case WLAN_FSTYPE_BEACON:
	  ReturnCode = analyseBeacon (packet, len);
	  Stats.Beacon++;
	  break;
	case WLAN_FSTYPE_ATIM:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_DISASSOC:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_AUTHEN:
	  ReturnCode = analyseMGMT (packet);
	  break;
	case WLAN_FSTYPE_DEAUTHEN:
	  ReturnCode = analyseMGMT (packet);
	  break;
	}
      break;
    case WLAN_FTYPE_CTL:
      debug (1, "Process Control Frame\n");
      switch (WLAN_GET_FC_FSTYPE (d80211b_Header->a3.fc))
	{
	  /* Control */
	case WLAN_FSTYPE_PSPOLL:
	  ReturnCode = analysePSPOLL (packet);
	  break;
	case WLAN_FSTYPE_RTS:
	  ReturnCode = analyseRTS (packet);
	  break;
	case WLAN_FSTYPE_CTS:
	  ReturnCode = analyseCTS (packet);
	  break;
	case WLAN_FSTYPE_ACK:
	  ReturnCode = analyseACK (packet);
	  break;
	case WLAN_FSTYPE_CFEND:
	  break;
	case WLAN_FSTYPE_CFENDCFACK:
	  break;
	}
      break;
    case WLAN_FTYPE_DATA:
      debug (1, "Process Data Frame\n");
      //CLEAR_TYPE_BIT_AP (Res.TypeOfClient);
      SET_TYPE_BIT_DATA (Res.TypeOfClient);
      switch (WLAN_GET_FC_FSTYPE (d80211b_Header->a3.fc))
	{
	  /* Data */
	case WLAN_FSTYPE_DATAONLY:
	  ReturnCode = analyseData (packet, len);
	  break;
	case WLAN_FSTYPE_DATA_CFACK:
	  ReturnCode = analyseData (packet, len);
	  break;
	case WLAN_FSTYPE_DATA_CFPOLL:
	  ReturnCode = analyseData (packet, len);
	  break;
	case WLAN_FSTYPE_DATA_CFACK_CFPOLL:
	  ReturnCode = analyseData (packet, len);
	  break;
	case WLAN_FSTYPE_NULL:
	  ReturnCode = analyseData (packet, len);
	  break;
	case WLAN_FSTYPE_CFACK:
	  break;
	case WLAN_FSTYPE_CFPOLL:
	  break;
	case WLAN_FSTYPE_CFACK_CFPOLL:
	  break;
	}
      break;
    }

  memcpy (&CRC, packet + len - 4, 4);

/*
  // Check if the packet is a good packet
// TODO : Find a good driver that calculate CRC cooreltly !!!
//#ifndef LWNG_15
//  if ((CRC == 0xFFFFFFFF)
//      || ((unsigned) len < sizeof (p80211msg_lnxind_wlansniffrm_t))) {
//#else
  if ((unsigned) len < sizeof (p80211msg_lnxind_wlansniffrm_t)) {
//#endif
    if (DebugLevel >= 1) {
      debug (0, "INVALID PACKET : BAD CRC (%08X) or too short (%04X)\n",
	     CRC, len);
      if ((unsigned) len > sizeof (p80211msg_lnxind_wlansniffrm_t)) {
	CRC =
	  doFCS (&packet[sizeof (p80211msg_lnxind_wlansniffrm_t)],
		 (len - 4 - sizeof (p80211msg_lnxind_wlansniffrm_t)));
	debug (1, "Calculed CRC is : %08X\n", CRC);
      }
    }
    strncpy (Res.TypeOfPacket, "INVLID", WLAN_SIZEOF_TYPEOFPACKET);
    Stats.INVLD++;
    ReturnCode = 0;
  } else */
  {
    strncpy (Res.TypeOfPacket, TypeOfPacketToString (d80211b_Header->a3.fc),
	     WLAN_SIZEOF_TYPEOFPACKET);
  }

  debug (1, "FrameType=%04X (type:%X subtype:%X)\n",
	 d80211b_Header->a3.fc,
	 WLAN_GET_FC_FTYPE (d80211b_Header->a3.fc),
	 WLAN_GET_FC_FSTYPE (d80211b_Header->a3.fc));

  return ReturnCode;
}

/* ***
 * Functions of Analyse
 */

int
analyseBeacon (unsigned char *packet, int len)
{
  p80211_hdr_t *d80211b_Header;
  FixedField_t *FixedField;
  (unsigned char *) d80211b_Header = packet;
  (char *) FixedField = packet + sizeof (p80211_hdr_a3_t);

  sprintf (Res.DestMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (Res.SrcMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);
  sprintf (Res.BssId, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a3[0], d80211b_Header->a3.a3[1],
	   d80211b_Header->a3.a3[2], d80211b_Header->a3.a3[3],
	   d80211b_Header->a3.a3[4], d80211b_Header->a3.a3[5]);

  Res.isAp = WLAN_GET_MGMT_CAP_INFO_ESS (FixedField->cap_info);
  Res.hasWep = WLAN_GET_MGMT_CAP_INFO_PRIVACY (FixedField->cap_info);

  // Here we know if it's an AP or an Ah-Hoc client
  SET_TYPE_BIT_BEACON (Res.TypeOfClient);
  if (WLAN_GET_MGMT_CAP_INFO_IBSS (FixedField->cap_info))
    SET_TYPE_BIT_P2P (Res.TypeOfClient);
  else
    CLEAR_TYPE_BIT_P2P (Res.TypeOfClient);
  if (WLAN_GET_MGMT_CAP_INFO_ESS (FixedField->cap_info))
    SET_TYPE_BIT_AP (Res.TypeOfClient);
  else
    CLEAR_TYPE_BIT_AP (Res.TypeOfClient);

  /* Log */
  LogDetectedClient (d80211b_Header->a3.a2);
  LogPutSN (d80211b_Header->a3.a2, Res.Signal);
  LogPutRate (d80211b_Header->a3.a2, Res.Rate);
  LogPutIsAP (d80211b_Header->a3.a2, Res.hasWep, FixedField->bcn_int);
  LogPutBSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a3);
  LogPutTimestamp (d80211b_Header->a3.a2, FixedField->ts);
  LogPutSeqNum (d80211b_Header->a3.a2, Res.DestMac, d80211b_Header->a3.seq);

  debug (1, "Dst:%s Src:%s Bssid:%s\n"
	 "Fragment number : 0x%04X\n"
	 "Sequence number : 0x%02X\n"
	 "Timestamp : 0x%016llX\n"
	 "Beacon Interval : 0x%X\n"
	 "Capabilities : 0x%X\n"
	 "S/N : %ld/%ld\n",
	 Res.DestMac, Res.SrcMac, Res.BssId,
	 WLAN_GET_SEQ_FRGNUM (d80211b_Header->a3.seq),
	 WLAN_GET_SEQ_SEQNUM (d80211b_Header->a3.seq), FixedField->ts,
	 FixedField->bcn_int, FixedField->cap_info, Res.Signal, Res.Noise);

  if ((unsigned int) len <= (sizeof (p80211_hdr_t) + sizeof (FixedField_t)))
    return 0;

  ProcessTagBits (packet, len, WLAN_FSTYPE_BEACON);

  return 1;
}

/*************************/

int
analyseProbeReq (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  FixedField_t *FixedField;
  (unsigned char *) d80211b_Header = packet;
  (char *) FixedField = packet + sizeof (p80211_hdr_a3_t);

  sprintf (Res.DestMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (Res.SrcMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);
  sprintf (Res.BssId, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a3[0], d80211b_Header->a3.a3[1],
	   d80211b_Header->a3.a3[2], d80211b_Header->a3.a3[3],
	   d80211b_Header->a3.a3[4], d80211b_Header->a3.a3[5]);

  debug (1, "Dst:%s Src:%s Bssid:%s\n"
	 "Fragment number : 0x%04X\n"
	 "Sequence number : 0x%02X\n"
	 "S/N : %ld/%ld\n", Res.DestMac, Res.SrcMac, Res.BssId,
	 WLAN_GET_SEQ_FRGNUM (d80211b_Header->a3.seq),
	 WLAN_GET_SEQ_SEQNUM (d80211b_Header->a3.seq), Res.Signal, Res.Noise);

  /* Log */
  LogDetectedClient (d80211b_Header->a3.a2);
  LogPutSN (d80211b_Header->a3.a2, Res.Signal);
  LogPutRate (d80211b_Header->a3.a2, Res.Noise);
  //LogPutIsAP (d80211b_Header->a3.a2, Res.hasWep, FixedField->bcn_int);
  LogPutBSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a3);
  LogPutSeqNum (d80211b_Header->a3.a2, Res.DestMac, d80211b_Header->a3.seq);

  Res.isAp = WLAN_GET_MGMT_CAP_INFO_ESS (FixedField->cap_info);
  Res.hasWep = WLAN_GET_MGMT_CAP_INFO_PRIVACY (FixedField->cap_info);

  return 1;			/* All is OK */
}

/*************************/

int
analyseProbeRep (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  FixedField_t *FixedField;
  (char *) d80211b_Header = packet;
  (char *) FixedField = packet + sizeof (p80211_hdr_a3_t);

  sprintf (Res.DestMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (Res.SrcMac, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);
  sprintf (Res.BssId, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a3[0], d80211b_Header->a3.a3[1],
	   d80211b_Header->a3.a3[2], d80211b_Header->a3.a3[3],
	   d80211b_Header->a3.a3[4], d80211b_Header->a3.a3[5]);

  debug (1, "Dst:%s Src:%s Bssid:%s\n"
	 "Fragment number : 0x%04X\n"
	 "Sequence number : 0x%02X\n"
	 "Timestamp : 0x%016llX\n"
	 "Capabilities : 0x%X\n"
	 "S/N : %ld/%ld\n", Res.DestMac, Res.SrcMac, Res.BssId,
	 WLAN_GET_SEQ_FRGNUM (d80211b_Header->a3.seq),
	 WLAN_GET_SEQ_SEQNUM (d80211b_Header->a3.seq),
	 FixedField->ts, FixedField->cap_info, Res.Signal, Res.Noise);

  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);
  /* STA */
  LogDetectedClient (d80211b_Header->a3.a2);	/* AP */

  LogPutSN (d80211b_Header->a3.a2, Res.Signal);
  LogPutRate (d80211b_Header->a3.a2, Res.Rate);
  LogPutIsAP (d80211b_Header->a3.a2,
	      WLAN_GET_MGMT_CAP_INFO_PRIVACY (FixedField->cap_info),
	      FixedField->bcn_int);
  LogPutBSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a3);
  LogPutTimestamp (d80211b_Header->a3.a2, FixedField->ts);
  LogPutSeqNum (d80211b_Header->a3.a2, Res.DestMac, d80211b_Header->a3.seq);

  Res.isAp = WLAN_GET_MGMT_CAP_INFO_ESS (FixedField->cap_info);
  Res.hasWep = WLAN_GET_MGMT_CAP_INFO_PRIVACY (FixedField->cap_info);

  return 1;			/* All is OK */
}


/**********/

int
analyseRTS (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  char strAddresse2[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (strAddresse2, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
  strncpy (Res.SrcMac, strAddresse2, WLAN_STR_ADDR_LEN);
  strncpy (Res.BssId, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);	/* STA */
  LogDetectedClient (d80211b_Header->a3.a2);	/* AP */
  LogPutSN (d80211b_Header->a3.a3, Res.Signal);
  LogPutRate (d80211b_Header->a3.a3, Res.Rate);

  return 1;			/* All is OK */
}

int
analyseCTS (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
  strncpy (Res.SrcMac, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  strncpy (Res.BssId, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);	/* STA */

  return 1;			/* All is OK */
}

/*
 * Function to analyse an Acknoledge frame
 */ int
analyseACK (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
  strncpy (Res.SrcMac, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  strncpy (Res.BssId, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);	/* STA or AP */

  return 1;			/* All is OK */
}

int
analysePSPOLL (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  char strAddresse2[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (strAddresse2, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  strncpy (Res.SrcMac, strAddresse1, WLAN_STR_ADDR_LEN);
  strncpy (Res.DestMac, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
  strncpy (Res.BssId, strAddresse2, WLAN_STR_ADDR_LEN);
  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);	/* STA */
  LogPutSN (d80211b_Header->a3.a1, Res.Signal);
  LogPutRate (d80211b_Header->a3.a1, Res.Rate);
  LogPutSeqNum (d80211b_Header->a3.a1, Res.DestMac, d80211b_Header->a3.seq);

  return 1;			/* All is OK */
}

/*
 * Function to analyse and parse a Data Frame
 */ int
analyseData (unsigned char *packet, int len)
{
  p80211_hdr_t *d80211b_Header;
  FixedField_t *FixedField;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  char strAddresse2[WLAN_STR_ADDR_LEN];
  char strAddresse3[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;
  (char *) FixedField = packet + sizeof (p80211_hdr_a3_t);

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (strAddresse2, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);
  sprintf (strAddresse3, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a3[0], d80211b_Header->a3.a3[1],
	   d80211b_Header->a3.a3[2], d80211b_Header->a3.a3[3],
	   d80211b_Header->a3.a3[4], d80211b_Header->a3.a3[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  if (WLAN_GET_FC_TODS (d80211b_Header->a3.fc))
    {
      if (WLAN_GET_FC_FROMDS (d80211b_Header->a3.fc))
	{
	  // Data From AP by AP to STA
	  debug (2, " AP to AP");

	  SET_TYPE_BIT_TODS (Res.TypeOfClient);
	  SET_TYPE_BIT_FROMDS (Res.TypeOfClient);

	  LogDetectedClient (d80211b_Header->a3.a1);
	  LogDetectedClient (d80211b_Header->a3.a2);
	  LogDetectedClient (d80211b_Header->a3.a3);

	  strncpy (Res.SrcMac, strAddresse2, WLAN_STR_ADDR_LEN);
	  strncpy (Res.DestMac, strAddresse3, WLAN_STR_ADDR_LEN);
	  strncpy (Res.BssId, "00:00:00:00:00:00", WLAN_STR_ADDR_LEN);
	  /* Log */
	  LogPutSN (d80211b_Header->a3.a1, Res.Signal);
	  LogPutRate (d80211b_Header->a3.a1, Res.Rate);
	  LogPutIsAP (d80211b_Header->a3.a2,
		      WLAN_GET_MGMT_CAP_INFO_PRIVACY (FixedField->cap_info),
		      FixedField->bcn_int);
	  Res.isAp = 1;
	}
      else
	{
	  // Data from DS to STA
	  debug (1, " TODS");
	  SET_TYPE_BIT_TODS (Res.TypeOfClient);

	  strncpy (Res.BssId, strAddresse1, WLAN_STR_ADDR_LEN);
	  strncpy (Res.SrcMac, strAddresse2, WLAN_STR_ADDR_LEN);
	  strncpy (Res.DestMac, strAddresse3, WLAN_STR_ADDR_LEN);
	  /* Log */
	  LogDetectedClient (d80211b_Header->a3.a2);
	  LogPutSN (d80211b_Header->a3.a2, Res.Signal);
	  LogPutRate (d80211b_Header->a3.a2, Res.Rate);
	  //LogPutIsAP (d80211b_Header->a3.a2, Res.hasWep, FixedField->bcn_int);
	  LogPutBSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a1);
	  //LogPutDS (d80211b_Header->a3.a3);
	}
    }
  else if (WLAN_GET_FC_FROMDS (d80211b_Header->a3.fc))
    {
      // Data to DS to STA
      debug (1, " FROMDS");

      SET_TYPE_BIT_FROMDS (Res.TypeOfClient);

      strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
      strncpy (Res.BssId, strAddresse2, WLAN_STR_ADDR_LEN);
      strncpy (Res.SrcMac, strAddresse3, WLAN_STR_ADDR_LEN);
      /* Log */
      LogDetectedClient (d80211b_Header->a3.a3);
      LogPutSN (d80211b_Header->a3.a3, Res.Signal);
      LogPutRate (d80211b_Header->a3.a3, Res.Rate);
      LogPutBSSID (d80211b_Header->a3.a3, d80211b_Header->a3.a2);
      LogPutDS (d80211b_Header->a3.a3);
    }
  else
    {
      // Data from STA to STA
      debug (1, " STA to STA");

      strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
      strncpy (Res.SrcMac, strAddresse2, WLAN_STR_ADDR_LEN);
      strncpy (Res.BssId, strAddresse3, WLAN_STR_ADDR_LEN);
      /* Log */
      LogDetectedClient (d80211b_Header->a3.a1);
      LogDetectedClient (d80211b_Header->a3.a2);
      LogPutSN (d80211b_Header->a3.a2, Res.Signal);
      LogPutRate (d80211b_Header->a3.a2, Res.Rate);
      LogPutBSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a3);
    }

  debug (1, "\n");

  // If it's a NULL Function Packet : 
  //    do not fill IV and do not try to detect if it crypted
  if (WLAN_GET_FC_FSTYPE (d80211b_Header->a3.fc) != WLAN_FSTYPE_NULL)
    Res.hasWep = dataIsCrypted (packet, len);

  // TODO : put LastIV in Struct ClientInfo

  return 1;			/* All is OK */
}

int
analyseMGMT (unsigned char *packet)
{
  p80211_hdr_t *d80211b_Header;
  char strAddresse1[WLAN_STR_ADDR_LEN];
  char strAddresse2[WLAN_STR_ADDR_LEN];
  char strAddresse3[WLAN_STR_ADDR_LEN];
  (unsigned char *) d80211b_Header = packet;

  sprintf (strAddresse1, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a1[0], d80211b_Header->a3.a1[1],
	   d80211b_Header->a3.a1[2], d80211b_Header->a3.a1[3],
	   d80211b_Header->a3.a1[4], d80211b_Header->a3.a1[5]);
  sprintf (strAddresse2, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a2[0], d80211b_Header->a3.a2[1],
	   d80211b_Header->a3.a2[2], d80211b_Header->a3.a2[3],
	   d80211b_Header->a3.a2[4], d80211b_Header->a3.a2[5]);
  sprintf (strAddresse3, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
	   d80211b_Header->a3.a3[0], d80211b_Header->a3.a3[1],
	   d80211b_Header->a3.a3[2], d80211b_Header->a3.a3[3],
	   d80211b_Header->a3.a3[4], d80211b_Header->a3.a3[5]);

  debug (1, "FC=%X", d80211b_Header->a3.fc);

  strncpy (Res.DestMac, strAddresse1, WLAN_STR_ADDR_LEN);
  strncpy (Res.SrcMac, strAddresse2, WLAN_STR_ADDR_LEN);
  strncpy (Res.BssId, strAddresse3, WLAN_STR_ADDR_LEN);
  /* Log */
  LogDetectedClient (d80211b_Header->a3.a1);
  LogDetectedClient (d80211b_Header->a3.a2);
  LogPutSN (d80211b_Header->a3.a2, Res.Signal);
  LogPutRate (d80211b_Header->a3.a2, Res.Rate);
  LogPutSeqNum (d80211b_Header->a3.a2, Res.DestMac, d80211b_Header->a3.seq);

  return 1;			/* All is OK */
}

// Function to filter not printable caractere in some SSID
UINT8
FilterESSID (char *ESSID)
{
  UINT8 i;
  UINT8 modified = FALSE;

  for (i = 0; i < strlen (ESSID); i++)
    {
      if (!isprint (ESSID[i]))
	{
	  ESSID[i] = '?';
	  modified++;
	}
    }
  ESSID[i] = 0x00;		// In case OF ... :-)
  return modified;
}

void
ProcessTagBits (unsigned char *packet, int len, int FrameType)
{
  static UINT8 NbTagSSID = 0;
  unsigned char *varBits;
  p80211_hdr_t *d80211b_Header;
  int tagType, tagLen;
  char Buff[0x101];
  UINT8 MaxRate, i;

  /* Get only that we want */
  (unsigned char *) d80211b_Header = packet;
  (unsigned char *) varBits = packet + sizeof (p80211_hdr_a3_t);

  /* Do a correction on the size of the Fixed Field */
  switch (FrameType)
    {
    case WLAN_FSTYPE_ASSOCREQ:
      break;
    case WLAN_FSTYPE_ASSOCRESP:
      break;
    case WLAN_FSTYPE_REASSOCREQ:
      break;
    case WLAN_FSTYPE_REASSOCRESP:
      break;
    case WLAN_FSTYPE_PROBEREQ:
      /* Size of FixedField = 0 */
      break;
    case WLAN_FSTYPE_PROBERESP:
      break;
    case WLAN_FSTYPE_BEACON:
      /* Size of FixedField = 0x0C */
      varBits += sizeof (FixedField_t);
      break;
    case WLAN_FSTYPE_ATIM:
      break;
    }
  NbTagSSID = 0;

  /* Get the tagged values 
   * There is a type and a length field
   * Followed by tha actual data 
   */

  tagType = 0;
  debug (1, "TagType(len) = ");

  /* -4 is the CRC, I don't want CRC :-)) */
  while ((varBits < (packet + len - 4)) && (tagType != 0xFF))
    {

      tagType = varBits[0];
      tagLen = varBits[1];
      varBits += 2;
      debug (1, "%02d(%02d),", tagType, tagLen);
      switch (tagType)
	{
	case WLAN_EID_SSID:	/* 0 */
	  strncpy (Res.SSID, varBits, tagLen);
	  FilterESSID (Res.SSID);
	  LogPutSSID (d80211b_Header->a3.a2, d80211b_Header->a3.a3, Res.SSID);

	  // Catch wellenreiter probes
	  if (!strncmp (Res.SSID, "this_is_used_for_wellenreiter", 32))
	    {
	      warning
		("\nWARNING : Somebody use wellenreiter to probe YOU!\n");
	    }
	  // Catch Windows XP probes
//      if (!strncmp (Res.SSID, WIN_XP_PROBE_SSID, 32)) {
//      warning ("\nWARNING : Windows XP is HERE !\n");
//      }

	  NbTagSSID++;
	  break;
	case WLAN_EID_SUPP_RATES:	/* 1 */
	  MaxRate = (*varBits) & 0x7F;
	  for (i = 0; i < tagLen; i++)
	    {
	      if (MaxRate < (*(varBits + i) & 0x7F))
		MaxRate = *(varBits + i) & 0x7F;
	    }
	  LogPutMaxRate (d80211b_Header->a3.a2, (UINT8) MaxRate);
	  break;
	case WLAN_EID_FH_PARMS:	/* 2 */
	  break;
	case WLAN_EID_DS_PARMS:	/* 3 */
	  if (tagLen == 1)
	    {
	      Res.Channel = *varBits;
	      LogPutChannel (d80211b_Header->a3.a2, d80211b_Header->a3.a3,
			     Res.Channel);
	    }
	  break;
	case WLAN_EID_CF_PARMS:	/* 4 */
	  SET_TYPE_BIT_AP (Res.TypeOfClient);
	  break;
	case WLAN_EID_TIM:	/* 5 */
	  // The TIM information is only present within beacon frames generated by APs
	  SET_TYPE_BIT_AP (Res.TypeOfClient);
	  break;
	case WLAN_EID_IBSS_PARMS:	/* 6 */
	  CLEAR_TYPE_BIT_AP (Res.TypeOfClient);
	  SET_TYPE_BIT_P2P (Res.TypeOfClient);
	  break;
	case WLAN_EID_CHALLENGE:	/* 16 */
	  break;
	  // Specific Vedor TAG
	case 133:		/* CISCO AIRONET */
	  // It's an AP NAME, Thx for the information :-)
	  strncpy (Buff, (varBits + 0x0A), 0x10);
	  UpdateOtherInformation (Buff);
	  LogPutOtherInformation (d80211b_Header->a3.a2,
				  Res.OtherInformation);
	  break;
	case 0xFF:
	  // End of Tag list
	  break;
	  // Other unknowned TAG
	default:
	  snprintf (Buff, 32, "An unknow tag is detected : %02X", tagType);
	  UpdateOtherInformation (Buff);
	  LogPutOtherInformation (d80211b_Header->a3.a2,
				  Res.OtherInformation);
	  debug (1,
		 "\nWARNING -- This TagType is unknowed =%02X\n"
		 "Please send one of this packet to the author\n", tagType);
	  break;
	}
      varBits += tagLen;
    }

  debug (1, "\nRes.TypeOfClient=%x\n", Res.TypeOfClient);

  if (NbTagSSID > 1)
    {
      // It seem to be a Windows XP STA :-)
      debug (1, "Tag 0 (SSID) is send more than one time (%d)\n", NbTagSSID);
      snprintf (Buff, 85,
		"Tag 0 (SSID) is send more than one time (%.2d) - "
		"This STA is probably under Windows XP", NbTagSSID);
      UpdateOtherInformation (Buff);
      LogPutOtherInformation (d80211b_Header->a3.a2, Res.OtherInformation);
    }
}

/* This function parse commandline and fill option ... */
void
ParseCommandLine (int argc, char **argv,
		  UINT8 * SingleChannel,
		  char **OutFileName, char **OutDumpFileName,
		  char **OutDotFileName,
		  UINT8 * DebugLevel,
		  unsigned int *TimeToSleepBeforeChangeChannel,
		  char *devname, UINT8 * ChannelHop,
		  UINT32 * MaxPacket, UINT8 * DateFormat,
		  UINT8 * DoNotDisplay, UINT8 * IDS_is_ON,
		  UINT8 * CheckScreenSize, UINT8 * TypeOfCard)
{
  char c;
  int n;


  while ((c =
	  getopt (argc, argv, "v:F:S:W:D:i:t:Vh?H:dM:N:IG:w:k:C:c")) != EOF)
    {
      switch (c)
	{
	case 'S':		// Set channel
	  *SingleChannel = atoi (optarg);
	  if (*SingleChannel < CHANNEL_MIN || *SingleChannel > CHANNEL_MAX)
	    {
	      fprintf (stderr, "Error : Channel must be between %d and %d\n",
		       CHANNEL_MIN, CHANNEL_MAX);
	      VersionAndBye ();
	    }
	  break;
	case 'F':		// Output file for real time information
	  if (strncmp ("auto", optarg, 5))
	    {
	      *OutFileName = optarg;
	    }
	  else
	    {

	    }
	  break;
#ifdef WITH_WIRETAP
	case 'W':		// PCAP output file
	  *OutDumpFileName = optarg;
	  break;
#endif
	case 'D':		// .DOT output file
	  *OutDotFileName = optarg;
	  break;
	case 'V':		// Version
	  VersionAndBye ();
	  break;
	case 'H':		// number of Hop for channel rotation
	  *ChannelHop = atoi (optarg);
	  if (*ChannelHop < HOP_MIN || *ChannelHop > HOP_MAX)
	    {
	      fprintf (stderr, "Error : Hop must be between %d and %d\n",
		       HOP_MIN, HOP_MAX);
	      VersionAndBye ();
	    }
	  break;
	case 'v':		// Verbose
	  *DebugLevel = atoi (optarg);
	  if (*DebugLevel < 1 || *DebugLevel > MAX_DEBUG_LEVEL)
	    *DebugLevel = MAX_DEBUG_LEVEL;
	  n = 0;		// Display all packets
	  break;
	case 't':		// Time to sleep before change channel
	  *TimeToSleepBeforeChangeChannel = atoi (optarg);
	  if (*TimeToSleepBeforeChangeChannel < 1
	      || *TimeToSleepBeforeChangeChannel > 10000)
	    *TimeToSleepBeforeChangeChannel = 10000;	/* Ten seconds is enougth */
	  break;
#ifdef LWNG_13
	case 'r':
	  *TimeToSleepBeforeChangeChannel = 0;	/* a little wait time :) */
	  break;
#endif
	case 'i':		// Interface
	  strncpy (devname, optarg, 5);
	  break;
	case 'd':		// Change date format : human/computer readable 
	  *DateFormat = 0;
	  break;
	case 'M':		// Max of packet to capture
	  // Check if outbound
	  if ((atoi (optarg) < 0)
	      || (atoi (optarg) > ((UINT32) 256 * 256 * 256 * 256)))
	    *MaxPacket = atoi (optarg);
	  break;
	case '?':		// Help
	case 'h':		// Help too
	  HelpAndBye ();
	  break;
	case 'I':		// IDS
	  *IDS_is_ON = TRUE;	// IDS is not active by default
	  //*OutIDSFileName = optarg;
	  break;
	case 'N':		// Don't display some packet (only in display)
	  n = 0;
	  fprintf (stdout, "Do not display: ");
	  // do while until end of option or 16 sub-option ...
	  while ((optarg[n] != 0) && (n < 16))
	    {
	      switch (optarg[n])
		{
		case 'A':
		case 'a':
		  *DoNotDisplay = (*DoNotDisplay) | (1 << 0);
		  fprintf (stdout, "ACK ");
		  break;
		case 'B':
		case 'b':
		  *DoNotDisplay = (*DoNotDisplay) | (1 << 1);
		  fprintf (stdout, "BEACON ");
		  break;
		case 'C':
		case 'c':
		  *DoNotDisplay = (*DoNotDisplay) | (1 << 2);
		  fprintf (stdout, "CONTROL ");
		  break;
		case 'D':
		case 'd':
		  *DoNotDisplay = (*DoNotDisplay) | (1 << 3);
		  fprintf (stdout, "DATA ");
		  break;
		default:
		  break;
		}
	      n++;
	    }
	  fprintf (stdout, "\n");
	  if (*DebugLevel > 1)
	    {
	      fprintf (stdout,
		       "Verbose level is more than 1 so all packets are display\n");
	    }
	  break;
	case 'G':		// ???
	  break;
	case 'w':		// Save Weak IV packet
	  break;
	case 'k':		// Save Keystream of Authen with IV
	  break;
	case 'C':		// Choice of CARD
	  // CISCO driver is very instable on my computer, so I can't test it
	  //    || (!strcmp (optarg, "cisco_cvs")) 
	  //    || (!strcmp (optarg, "cisco"))
	  if ((!strcmp (optarg, "prism")) || (!strcmp (optarg, "hostap")))
	    {
	      if (!strcmp (optarg, "cisco"))
		{
		  *TypeOfCard = CISCO_CARD;
		}
	      if (!strcmp (optarg, "cisco_cvs"))
		{
		  *TypeOfCard = CISCO_CVS_CARD;
		}
	      if (!strcmp (optarg, "prism"))
		{
		  *TypeOfCard = WLAN_NG_CARD;
		}
	      if (!strcmp (optarg, "hostap"))
		{
		  *TypeOfCard = HOSTAP_CARD;
		}
	    }
	  else
	    {
	      fprintf (stderr,
		       "Only 'prism', 'hostap', 'cisco_cvs' or 'cisco'"
		       " card is allowed.\n");
	      VersionAndBye ();
	    }
	case 'c':		// Do Not CheckScreenSize
	  *CheckScreenSize = FALSE;
	  break;
	default:
	  break;
	}
    }

  /* Summary of options */
  fprintf (stdout, "Use of interface:%s\n", devname);
  if (*DebugLevel == MAX_DEBUG_LEVEL)
    fprintf (stdout, "Debug level is Crazy level ;-))\n");
  else if (*DebugLevel)
    fprintf (stdout, "Debug level is %d\n", *DebugLevel);
  if (*OutDumpFileName != NULL)
    fprintf (stdout, "Output data to %s in tcpdump file format\n",
	     *OutDumpFileName);
  if (*OutFileName != NULL)
    fprintf (stdout, "Output information also to %s\n", *OutFileName);
  if (*SingleChannel)
    fprintf (stdout, "Scan only channel %d\n", *SingleChannel);
  else
    {
      if (*TimeToSleepBeforeChangeChannel != 0)
	{
	  fprintf (stdout, "I sleep %dms before change channel\n",
		   *TimeToSleepBeforeChangeChannel);
	  fprintf (stdout, "I try to scan %02d channels per second\n",
		   1000 / (*TimeToSleepBeforeChangeChannel));
	}
      else
	fprintf (stdout, "I don't sleep any s before change channel\n");
    }
}

void
ChannelToStr (char *strChannel, UINT8 Channel, UINT8 SChannel)
{
  if (Channel == 0)
    {
      sprintf (strChannel, "%02d(%02d)", Channel, SChannel);
    }
  else
    {
      sprintf (strChannel, "%02d(%+02d)", Channel, Channel - SChannel);
    }
  debug (1, "S-C=%02d-%02d\n", Channel, SChannel);
}
