/* File   : spak_func.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Functions for use by the programs in the spak release.  The
 *          individual functions are detailed before the function below.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"          /* AUTO_OPT_PAD         */
#include "spak_func.h"       /* Function prototypes  */
#include "spak_func_conf.h"  /* INBUF                */
#include <stdio.h>           /* stderr               */
#include <stdlib.h>          /* strtoul(), realloc() */
#include <string.h>          /* memcpy()             */
#include <limits.h>          /* ULONG_MAX            */
#include <errno.h>           /* errno                */
#include <netdb.h>           /* gethostbyname()      */
#include <unistd.h>          /* STDIN_FILENO         */
#include <sys/ioctl.h>       /* ioctl()              */
#include <sys/types.h>       /* socket types         */
#include <sys/socket.h>      /* socket()             */

#ifdef SunOS
#include <sys/sockio.h>      /* SIOCGENADDR          */
#endif  /* SunOS */

/* Set defaults if not defined previously or not valid.  Do not change. */
#ifndef INBUF
# define INBUF 512
#elif (INBUF < 2)
# define INBUF 512
#endif  /* INBUF */


/* Function: append_data
 * Input   : A file name from which to read data, (if the name is -, then
 *           stdin will be used for input), an existing string to which the
 *           data should be appended, (may be NULL), and an offset into the
 *           passed string where the new data should be placed.
 * Output  : The read data is placed into the passed string starting at the
 *           passed offset into the string.
 * Return  : A pointer to the new string.
 */
char *append_data (char *data, int *offset, char *infile) {
  char *inbuf;
  int new_data_len;

  /* Read the data */
  if (((inbuf = read_data(infile, &new_data_len)) == NULL) ||
      (new_data_len == 0))
    return(data);

  /* Reallocate space to hold the new data */
  if ((data = (char *) realloc(data, *offset + new_data_len)) == NULL) {
    fprintf(stderr, "append_data(): Out Of Memory\n");
    exit(1);
  }

  /* Copy the new data */
  if ((data = memcpy(data + *offset, inbuf, new_data_len)) == NULL) {
    fprintf(stderr, "memcpy(): Unknown Error\n");
    exit(1);
  }

  *offset += new_data_len;
  return(data);
}


/* Function: checksum
 * Input   : A pointer to a header and the length in bytes of the header.
 * Return  : A 16-bit, ones compliment checksum value for the passed header.
 */
unsigned short checksum (void *header, int len) {
  unsigned long retval = 0;
  unsigned short *ptr;
  
  /* Add all of the data values, add the overflow, chop off all but */
  /* the first 16 bits, and then take the ones compliment.          */
  if (header != NULL) {
    for (ptr = (unsigned short *) header; len > 1; len -= 2)
      retval += *ptr++;
    retval = (retval >> 16) + (retval & 0xFFFF);
    retval += (retval >> 16);
    retval = ~retval & 0xFFFF;
  }

  return((short) retval);
}


/* Function: get_default
 * Input   :
 * Output  :
 * Return  : -1 if error, otherwise 1.
 */
int get_default (struct ifreq *ifr, int type) {
  int sockid;

  sockid = socket(AF_INET, SOCK_DGRAM, 0);
  ifr->ifr_addr.sa_family = AF_INET;

  switch(type) {

    /* Get the interface IP address */ 
  case IP:
    if (ioctl(sockid, SIOCGIFADDR, ifr) < 0) {
      perror("SIOCGIFADDR");
      return(-1);
    }
    break;

    /* Get the interface MAC address */ 
  case MAC:
#ifdef SunOS
    if (ioctl(sockid, SIOCGENADDR, ifr) < 0) {
      perror("SIOCGENADDR");
      return(-1);
    }
#endif  /* SunOS */
#ifdef LINUX
    if (ioctl(sockid, SIOCGIFHWADDR, ifr) < 0) {
      perror("SIOCGIFHWADDR");
      return(-1);
    } 
#endif  /* LINUX */

  default:
    return(-1);
  }

  return(1);
}


/* Function: my_atoi
 * Input   : An string representation of an integer, the maximum number of
 *           bits the return value is allowed to have, and the name of the
 *           argument being assigned, (for error messages).
 * Return  : The integer corresponding to the passed string.
 */
int my_atoi (char *value, int width, char *field_name) {
  char **endptr;
  unsigned long int retval, mask;

  /* Allocate space */
  if ((endptr = (char **) malloc(sizeof(char *))) == NULL) {
    fprintf(stderr, "Out of Memory\n");
    exit(1);
  }
  *endptr = NULL;

  /* Test if a value was given to the argument */
  if (value == NULL) {
    fprintf(stderr, "Argument %s requires an argument.\n",
	    (field_name == NULL) ? "<unknown>" : field_name);
    exit(1);
  }

  /* Convert the value to an integer */
  if (((retval = strtoul(value, endptr, 0)) == ULONG_MAX) &&
      (errno == ERANGE)) {
    fprintf(stderr, "Value for %s argument is too large\n",
	    (field_name == NULL) ? "<unknown>" : field_name);
    exit(1);
  }
  if (**endptr != '\0') {
    fprintf(stderr, "Value for %s argument must be an integer number.\n",
	    (field_name == NULL) ? "<unknown>" : field_name);
    exit(1);
  }

  /* Test if the value is in range - The width != 32 test is a hack */
  mask = (~0 << width);
  if ((width != 32) && (retval & mask)) {
    fprintf(stderr, "Value of %s must be from 0 to %d.\n",
	    (field_name == NULL) ? "<unknown>" : field_name, ~mask);
    exit(1);
  }

  return(retval);
}


/* Function: getaddrbyname
 * Input   : A host name to lookup.
 * Return  : The numerical address of the given host name in network byte
 *           order.
 */
unsigned long getaddrbyname (char *name) {
  struct hostent *data;

  if ((data = gethostbyname(name)) == NULL) {
    fprintf(stderr, "Could not resolve %s.  Try using an IP number.\n", name);
    exit(1);
  }

  return(* (long *) *data->h_addr_list);
}


/* Function: get_options
 * Input   : A file to read the options from, and a pointer to an integer.
 *           The input file should be in plain text and contain a list of
 *           8-bit numbers, (in decimal form), seperated by white space.  The
 *           integer pointer passed will lose any contents it may have.
 * Output  : The number of values in the return buffer are stored in the
 *           passed integer pointer.
 * Return  : Each number in the input file converted to its 8-bit value.
 */
char *get_options (char *input_file, int *size) {
  char c, inbuf[4];
  char *ptr = NULL, *retval = NULL;
  FILE *ifp;
  int place = 0, pos = 0;
  unsigned short flag = 0;

  /* Open the data file */
  if ((ifp = fopen(input_file, "r")) == NULL) {
    fprintf(stderr, "Fatal Error: Unable to open %s for reading\n",
	    input_file);
    exit(1);
  }

  *size = 0;

  /* Read the data */
  while (c = getc(ifp)) {
    if (pos > 4) {
      fprintf(stderr, "Option value is too large, (each number must be less than 256).\n");
      if (strcmp(input_file, "-") != 0)
	fprintf(stderr, "Input file with error: %s\n", input_file);
      exit(1);
    }

    /* Test if the end of a comment is reached */
    if (flag == 1) {
      if (c == EOF)
	break;
      if (c == '\n')
	flag = 0;
      continue;
    }

    /* Test if an end-of-integer marker is found, (white space, EOF, or */
    /* comment begining character).                                     */
    if (((c == ' ') || (c == '\n') || (c == '\t') || (c == EOF) ||
	 (c == '#')) && (pos != 0)) {

      /* Resize the return buffer */
      if ((++(*size) > place * INBUF) &&
	  ((retval = (void *) realloc(retval, INBUF * ++place))) == NULL) {
	fprintf(stderr, "Out Of Memory\n");
	exit(1);
      }

      /* Set the pointer to the end of the return data */
      if (ptr == NULL)
	ptr = retval;
      else ++ptr;

      /* Enter the new data into the return data and reset flags */
      inbuf[pos] = '\0';
      *ptr = my_atoi(inbuf, 8, "option data");
      pos = 0;
    }

    /* Skip over comments */
    else if (c == '#')
      flag = 1;

    /* Skip over blank lines */
    else if (c == '\n')
      continue;

    else inbuf[pos++] = c;

    if (c == EOF)
      break;
  }

#ifdef AUTO_OPT_PAD
  /* Pad the options so they end on a 32-bit boundry and return the data */
  if (*size % 4) {
    if (((*size + (*size % 4)) > (INBUF * place)) &&
	((retval = (void *) realloc(retval, *size + (*size % 4))) == NULL)) {
      fprintf(stderr, "Fatal Error: Out of Memory\n");
      exit(1);
    }
    while (*size % 4) {
      *(++ptr) = 0;
      *size++;
    }
  }
#endif  /* AUTO_OPT_PAD */

  return(retval);
}


/* Function: read_data
 * Input   : A file name to read the data from, and a pointer to an integer.
 *           If the filename is -, data will be read from stdin.
 * Output  : The size of the data read is stored in the passed integer
 *           pointer.
 * Return  : The data read.
 */
char *read_data (char *input_file, int *size) {
  char *retval = NULL;
  char tmpchar;
  FILE *ifp;
  int place = 0;

  *size = 0;

  /* Open the input stream */
  if (input_file == NULL) {
    fprintf(stderr, "Argument -i requires a file name\n");
    exit(1);
  }
  if (strcmp(input_file, "-") == 0) {
    if ((ifp = fdopen(STDIN_FILENO, "r")) == NULL) {
      fprintf(stderr, "Unable to open stdin for reading\n");
      exit(1);
    }
  } else if ((ifp = fopen(input_file, "r")) == NULL) {
    fprintf(stderr, "Unable to open %s for reading\n", input_file);
    exit(1);
  }

  /* Read the data */
  while ((tmpchar = fgetc(ifp)) != EOF) {
    if ((place >= (INBUF * place)) &&
	((retval = (char *) realloc(retval, INBUF * ++place)) == NULL)) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    retval[*size] = tmpchar;
    *size += 1;
  }
  fclose(ifp);

  return(retval);
}


/* Function: write_packet
 * Input   : A file name to write to or NULL is stdout should be used, a
 *           pointer to the data to write, and the length of the data.
 * Output  : The passed data is written to the passed output file.
 */
void write_packet (char *output_file, void *packet, unsigned int length) {
  FILE *ofp = NULL;

  if (packet == NULL) {
    fprintf(stderr, "Attempt to write zero length data\n");
    return;
  }

  /* Open the output stream */
  if (output_file != NULL) {
    umask(022);
    if ((ofp = fopen(output_file, "w")) == NULL) {
      fprintf(stderr, "Fatal Error: Unable to open %s for writing\n",
              output_file);
      exit(1);
    }
  } else if ((ofp = fdopen(STDOUT_FILENO, "w")) == NULL) {
    fprintf(stderr, "Fatal Error: Unable to open stdout for writing\n");
    exit(1);
  }

  /* Write the data */
  if (fwrite(packet, 1, length, ofp) != length)
    fprintf(stderr, "Warning: All packet data may not have been written\n");
}
