/* File   : sendeth.c
 * Author : J Hadi Salim <ad828@freenet.carleton.ca>
 * Purpose: Send an ethernet frame.
 *
 * Notes  : This file has been modified by Karyl F. Stein <xenon@xenos.net>.
 *          Any bugs are my fault.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>

#ifndef LINUX
main (void) {
  fprintf(stderr, "Sorry, sendeth only works under Linux right now\n");
}
#else

#include "sendeth_conf.h"

#ifdef SunOS
#include <sys/types.h>         /* u_short */
#include <sys/socket.h>        /* AF_INET */
#endif  /* SunOS */

#include <fcntl.h>             /* O_RDONLY              */
#include <netinet/in.h>        /* Socket data structure */
#include <unistd.h>            /* read()                */
#include "ethernet.h"          /* Ethernet definitions */
#include "datadisp.h"          /* Function prototypes   */
#include "spak_func.h"         /* Function prototypes   */

/* Values used by the flag variable */
#define VERBOSE  0x01
#define VVERBOSE 0x02
#define QUIET    0x04

/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit.
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s\n", name);
  fprintf(stderr, "       [-d <device>] [-i <input_file>] [-v]\n");
  fprintf(stderr, " -d   Sets the device through which to send data out to <device>.  ");
#ifdef DEF_ETH_DEVICE
  fprintf(stderr, "Default:\n      %s\n", DEF_ETH_DEVICE);
#else
  fprintf(stderr, "Unless\n      this is defined, a valid interface will be guessed by the program.\n");
#endif  /* DEF_ETH_DEVICE */
  fprintf(stderr, " -i   Read the frame data from the file <input_file>.  If <input_file> is -,\n");
  fprintf(stderr, "      or this argument is not given, stdin is used.\n");
  fprintf(stderr, " -v   Verbose output.\n");
  fprintf(stderr, " -vv  Very verbose output, (packet data is printed if possible).\n");
  exit(0);
}


int main (int argc, char *argv[]) {
  char *data = NULL, *device = NULL, *ptr;
  int data_len = 0, flag = 0, sockid, tmpint;
  my_ethhdr *frame;
  struct sockaddr sa;

  /* Parse the arguments */
  tmpint = 0;
  while (++tmpint < argc) {
    if (*(ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Output Device */
    if (strcmp(ptr, "d") == 0) {
      if (argv[++tmpint] == NULL) {
	if (!flag & QUIET)
	  fprintf(stderr, "Warning: Argument -d expects a device\n", argv[0]);
	continue;
      }
      if (device != NULL) {
	if (!flag & QUIET)
	  fprintf(stderr, "Warning: Multiple -d arguments found.  Using %s\n",
		  argv[tmpint]);
	free(device);
      }
      if ((device = (char *) malloc(strlen(argv[tmpint]) + 1)) == NULL) {
	fprintf(stderr, "%s Error: Out Of Memory\n", argv[0]);
	exit(1);
      }
      strcpy(device, argv[tmpint]); 
    }

    /* Print help screen */
    else if (strcmp(ptr, "h") == 0)
      usage(argv[0]);

    /* Input Data */
    else if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Set verbose flags */
    else if (*ptr == 'v') {
      flag |= VERBOSE;
      if (strcmp(ptr, "vv") == 0)
	flag |= VVERBOSE;
    }

    else usage(argv[0]);
  }

  if (device == NULL) {
#ifdef DEF_ETH_DEVICE
    if ((device = (char *) malloc(strlen(DEF_ETH_DEVICE) + 1)) == NULL) {
      fprintf(stderr, "Out Of Memory\n");
      exit(1);
    }
    strcpy(device, DEF_ETH_DEVICE);
#else
    fprintf(stderr, "%s: You must specify an ethernet device to use.\n",
	    argv[0]);
#endif  /* DEF_ETH_DEVICE */
  }

  if (data_len == 0) {
    fprintf(stderr, "%s: Attempt to send zero length data\n", argv[0]);
    exit(1);
  }
  if ((data_len + ETH_HLEN) < ETH_ZLEN) {
    tmpint = ETH_ZLEN - (ETH_HLEN + data_len);
    fprintf(stderr,"\n%s: Frame possibly under-sized by %d bytes\n",
	    argv[0], tmpint);
  }

  frame = (my_ethhdr *) data;

  if (flag & VERBOSE) {
    print_eth_data(frame);
    if (flag & VVERBOSE) {
      tmpint = print_ip_data((struct ip *) (data + ETH_HLEN));
      if (ip_protocol == IPPROTO_TCP) {
	print_tcp_data((struct tcphdr *) (data + tmpint));
      } else {
	if (ip_protocol == IPPROTO_UDP) {
	  print_udp_data((struct udphdr *) (data + tmpint));
	} else {
	  fprintf (stderr,"\nDont know how to format: Unknown IP protocl number %d\n", ip_protocol);
	}
      }
    }
  }

  /* Send the frame */
  sa.sa_family = AF_INET;
  strcpy(sa.sa_data, device);

  if((sockid = socket(AF_INET, SOCK_PACKET, htons(ETH_P_ALL))) < 0) {
    perror("Socket call failed:");
    exit(-1);
  }

  fcntl(sockid, F_SETFL, O_NDELAY);

  if ((tmpint = sendto(sockid, data, data_len, 0, &sa, sizeof(sa))) < 0) {
    perror("Sendto failed:");
    exit(-1);
  }

  if (flag & VERBOSE)
    printf("\n============ Sent %d Bytes On Device %s ============\n",
	   tmpint, device);
}

#endif  /* LINUX */
