/* File   : makeip.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Create an IP packet.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>

#ifdef SunOS
#include <netinet/in.h>       /* sockaddr_in */
#include <arpa/nameser.h>     /* MAXDNAME    */
#include <resolv.h>           /* in_addr     */
#include <netinet/in_systm.h> /* n_long      */
#endif /* SunOS */

#ifdef BSDI31
#include <netinet/in.h>       /* sockaddr_in */
#include <netinet/in_systm.h> /* n_long      */
#endif  /* BSDI31 */

#include <string.h>           /* memcpy()            */
#include <netinet/ip.h>       /* IP header struct    */
#include "makeip_conf.h"      /* IP header defaults  */
#include "makeip_func.h"      /* Function prototypes */
#include "spak_func.h"        /* Function prototypes */
#include "datadisp.h"         /* Function prototypes */

/* Values used by the flag variable */
#define GOT_CKSUM 0x01
#define GOT_IHL   0x02
#define GOT_OFF   0x04
#define GOT_TL    0x08
#define VERBOSE   0x10

/* Values used to set various bits */
#define BIT_TOS7 0x01
#define BIT_TOS6 0x02
#define BIT_R    0x04
#define BIT_T    0x08
#define BIT_D    0x10
#define BIT_F1   0x8000

/* Set defaults if not defined previously or not valid */
#ifndef IP_DEFAULT_PROTOCOL
# define IP_DEFAULT_PROTOCOL 6
#endif  /* IP_DEFAULT_PROTOCOL */
#ifndef IP_DEFAULT_TTL
# define IP_DEFAULT_TTL 64
#endif  /* IP_DEFAULT_TTL */
#ifndef IP_DEFAULT_VERSION
# define IP_DEFAULT_VERSION 4
#endif  /* IP_DEFAULT_VERSION */
#ifndef IP_DEFAULT_HEADER_SIZE
# define IP_DEFAULT_HEADER_SIZE 20
#endif  /* IP_DEFAULT_HEADER_SIZE */


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s <src> <dest>\n", name);
  fprintf(stderr, "       [-h] [-vr <version>] [-p <protocol>] [-pr <priority>] [-id <id>]\n");
  fprintf(stderr, "       [-hl <hdr_len>] [-c <hdr_cksum>] [-tl <total_len>] [-s6] [-s7] [-sd]\n");
  fprintf(stderr, "       [-sf] [-smf] [-snf] [-sr] [-st] [-fo <frag_offset>] [-ttl <time>]\n");
  fprintf(stderr, "       [-i <input_file>] [-o <output_file>] [-of <opt_file>] [-v]\n");
  fprintf(stderr, " <src>    Source address for the packet, (in numerical or word form).\n");
  fprintf(stderr, " <dest>   Destination address for the packet, (in numerical or word form).\n");
  fprintf(stderr, " -c       Set the header checksum to <hdr_cksum>.  Default: correct value\n");
  fprintf(stderr, " -fo      Set the fragment offset to <fragment_offset>.  Default: 0\n");
  fprintf(stderr, " -h       Print this help message.\n");
  fprintf(stderr, " -hl      Set the header length to <hdr_len>.  Default: correct value\n");
  fprintf(stderr, " -i       Read the packet data from the file <input_file>.  If <input_file>\n");
  fprintf(stderr, "          is -, stdin will be used.\n");
  fprintf(stderr, " -id      Set the packet ID to <id>.  Default: 0\n");
  fprintf(stderr, " -o       Print the packet data to <output_file>.  Default: stdout\n");
  fprintf(stderr, " -of      Set the options listed in <opt_file>, (under construction).\n");
  fprintf(stderr, " -p       Set the protocol to <protocol>, (decimal number).  Default: %d\n", IP_DEFAULT_PROTOCOL);
  fprintf(stderr, " -pr      Set the priority to <priority>.  Default: 0\n");
  fprintf(stderr, " -tl      Set the total packet length.  Default: correct value\n");
  fprintf(stderr, " -ttl     Set time to live to <time>.  Default: %d\n",
	  IP_DEFAULT_TTL);
  fprintf(stderr, " -s6      Set the 6th bit in the type of service field.  Default: not set\n");
  fprintf(stderr, " -s7      Set the 7th bit in the type of service field.  Default: not set\n");
  fprintf(stderr, " -sd      Set the \"low delay\" bit.  Default: not set\n");
  fprintf(stderr, " -sf      Set the first bit in the flag field.  Default: not set\n");
  fprintf(stderr, " -sdf     Set the \"don't fragment\" bit.  Default: not set\n");
  fprintf(stderr, " -smf     Set the \"more fragments\" bit.  Default: not set\n");
  fprintf(stderr, " -sr      Set the \"high relibility\" bit.  Default: not set\n");
  fprintf(stderr, " -st      Set the \"high throughput\" bit.  Default: not set\n");
  fprintf(stderr, " -vr      Set the IP version to <version>.  Default: %d\n",
	  IP_DEFAULT_VERSION);
  fprintf(stderr, " -v       Turn on verbose mode, (print packet data).\n");
  exit(0);
}


int main (int argc, char *argv[]) {
  char *data = NULL, *option_file = NULL, *options = NULL;
  char *output_file = NULL, *ptr;
  int data_len = 0, flag = 0, opt_len = 0, tmpint;
  struct ip *packet;
  void *datagram;

  /* Make sure the correct number of arguments are given */
  if (argc <= 2)
    usage(argv[0]);

  /* Create a new packet */
  datagram = (void *) new_ip_packet();
  packet = (struct ip *) datagram;

  /* Set the source and destination addresses */
#ifdef INET_ATON
  if (inet_aton(argv[1], (struct ip *) &packet->ip_src) == 0)
#else
  if ((packet->ip_src.s_addr = inet_addr(argv[1])) == -1)
#endif /* INET_ATON */
    packet->ip_src.s_addr = getaddrbyname(argv[1]);
#ifdef INET_ATON
  if (inet_aton(argv[2], &packet->ip_dst) == 0)
#else
  if ((packet->ip_dst.s_addr = inet_addr(argv[2])) == -1)
#endif /* INET_ATON */
    packet->ip_dst.s_addr = getaddrbyname(argv[2]);

  /* Parse the arguments */
  tmpint = 2;
  while (++tmpint < argc) {
    if (*(ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Header Checksum */
    if (strcmp(ptr, "c") == 0) {
      packet->ip_sum = my_atoi(argv[++tmpint], 16, "-c");
      flag |= GOT_CKSUM;
    }

    /* Fragment offset */
    else if (strcmp(ptr, "fo") == 0)
      packet->ip_off = (packet->ip_off & 0xE000) |
	(my_atoi(argv[++tmpint], 13, "-fo"));

    /* Header Length */
    else if (strcmp(ptr, "hl") == 0)
      packet->ip_hl = my_atoi(argv[++tmpint], 4, "-hl");

    /* Input Data */
    else if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Identification */
    else if (strcmp(ptr, "id") == 0)
      packet->ip_id = my_atoi(argv[++tmpint], 16, "-id");

    /* Output File */
    else if (strcmp(ptr, "o") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -o argument expects a file name\n");
      else {
        if (output_file != NULL)
          free(output_file);
        output_file = argv[tmpint];
      }
    }

    /* Option File */
    else if (strcmp(ptr, "op") == 0) {
      if (argv[++tmpint] == NULL)
	fprintf(stderr, "Warning: The -op argument expects a file name\n");
      else {
	if (option_file != NULL)
	  free(option_file);
	option_file = argv[++tmpint];
      }
    }

    /* Protocol */
    else if (strcmp(ptr, "p") == 0)
      packet->ip_p = my_atoi(argv[++tmpint], 16, "-p");

    /* Priority */
    else if (strcmp(ptr, "pr") == 0)
      packet->ip_tos = (my_atoi(argv[++tmpint], 3, "-pr") << 5) |
	packet->ip_tos;

    /* Set various bits */
    else if (* (char *) ptr == 's') {
      ptr++;

      /* Set delay bit */
      if (strcmp(ptr, "d") == 0)
	packet->ip_tos |= BIT_D;

      /* Set throughput bit */
      else if (strcmp(ptr, "t") == 0)
	packet->ip_tos |= BIT_T;

      /* Set relibility bit */
      else if (strcmp(ptr, "r") == 0)
	packet->ip_tos |= BIT_R;

      /* Set bit number 6 in type of service field */
      else if (strcmp(ptr, "6") == 0)
	packet->ip_tos |= BIT_TOS6;

      /* Set bit number 7 in type of service field */
      else if (strcmp(ptr, "7") == 0)
	packet->ip_tos |= BIT_TOS7;

      /* Set first flag bit (invalid) */
      else if (strcmp(ptr, "f") == 0)
	packet->ip_off |= BIT_F1;

      /* Set no fragment */
      else if (strcmp(ptr, "df") == 0)
	packet->ip_off |= IP_DF;

      /* Set more fragment bit */
      else if (strcmp(ptr, "mf") == 0)
	packet->ip_off |= IP_MF;

      else usage(argv[0]);
    }

    /* Total Length */
    else if (strcmp(ptr, "tl") == 0) {
      packet->ip_len = my_atoi(argv[++tmpint], 16, "-tl");
      flag |= GOT_TL;
    }

    /* Time to Live */
    else if (strcmp(ptr, "ttl") == 0)
      packet->ip_ttl = my_atoi(argv[++tmpint], 8, "-ttl");
 
    /* Set verbose flag */
    else if (strcmp(ptr, "v") == 0)
      flag |= VERBOSE;

    /* IP Version */
    else if (strcmp(ptr, "vr") == 0)
      packet->ip_v = my_atoi(argv[++tmpint], 4, "-vr");

    else usage(argv[0]);
  }

  /* Append options to the packet */
  if (option_file != NULL) {
    options = get_options(option_file, &opt_len);
    if ((datagram = (void *) realloc(datagram, IP_DEFAULT_HEADER_SIZE +
				     opt_len)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + IP_DEFAULT_HEADER_SIZE, options, opt_len);
    if (!(flag & GOT_IHL))
      packet->ip_hl = packet->ip_hl + (opt_len / 4);
    packet = (struct ip *) datagram;
  }

  /* Append data to the packet */
  if (data != NULL) {
    if ((datagram = (void *) realloc(datagram, IP_DEFAULT_HEADER_SIZE +
				     opt_len + data_len)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + IP_DEFAULT_HEADER_SIZE + opt_len, data, data_len);
    packet = (struct ip *) datagram;
  }

  /* Set the packet length */
  if (!(flag & GOT_TL))
    packet->ip_len = (IP_DEFAULT_HEADER_SIZE + opt_len + data_len);

  /* Convert fields to network byte order */
  packet->ip_len = htons(packet->ip_len);
  packet->ip_id = htons(packet->ip_id);
  packet->ip_off = htons(packet->ip_off);

  /* Generate Checksum */
  if (!(flag & GOT_CKSUM))
    packet->ip_sum = checksum(datagram, IP_DEFAULT_HEADER_SIZE + opt_len);

  /* Print packet data */
  if (flag & VERBOSE)
    print_ip_data(packet);

  /* Write the packet to a file */
  write_packet(output_file, datagram,
	       IP_DEFAULT_HEADER_SIZE + opt_len + data_len);
}
