/* File   : makeeth.c
 * Author : J Hadi Salim <ad828@freenet.carleton.ca>
 * Purpose: Create an ethernet frame.
 *
 * Notes  : This file has been modified by Karyl F. Stein <xenon@xenos.net>.
 *          Any bugs are my fault.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>

#ifdef SunOS
#include <sys/types.h>   /* u_short              */
#include <sys/socket.h>  /* struct sockaddr      */
#endif  /* SunOS */

#ifdef BSDI31
#include <sys/types.h>   /* u_short              */
#include <sys/socket.h>  /* struct sockaddr      */
#endif  /* BSDI31 */

#include <netinet/in.h>  /* Protocol numbers     */
#include <net/if.h>      /* ifreq struct         */
#include <stdlib.h>      /* realloc()            */
#include "ethernet.h"    /* Ethernet definitions */
#include "datadisp.h"    /* Function prototypes  */
#include "spak_func.h"   /* Function prototypes  */
#include "ether_func.h"  /* Function prototypes  */

/* Values used by the flag variable */
#define GOT_SMAC 0x01
#define VERBOSE  0x02
#define VVERBOSE 0x04

/* Set defaults if not defined previously or not valid.  Change these */
/* values in makeeth_conf.h and not here.                             */
#ifndef DEF_ETH_TYPE
# define DEF_ETH_TYPE 0x0800
#endif  /* DEF_ETH_TYPE */
#ifndef DEF_DST_MAC
# define DEF_DST_MAC "00:00:00:00:00:00"
#endif  /* DEF_DST_MAC */


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit.
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s [-s <src_mac>] [-d <dest_mac>] [-t <ether_type>]\n", name);
  fprintf(stderr, "       [<-v>] [-i <input_file>] [-o <out_file>]\n");
  fprintf(stderr, " -d  Set the destination MAC address of the frame to <dest_mac>.  The\n");
  fprintf(stderr, "     <dest_mac> argument takes the form a:b:c:d:e:f where a through f are\n");
  fprintf(stderr, "     two digit hexadecimal numbers.  Default: %s\n", DEF_DST_MAC);
  fprintf(stderr, " -i  Read the frame data from the file <input_file>.  If <input_file> is -,\n");
  fprintf(stderr, "     then stdin is used.\n");
  fprintf(stderr, " -o  Write the frame data to <out_file>.  Default: stdout\n");
  fprintf(stderr, " -s  Set the source MAC address of the frame to <src_mac>.  The <src_mac>\n");
  fprintf(stderr, "     argument takes the form a:b:c:d:e:f where a through f are two digit\n");
  fprintf(stderr, "     hexadecimal numbers.  If <src_mac> is -, then the source MAC address\n");
  fprintf(stderr, "     is guessed by the program.  Default: ");
#ifdef DEF_SRC_MAC
  fprintf(stderr, "%s\n", DEF_SRC_MAC);
#else
  fprintf(stderr, "Guessed by the program\n");
#endif  /* DEF_SRC_MAC */
  fprintf(stderr, " -t  Set the protocol of the frame data to <ether_type>.  The <ether_type>\n");
  fprintf(stderr, "     argument should be a hexidecimal number.  Default: 0x%x\n", DEF_ETH_TYPE);
  fprintf(stderr, " -v  Turn on verbose mode, (print frame header data).\n");
  fprintf(stderr, " -vv Turn on very verbose mode, (print all packet data).\n");
  exit(0);
}


/* Function: new_ethernet_frame
 * Return  : A new ethernet frame.
 */
my_ethhdr *new_ethernet_frame (void) {
  my_ethhdr *retval;

  if ((retval = (my_ethhdr *) malloc(sizeof(my_ethhdr))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }

  return(retval);
}


int main (int argc, char *argv[]) {
  char *data = NULL, *output_file = NULL, *ptr;
  int data_len = 0, flag = 0, tmpint;
  my_ethhdr *frame;
  struct ifreq ifr;
  void *datagram;

  datagram = (void *) new_ethernet_frame();
  frame = (my_ethhdr *) datagram;

  /* Set the default ethernet parameters */
  if (hwaddread((char *) &frame->ETH_FIELD_DEST, DEF_DST_MAC) == -1)
    hwaddread((char *) &frame->ETH_FIELD_DEST, "00:00:00:00:00:00");
  frame->ETH_FIELD_TYPE = htons(DEF_ETH_TYPE);

  /* Parse the arguments */
  tmpint = 0;
  while (++tmpint < argc) {
    if (*(ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Set destination MAC address */
    if (strcmp(ptr, "d") == 0) {
      if (argv[++tmpint] == NULL)
	fprintf(stderr, "Warning: The -d argument expects a MAC address\n");
      else {
	if (hwaddread((char *) &frame->ETH_FIELD_DEST, argv[tmpint]) < 0) {
	  fprintf(stderr, "Invalid destination MAC address (%s)\n",
		  argv[tmpint]);
	  exit(1);
	}
      }
    }

    /* Input Data */
    else if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
	fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Output File */
    else if (strcmp(ptr, "o") == 0) {
      if (argv[++tmpint] == NULL)
	fprintf(stderr, "Warning: The -o argument expects a file name\n");
      else {
	if (output_file != NULL) {
	  fprintf(stderr, "Warning: More than one output file specified.  Using %s.", argv[tmpint]);
	  free(output_file);
	}
	output_file = argv[tmpint];
      }
    }

    /* Set the source MAC address */
    else if (strcmp(ptr, "s") == 0) {
      if (argv[++tmpint] == NULL) {
	fprintf(stderr, "%s Warning: The -s argument expects a MAC address\n",
		argv[0]);
	continue;
      }

      /*********** THIS needs to be fixed *********************
      if (strcmp(argv[tmpint], "-") == 0) {
	get_default(&ifr, MAC);
	if (hwaddread(frame->ETH_FIELD_SRC, ifr.IFR_FIELD_ETH) == -1) {
	  fprintf(stderr, "Unable to guess source MAC address\n");
	  exit(1);
	}
      } else
      ********************************************************/

      if (hwaddread((char *) &frame->ETH_FIELD_SRC, argv[tmpint]) == -1) {
	fprintf(stderr,"Invalid source MAC address (%s)\n",  argv[tmpint]);
	exit(1);
      }
      flag |= GOT_SMAC;
    }

    /* Set the data type */
    else if (strcmp(ptr, "t") == 0) {
      if (argv[++tmpint] == NULL)
	fprintf(stderr, "%s Warning: The -t argument expects an hexidecimal number\n", argv[0]);
      else {
        sscanf(argv[tmpint], "%x", &frame->ETH_FIELD_TYPE);
        frame->ETH_FIELD_TYPE = htons(frame->ETH_FIELD_TYPE);
      }
    }

    /* Set verbose flags */
    else if (*ptr == 'v') {
      flag |= VERBOSE;
      if (strcmp(ptr, "vv") == 0)
        flag |= VVERBOSE;
    }

    else usage(argv[0]);
  }

  /* Append data to the packet */
  if (data != NULL) {
    if ((datagram = (void *) realloc(datagram, ETH_HLEN + data_len)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + ETH_HLEN, data, data_len);
    frame = (my_ethhdr *) datagram;
  }

  /* Set the source MAC address */
  if (!flag & GOT_SMAC) {
#ifdef DEF_SRC_MAC
    hwaddread(frame->ETH_FIELD_SRC, DEF_SRC_MAC);
#else
    get_default((struct ifreq *) &frame->ETH_FIELD_SRC, MAC);
#endif  /* DEF_SRC_MAC */
  }

  /* Check the validity of the data */
#ifdef LINUX
  if (data_len > ETH_DATA_LEN)
#endif  /* LINUX */
#ifdef SunOS
  if (data_len > ETHERMTU)
#endif  /* SunOS */
    fprintf(stderr, "Warning: Packet data length greater than maximum transmit unit allowed\n");
#ifdef LINUX
  if ((data_len + ETH_HLEN) < ETH_ZLEN)
    fprintf(stderr,"Warning: Frame possibly under-sized by %d bytes\n",
	    ETH_ZLEN - (ETH_HLEN + data_len));
#endif  /* LINUX */
#ifdef SunOS
  if (data_len < ETHERMIN)
    fprintf(stderr,"Warning: Frame possibly under-sized by %d bytes\n",
	    ETHERMIN - data_len);
#endif  /* SunOS */

  /* Print the packet data */
  if (flag & VERBOSE) {
    print_eth_data(frame);
    if (flag & VVERBOSE) {
      tmpint = print_ip_data((struct ip *) data);
      if (ip_protocol == IPPROTO_TCP)
	print_tcp_data((struct tcphdr *) (data + tmpint));
      else if (ip_protocol == IPPROTO_UDP)
	print_udp_data((struct udphdr *) (data + tmpint));
      else
	fprintf (stderr, "Warning: Unknown IP protocol number %d\n",
		 ip_protocol);
    }
  }

  /* Write the frame data to a file or stdout */
  write_packet(output_file, datagram, data_len + sizeof(my_ethhdr));
}
