/* File   : datadisp.c
 * Author : J Hadi Salim <ad828@freenet.carleton.ca>
 * Purpose: Functions for displaying data generated by various programs
 *          included with Spak.  The individual functions are detailed below.
 *
 * Notes  : This file has been modified by Karyl F. Stein <xenon@xenos.net>.
 *          Any bugs are my fault.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */     

#include "config.h"
#include "makeudp_sysconf.h"
#include <stdio.h>
#include "ethernet.h"    /* Ethernet definitions */
#include <netinet/in.h>  /* ntohs()              */
#include "datadisp.h"    /* Function prototypes  */

/* Values used to set various bits for IP.  Do not change. */
#define BIT_TOS7 0x01
#define BIT_TOS6 0x02
#define BIT_R    0x04
#define BIT_T    0x08
#define BIT_D    0x10
#define BIT_F1   0x8000

/* Values used to set various bits for TCP .  Do not change. */
#define BIT_R1 0x01
#define BIT_R2 0x02
#define BIT_R3 0x04
#define BIT_R4 0x08
#define BIT_R5 0x40
#define BIT_R6 0x80


/* Function: hwaddrprint
 * Input   :
 * Output  :
 */
void hwaddrprint (unsigned char *addr) {
  fprintf(stderr, "%x:%x:%x:%x:%x:%x",
	  addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
}


/* Function: print_arp_data
 * Input   :
 * Output  :
 */
void print_arp_data (struct arphdr *ahdr) {
  char *ptr = (char *) ahdr;
  int offset;

  fprintf(stderr, "\n============ ARP Data ============\n");
  fprintf(stderr, "Hardware Type          : %d\tProtocol Type          : 0x%x\n", ntohs(ahdr->ar_hrd), ntohs(ahdr->ar_pro));
  fprintf(stderr, "Hardware Address Length: %d\tProtocol Address Length: %d\n", ahdr->ar_hln, ahdr->ar_pln);
  fprintf(stderr, "OP Code                : %d\n\n", ntohs(ahdr->ar_op));

  ptr += 8;
  switch (ntohs(ahdr->ar_hrd)) {
  case 1:  /* Ethernet 10Mbps */
    fprintf(stderr, "Sender Hardware Address: ");
    hwaddrprint(ptr);
    ptr += 6;
    break;
  }
  switch (ntohs(ahdr->ar_pro)) {
  case 0x800:
    fprintf(stderr, "\nSender Protocol Address: ");
    print_ip(ptr);
    ptr += 4;
    break;
  }
  switch (ntohs(ahdr->ar_hrd)) {
  case 1:  /* Ethernet 10Mbps */
    fprintf(stderr, "\nTarget Hardware Address: ");
    hwaddrprint(ptr);
    ptr += 6;
    break;
  }
  switch (ntohs(ahdr->ar_pro)) {
  case 0x800:
    fprintf(stderr, "\nTarget Protocol Address: ");
    print_ip(ptr);
    ptr += 4;
    break;
  }
  fprintf(stderr, "\n");
}


/* Function: print_eth_data
 * Input   : An ethernet frame.
 * Output  : The header data is printed to the screen.
 */
void print_eth_data (my_ethhdr *frame) {
  fprintf(stderr, "\n============ Ethernet Header Data ============\n");
  fprintf(stderr, "Source MAC Address     : ");
  hwaddrprint((char *) &frame->ETH_FIELD_SRC);
  fprintf(stderr, "\nDestination MAC Address: ");
  hwaddrprint((char *) &frame->ETH_FIELD_DEST);
  fprintf(stderr, "\nProtocol               : 0x%x\n", frame->ETH_FIELD_TYPE);
}


/* Function: print_full_tcp_data
 * Input   :
 * Output  :
 */
void print_full_tcp_data (struct full_tcphdr *packet) {
  fprintf(stderr, "\n============ TCP Pseudo-Header Data ============\n");
  fprintf(stderr, "Source Address     : %s\n", inet_ntoa(packet->source));
  fprintf(stderr, "Destination Address: %s\n", inet_ntoa(packet->destination));
  fprintf(stderr, "Zero Pad: %d\tProtocol: %d\tTCP Length: %d bytes\n",
	  packet->zero, packet->protocol, ntohs(packet->length));
  print_tcp_data(&packet->packet);
}


/* Function: print_full_udp_data
 * Input   :
 * Output  :
 */
void print_full_udp_data (struct full_udphdr *packet) {
  fprintf(stderr, "\n============ UDP Pseudo-Header Data ============\n");
  fprintf(stderr, "Source Address     : %s\n", inet_ntoa(packet->source));
  fprintf(stderr, "Destination Address: %s\n", inet_ntoa(packet->destination));
  fprintf(stderr, "Zero Pad: %d\tProtocol: %d\tPacket Length: %d bytes\n",
	  ntohs(packet->length), packet->zero, packet->protocol);
  print_udp_data(&packet->packet);
}


/* Function: print_ip
 * Input   :
 * Output  :
 *
 * Notes: Only works for arp header ip addresses.
 */
void print_ip(unsigned char *addr)  {
  fprintf(stderr,"%d.%d.%d.%d", addr[0], addr[1], addr[2], addr[3]);
}


/* Function: print_ip_data
 * Input   :
 * Output  :
 * Return  :
 */
int print_ip_data (struct ip *packet) {
  extern int ip_protocol;

  fprintf(stderr, "\n============ IP Data ============\n");
  fprintf(stderr, "Source Address     : %s\n", inet_ntoa(packet->ip_src));
  fprintf(stderr, "Destination Address: %s\n", inet_ntoa(packet->ip_dst));
  fprintf(stderr, "IP Version         : %-6d\tHeader Length (32bit words): %-4d\n",packet->ip_v, packet->ip_hl);
  fprintf(stderr, "Type of Service    : %-6d\tTotal Length               : %-4d\n", (packet->ip_tos >> 5) & 0x07, ntohs(packet->ip_len));
  fprintf(stderr, "Identification     : 0x%-4x\tFragment Offset            : %-4d\n", ntohs(packet->ip_id), packet->ip_off & 0x1FFF);
  fprintf(stderr, "Time to Live       : %-4d\tProtocol                   : %d\n", packet->ip_ttl, packet->ip_p);
  fprintf(stderr, "Header Checksum    : 0x%-4x\n", ntohs(packet->ip_sum));
  fprintf(stderr, "Bits Set           :%s%s%s%s%s%s%s%s\n", 
	  (packet->ip_off & IP_DF) ? " DF" : "",
	  (packet->ip_off & IP_MF) ? " MF" : "", 
	  (packet->ip_tos & BIT_D) ? " DELAY" : "", 
	  (packet->ip_tos & BIT_T) ? " THROUGHPUT" : "", 
	  (packet->ip_tos & BIT_R) ? " RELIBILITY" : "", 
	  (packet->ip_tos & BIT_TOS6) ? " TOS_BIT_6" : "", 
	  (packet->ip_tos & BIT_TOS7) ? " TOS_BIT_7" : "", 
	  (packet->ip_off & BIT_F1) ? " FLAG_BIT_1" : "");

  /* Set the global protocol type */
  ip_protocol = packet->ip_p;
  return (packet->ip_hl*4);
}


/* Function: print_tcp_data
 * Input   :
 * Output  :
 * Return  :
 */
void print_tcp_data (struct tcphdr *packet) {
  fprintf(stderr, "\n============ TCP Data ============\n");
  fprintf(stderr, "Source Port     : %-10d\t", ntohs(packet->th_sport));
  fprintf(stderr, "Destination Port      : %d\n", ntohs(packet->th_dport));
  fprintf(stderr, "Sequence Number : 0x%-8x\tAcknowledgement Number: 0x%-8x\n",
	  ntohl(packet->th_seq), ntohl(packet->th_ack));
  fprintf(stderr, "Data Offset     : %-10d\tUrgent Offset         : %-10d\n",
	  packet->th_off, ntohs(packet->th_urp));
  fprintf(stderr, "Window Size     : 0x%-8x\tChecksum              : 0x%-8x\n",
	  ntohs(packet->th_win), ntohs(packet->th_sum));
  fprintf(stderr, "Bits Set        :%s%s%s%s%s%s%s%s%s%s%s%s\n",
	  (packet->th_flags & TH_FIN) ? " FIN" : "",
	  (packet->th_flags & TH_SYN) ? " SYN" : "",
	  (packet->th_flags & TH_RST) ? " RST" : "",
	  (packet->th_flags & TH_PUSH) ? " PUSH" : "",
	  (packet->th_flags & TH_ACK) ? " ACK" : "",
	  (packet->th_flags & TH_URG) ? " URG" : "",
	  (packet->th_x2 & BIT_R1) ? " R1" : "",
	  (packet->th_x2 & BIT_R2) ? " R2" : "",
	  (packet->th_x2 & BIT_R3) ? " R3" : "",
	  (packet->th_x2 & BIT_R4) ? " R4" : "",
	  (packet->th_flags & BIT_R5) ? " R5" : "",
	  (packet->th_flags & BIT_R6) ? " R6" : "");
}


/* Function: print_udp_data
 * Input   :
 * Output  :
 * Return  :
 */
void print_udp_data (struct udphdr *packet) {
  fprintf(stderr, "\n============ UDP Data ============\n");
  fprintf(stderr, "Source Port     : %d\nDestination Port: %d\n",
          ntohs(packet->UDP_FIELD_SRC), ntohs(packet->UDP_FIELD_DEST));
  fprintf(stderr, "Message Length  : %d\tChecksum        : 0x%-4x\n",
          ntohs(packet->UDP_FIELD_LEN), ntohs(packet->UDP_FIELD_CSUM));
}
