/* File   : sendpacket.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Send a packet to a socket.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <stdio.h>

#ifdef SunOS
#include <sys/types.h>          /* u_short               */
#include <sys/socket.h>         /* AF_INET               */
#endif /* SunOS */

#include <fcntl.h>              /* O_RDONLY              */
#include <netinet/in.h>         /* Socket data structure */
#include <unistd.h>             /* read()                */
#include "spak_func.h"          /* Function prototypes   */
#include "datadisp.h"           /* Function prototypes   */

/* Values used by the flag variable */
#define VERBOSE   0x01


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s <dest> [-v] [-i <input_file>]\n", name);
  fprintf(stderr, " <dest>  Destination address of the packet, (in numerical or word form).\n");
  fprintf(stderr, " -i      Read the packet data from the file <input_file>.  If <input_file>\n");
  fprintf(stderr, "         is -, or if the -i argument is not given, stdin is used.\n");
  fprintf(stderr, " -v      Turn on verbose mode, (print packet data).\n");
  exit(0);
}


int main (int argc, char *argv[]) {
  char *data = NULL, *ptr;
  int data_len = 0, flag = 0, send_stat, sfd, tmpint;
  struct in_addr dest;
  struct sockaddr_in sock_data;

  /* Check to see that the correct number of arguments were given */
  if (argc <= 1)
    usage(argv[0]);

  /* Set the destination addresses */
#ifdef INET_ATON
  if (inet_aton(argv[1], &dest) == 0)
#else
  if ((dest.s_addr = inet_addr(argv[1])) == -1)
#endif /* INET_ATON */
    dest.s_addr = getaddrbyname(argv[1]);

  /* Parse the arguments */
  tmpint = 1;
  while (++tmpint < argc) {
    if (*(ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Input Data */
    if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Set verbose flag */
    else if (strcmp(ptr, "v") == 0)
      flag |= VERBOSE;

    else usage(argv[0]);
  }

  /* Read data from stdin if no data read so far */
  if ((data == NULL) &&
      (((data = read_data("-", &data_len)) == NULL) || (data_len == 0))) {
    fprintf(stderr, "%s: No data read.\n", argv[0]);
    exit(1);
  }

  /* Print packet data */
  if (flag & VERBOSE) {
    tmpint = print_ip_data((struct ip *) data);
    if (ip_protocol == IPPROTO_TCP)
      print_tcp_data((struct tcphdr *) (data + tmpint));
    else if (ip_protocol == IPPROTO_UDP)
      print_udp_data((struct udphdr *) (data + tmpint));
    else fprintf(stderr, "\n%s: Unknown IP protocol number (%d)\n",
		 argv[0], ip_protocol);
  }

  /* Open a socket */
  if ((sfd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) == -1) {
    fprintf(stderr, "Unable to open socket\n");
    exit(1);
  }

  bzero((char *) &sock_data, sizeof(struct sockaddr));
  sock_data.sin_family = AF_INET;
  sock_data.sin_addr.s_addr = dest.s_addr;

  /* Send the packet */
  send_stat = sendto(sfd, data, data_len, 0, (struct sockaddr *) &sock_data,
		     sizeof(struct sockaddr));
  if (send_stat == -1) {
    fprintf(stderr, "%s: Unable to Send Packet\n", argv[0]);
    exit(1);
  }
  if (send_stat != data_len) {
    fprintf(stderr, "%s Warning: Incomplete packet sent\n", argv[0]);
    exit(1);
  }
}
