/* File   : maketcp.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Create a TCP packet
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "maketcp_conf.h"
#include <stdio.h>

#ifdef SunOS
#include <sys/types.h>        /* u_char      */
#include <netinet/in.h>       /* sockaddr_in */
#include <arpa/nameser.h>     /* MAXDNAME    */
#include <resolv.h>           /* in_addr     */
#include <netinet/in_systm.h> /* n_long      */
#endif /* SunOS */

#include <unistd.h>           /* STDIN_FILENO         */
#include <netinet/tcp.h>      /* TCP header struct    */
#include <netinet/in.h>       /* in_addr struct       */
#include "spak_func.h"        /* Function prototypes  */
#include "maketcp_func.h"     /* Function prototypes  */
#include "datadisp.h"         /* Function prototypes  */

/* Values used by the flag variable */
#define GOT_CKSUM 0x01
#define GOT_OFF   0x02
#define GOT_PLEN  0x04
#define VERBOSE   0x08

/* Values used to set various bits */
#define BIT_R1 0x01
#define BIT_R2 0x02
#define BIT_R3 0x04
#define BIT_R4 0x08
#define BIT_R5 0x40
#define BIT_R6 0x80

/* Set defaults if not defined previously or not valid.  Change these in */
/* maketcp_conf.h and not here.                                          */
#ifndef TCP_DEFAULT_HEADER_SIZE
# define TCP_DEFAULT_HEADER_SIZE 20
#endif  /* TCP_DEFAULT_HEADER_SIZE */
#ifndef TCP_PSEUDO_HEADER_SIZE
#define TCP_PSEUDO_HEADER_SIZE 12
# endif  /* TCP_PSEUDO_HEADER_SIZE */
#ifndef TCP_DEFAULT_WINDOW_SIZE
# define TCP_DEFAULT_WINDOW_SIZE 512
#endif  /* TCP_DEFAULT_WINDOW_SIZE */
#ifndef TCP_DEFAULT_PROTOCOL
# define TCP_DEFAULT_PROTOCOL 6
#endif  /* TCP_DEFAULT_PROTOCOL */


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s <src> <s_port> <dest> <d_port>\n", name);
  fprintf(stderr, "       [-h] [-s <seq_num>] [-a <ack_num>] [-do <offset>] [-uo <offset>]\n");
  fprintf(stderr, "       [-w <win_size>] [-c <hdr_cksum>] [-su] [-sa] [-sp] [-sr] [-ss] [-sf]\n");
  fprintf(stderr, "       [-i <input_file>] [-o <output_file] [-of <opt_file>] [-v]\n");
  fprintf(stderr, "       [-pd <dest>] [-pl <length>] [-pp <protocol>] [-ps <src>]\n");
  fprintf(stderr, " <src>     Source IP for the packet to have, (in numerical or word form).\n");
  fprintf(stderr, " <s_port>  Source port for the packet to have, (in numerical or word form).\n");
  fprintf(stderr, " <dest>    Destination IP for the packet to have, (in numerical or word form).\n");
  fprintf(stderr, " <d_port>  Source port for the packet to have, (in numerical or word form).\n");
  fprintf(stderr, " -a        Set the acknowledge number to <ack_num>.  Default: Random\n");
  fprintf(stderr, " -do       Set the data offset to <offset>.  Default: Correct value\n");
  fprintf(stderr, " -h        Print this help message.\n");
  fprintf(stderr, " -c        Set the header checksum to <hdr_cksum>.  Default: Correct value\n");
  fprintf(stderr, " -i        Read the packet data from the file <input_file>.  If <input_file>\n");
  fprintf(stderr, "           is -, stdin will be used for input.\n");
  fprintf(stderr, " -o        Send the packet data to the file <output_file>.  Default: stdout\n");
  fprintf(stderr, " -pd       Set the destination address of the packet as stored in the pseudo-\n");
  fprintf(stderr, "           header to <dest>, (argument may be given in numerical or word\n");
  fprintf(stderr, "           form).  Default: Correct value\n");
  fprintf(stderr, " -pl       Set the length of the header and data as stored in the pseudo-\n");
  fprintf(stderr, "           header to <length>.  Default: Correct value\n");
  fprintf(stderr, " -pp       Set the protocol as stored in the pseudo-header to <protocol>.\n");
  fprintf(stderr, "           Default: %d\n", TCP_DEFAULT_PROTOCOL);
  fprintf(stderr, " -ps       Set the source address of the packet as stored in the pseudo-\n");
  fprintf(stderr, "           to <src>, (argument may be given in numerical or word form).\n");
  fprintf(stderr, "           Default: Correct value\n");
  fprintf(stderr, " -pz       Set the value of the 8-bit 0 padding in the pseudo-header to\n");
  fprintf(stderr, "           <value>.  Defalt: 0\n");
  fprintf(stderr, " -r1 - -r6 Set reserved bits, (-r5 sets bit 5).  Default: not set\n");
  fprintf(stderr, " -s        Set the sequence number to <seq_num>.  Default: Random\n");
  fprintf(stderr, " -sa       Set the ACK bit.  Default: Not set\n");
  fprintf(stderr, " -sf       Set the FIN bit.  Default: Not set\n");
  fprintf(stderr, " -sp       Set the PSH bit.  Default: Not set\n");
  fprintf(stderr, " -sr       Set the RST bit.  Default: Not set\n");
  fprintf(stderr, " -ss       Set the SYN bit.  Default: Not set\n");
  fprintf(stderr, " -su       Set the URG bit.  Default: Not set\n");
  fprintf(stderr, " -uo       Set the urgent offset to <offset>.  Default: 0\n");
  fprintf(stderr, " -v        Turn on verbose mode, (print packet data).\n");
  fprintf(stderr, " -w        Set the window size to <win_size>.  Default: %d\n", TCP_DEFAULT_WINDOW_SIZE);
  exit(0);
}


int main (int argc, char *argv[]) {
  char tmpchar;
  char *data = NULL, *input_file = NULL, *options = NULL, *output_file = NULL;
  char *option_file = NULL;
  FILE *ifp, *ofp;
  int data_len = 0, flag = 0, opt_len = 0, place = 0, tmpint;
  struct in_addr destination, source;
  struct tcphdr *packet;
  void *datagram, *ptr;

  /* Make sure the correct number of arguments are given */
  if (argc <= 4)
    usage(argv[0]);

  /* Create a new packet */
  datagram = (void *) new_tcp_packet();
  packet = & ((struct full_tcphdr *) datagram)->packet;

  /* Set source and destination addresses and ports */
#ifdef INET_ATON
  if (inet_aton(argv[1], &source) == 0)
#else
  if ((source.s_addr = inet_addr(argv[1])) == -1)
#endif
    source.s_addr = getaddrbyname(argv[1]);
#ifdef INET_ATON
  if (inet_aton(argv[3], &destination) == 0)
#else
  if ((destination.s_addr = inet_addr(argv[3])) == -1)
#endif
    destination.s_addr = getaddrbyname(argv[3]);

  ((struct full_tcphdr *) datagram)->source = source.s_addr;
  ((struct full_tcphdr *) datagram)->destination = destination.s_addr;
  packet->th_sport = my_atoi(argv[2], 16, "source port");
  packet->th_dport = my_atoi(argv[4], 16, "destination port");

  /* Parse the arguments */
  tmpint = 4;
  while (++tmpint < argc) {
    if (*((char *) ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Acknowledgement Number */
    if (strcmp(ptr, "a") == 0)
      packet->th_ack = my_atoi(argv[++tmpint], 32, "-a");

    /* Header Checksum */
    else if (strcmp(ptr, "c") == 0) {
      packet->th_sum = my_atoi(argv[++tmpint], 16, "-c");
      flag |= GOT_CKSUM;
    }

    /* Data Offset */
    else if (strcmp(ptr, "do") == 0) {
      packet->th_off = my_atoi(argv[++tmpint], 4, "-do");
      flag |= GOT_OFF;
    }

    /* Input File */
    else if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Output File */
    else if (strcmp(ptr, "o") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -o argument expects a file name\n");
      else {
        if (output_file != NULL)
          free(output_file);
	output_file = argv[tmpint];
      }
    }

    /* Option File */
    else if (strcmp(ptr, "of") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -of argument expects a file name\n");
      else {
	if (option_file != NULL)
	  free(option_file);
	option_file = argv[tmpint];
      }
    }

    /* Destination Address (for pseudo-header) */
    else if (strcmp(ptr, "pd") == 0) {
#ifdef INET_ATON
      if (inet_aton(argv[++tmpint], &destination) == 0)
#else
      if ((destination.s_addr = inet_addr(argv[++tmpint])) == -1)
#endif  /* INET_ATON */
	destination.s_addr = getaddrbyname(argv[tmpint]);
      ((struct full_tcphdr *) datagram)->destination = destination.s_addr;
    }

    /* Packet Length (for pseudo-header) */
    else if (strcmp(ptr, "pl") == 0)
      ((struct full_tcphdr *) datagram)->length =
	my_atoi(argv[++tmpint], 16, "-pl");

    /* Protocol (for pseudo-header) */
    else if (strcmp(ptr, "pp") == 0)
      ((struct full_tcphdr *) datagram)->protocol =
	my_atoi(argv[++tmpint], 8, "-pp");

    /* Source Address (for pseudo-header) */
    else if (strcmp(ptr, "ps") == 0) {
#ifdef INET_ATON
      if (inet_aton(argv[++tmpint], &source) == 0)
#else
      if ((source.s_addr = inet_addr(argv[++tmpint])) == -1)
#endif  /* INET_ATON */
	source.s_addr = getaddrbyname(argv[tmpint]);
      ((struct full_tcphdr *) datagram)->source = source.s_addr;
    }

    /* Zero pad (for pseudo-header--illegal if not 0) */
    else if (strcmp(ptr, "pz") == 0)
      ((struct full_tcphdr *) datagram)->zero =
        my_atoi(argv[++tmpint], 8, "-pz");

    /* Reserved Bits (illegal) */
    else if (* (char *) ptr == 'r') {
      ptr++;
      if (strcmp(ptr, "1") == 0)
	packet->th_x2 |= BIT_R1;
      else if (strcmp(ptr, "2") == 0)
	packet->th_x2 |= BIT_R2;
      else if (strcmp(ptr, "3") == 0)
	packet->th_x2 |= BIT_R3;
      else if (strcmp(ptr, "4") == 0)
	packet->th_x2 |= BIT_R4;

      else if (strcmp(ptr, "5") == 0)
	packet->th_flags |= BIT_R5;
      else if (strcmp(ptr, "6") == 0)
	packet->th_flags |= BIT_R6;

      else usage(argv[0]);
    }

    /* Sequence Number */
    else if (strcmp(ptr, "s") == 0)
      packet->th_seq = my_atoi(argv[++tmpint], 32, "-s");

    /* Set various bits */
    else if (* (char *) ptr == 's') {
      ptr++;

      /* Set acknowledge bit */
      if (strcmp(ptr, "a") == 0)
	packet->th_flags |= TH_ACK;

      /* Set finish bit */
      else if (strcmp(ptr, "f") == 0)
	packet->th_flags |= TH_FIN;

      /* Set push bit */
      else if (strcmp(ptr, "p") == 0)
	packet->th_flags |= TH_PUSH;

      /* Set reset bit */
      else if (strcmp(ptr, "r") == 0)
	packet->th_flags |= TH_URG;

      /* Set syncronize bit */
      else if (strcmp(ptr, "s") == 0)
	packet->th_flags |= TH_SYN;

      /* Set urgent bit */
      else if (strcmp(ptr, "u") == 0)
	packet->th_flags |= TH_URG;

      else usage(argv[0]);
    }

    /* Urgent Offset */
    else if (strcmp(ptr, "u") == 0)
      packet->th_urp = my_atoi(argv[++tmpint], 16, "-u");

    /* Set verbose mode */
    else if (strcmp(ptr, "v") == 0)
      flag |= VERBOSE;

    /* Window Size */
    else if (strcmp(ptr, "w") == 0)
      packet->th_win = my_atoi(argv[++tmpint], 16, "-w");

    else usage(argv[0]);
  }

  /* Convert header fields to network byte order */
  packet->th_sport = htons(packet->th_sport);
  packet->th_dport = htons(packet->th_dport);
  packet->th_seq = htonl(packet->th_seq);
  packet->th_ack = htonl(packet->th_ack);
  packet->th_win = htons(packet->th_win);
  packet->th_urp = htons(packet->th_urp);

  /* Append options */
  if (option_file != NULL) {
    options = get_options(option_file, &opt_len);
    if ((datagram = (void *) realloc(datagram, TCP_DEFAULT_HEADER_SIZE +
				     TCP_PSEUDO_HEADER_SIZE +
				     opt_len)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + TCP_DEFAULT_HEADER_SIZE + TCP_PSEUDO_HEADER_SIZE,
	   options, opt_len);
    if (!(flag & GOT_OFF))
      packet->th_off += opt_len / 4;
    packet = & ((struct full_tcphdr *) datagram)->packet;
  }

  /* Append data */
  if (data != NULL) {
    if ((datagram = (void *) realloc(datagram, TCP_DEFAULT_HEADER_SIZE +
				     opt_len + data_len +
				     TCP_PSEUDO_HEADER_SIZE)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + TCP_DEFAULT_HEADER_SIZE + opt_len +
	   TCP_PSEUDO_HEADER_SIZE, data, data_len);
    packet = & ((struct full_tcphdr *) datagram)->packet;
  }

  /* Set packet length */
  if (!(flag & GOT_PLEN))
    ((struct full_tcphdr *) datagram)->length =
      htons(TCP_DEFAULT_HEADER_SIZE + opt_len + data_len);

  /* Generate Checksum */
  if (!(flag & GOT_CKSUM))
    packet->th_sum = checksum(datagram, TCP_DEFAULT_HEADER_SIZE +
			      opt_len + data_len + TCP_PSEUDO_HEADER_SIZE);

  /* Print packet information */
  if (flag & VERBOSE)
    print_full_tcp_data((struct full_tcphdr *) datagram);

  /* Write the packet data to a file or stdout */
  write_packet(output_file, datagram + TCP_PSEUDO_HEADER_SIZE,
	       TCP_DEFAULT_HEADER_SIZE + opt_len + data_len);
}
