/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/*
 * deal with non-expression statements (control constructs like
 * for, if, and while).
 */

#include "casl.h"
#include <setjmp.h>

extern int Debug;

#define NO_LOOP         0
#define IN_LOOP         1
#define BREAK_LOOP      2
#define CONTINUE_LOOP   3 

int Looping = NO_LOOP;

extern int casl_line;
extern char *casl_file;

/* ---------------------------------------------------------------------------
** interpret a "statement", which is a confusing word, since  
** a "statement" node is usually actually a list node containing 
** the individual statements of a brace-enclosed statement block.
**
** statements are the executive portion of the CASL language -
** everything that actually "does" something in a CASL script
** is a statement. 
**
** statements do not have return values (that's why they're statements,
** not expressions.) The only situation in which we return a value is
** if we're evaluating a "return" statement, in which case we pass the
** return value back down. 
*/

asr_t *eval_statement(asr_t *node) {
	static int Dump = 0;

	asr_t *rv = NULL;

#ifdef TRACE_CALLS
	Dprintf("eval_statement(%p)\n", node);
#endif

	/* if the dump flag got set, print this node out */

	if(Dump) {
		Dump--;
		asr_print(node);
	}

	if(!node)

		/* XXX this shouldn't ever happen, but at least 
		 * two places in this code call eval_statement 
		 * assuming it will handle nonexistant statements
		 * transparently, to avoid an extra check.
		 */

		return(NULL);

	casl_line = node->asr_line;

	if(node->asr_file)
		casl_file = node->asr_file;

	switch(node->asr_type) {	       
	case N_DEBUG:

		/* jump into the debugger. Right now this is GDB. */

		debugger();
		break;

	case N_TRON:
		Debug = 1;
		break;

	case N_TROFF:
		Debug = 0;
		break;

	case N_DUMP:
		
		/* print the next ASR tree section we parse */

		Dump++;
		break;

	case N_BREAK:
	case N_CONTINUE:
		if(Looping == NO_LOOP)
			error(E_USER, "attempt to alter loop control flow when not in loop");

		Looping = node->asr_type == N_BREAK ? BREAK_LOOP : CONTINUE_LOOP;
		break;

	case N_EXPR:

		/* evaluate an expression in a statement context. This
		 * is almost always going to be an assignment or a
		 * function call. Note that I don't take the return 
		 * value of a conditional here; statement-context 
		 * expressions need to discard the return value (which
		 * is meaningless anyways here) or else the bad 
		 * return() code will think the expression contained
		 * an embedded return() statement. Weee.
		 */

		rv = eval_expr(node->asr_kids[0]);
		if(rv) {
			alloc_downref (rv);
			rv = NULL;
		}

		break;

	case N_WHILE:

		/* a while() loop */

		rv = eval_while(node);
		break;	

	case N_FOR:

		/* a for(;;) loop */

		rv = eval_for(node);
		break;

	case N_ITE:

		/* an if-then-else conditional */

		rv = eval_ite(node);
		break;

	case N_FOREACH:

		/* a foreach (list iteration) loop */

		rv = eval_foreach(node);
		break;

	case N_STRUCT:

		/* a structure definition */

		eval_struct(node);
		break;

	case N_RETURN:

		/* a return statement. 
		 *
		 * My handling of this is quite lame right now; notice
		 * that almost all the other statements return a value.
		 * each of these statements is catching the return 
		 * value obtained from the return() case here, and if 
		 * they get a nonnull value, the end of this function
		 * catches it.
		 *
		 * YECH.
		 */

		if(!Level) {
			/* if we're returning and in global scope, we're
			 * effectively exiting the program.
			 */

			if(node->asr_kids[0]) {
				asr_t *a = eval_expr(node->asr_kids[0]);
				int i;

				/* get the exit code */

				if(a->asr_type == N_CHAR)
					i = a->asr_character;
				else if(a->asr_type == N_INT || a->asr_type == N_NEGINT)
					i = a->asr_integer;
				else
					i = 0;

				exit(i);
			} else {
				exit(0);
			}
		} else  {
			/* this is the one case in all this code where a 
			 * statement evaluation has a return value.
			 */
			
			asr_t *rv = node->asr_kids[0] ? 
				    eval_expr(node->asr_kids[0]) :
				    asr_int(0, 0);

			/* return values are temporary globals */

//			alloc_upref(rv);

			/* this effectively terminates the evaluation
			 * of a function body in eval_func().
			 */

			return(rv);
		}
	    
	case LIST: {
		/* evaluate a list of statements - this is what a brace-enclosed
		 * block of statements is passed to us as. 
		 */

		asr_t *n;
		int s = Looping;

		for(n = node; n; n = n->asr_kids[1]) {
	
			/* "return" statement */

			if((rv = eval_statement(n->asr_kids[0])))
				return(rv);

			/* "break" or "continue" */

			if(s != Looping)
				return(NULL);

		}

	}


		break;

	default: 
		/* A function declaration.  Bad parser.  Bad bad parser. */

		return(NULL);
		/* assert(0); */
		break;
	}

	return(rv);
}

/* ---------------------------------------------------------------------------
** interpret a while loop. The statement block to execute each iteration is 
** the second child of our node; evaluate the expression in the first child
** and, while it's nonzero, execute the statement block.
*/

asr_t *eval_while(asr_t *node) {
	asr_t *bool, *statement;
	asr_t *rv = NULL;
	int oldloop = Looping;

#ifdef TRACE_CALLS     
	Dprintf("eval_while(%p)\n", node);
#endif

	bool = node->asr_kids[0];
	statement = node->asr_kids[1];

	if(!bool || !statement)
		return(NULL);

	for(;;) {
		/* Catch break and continue */

		Looping = IN_LOOP;

		if(!asr_getint(bool))
			break;

		/* execute the statement block */
	
		if((rv = eval_statement(statement)))
			break;

		if(Looping == BREAK_LOOP)
			break;
	}

	Looping = oldloop;

	return(rv);
}

/* ---------------------------------------------------------------------------
** if then else
*/

asr_t *eval_ite(asr_t *node) {
	asr_t *bool, *sthen, *selse, *rv = NULL;

#ifdef TRACE_CALLS
	Dprintf("eval_ite(%p)\n", node);
#endif

	bool = node->asr_kids[0];
	sthen = node->asr_kids[1];
	selse = node->asr_kids[2];

	if(!bool || !sthen)
		return(NULL);

	if(asr_getint(bool))
		rv = eval_statement(sthen);
	else if(selse) 
		rv = eval_statement(selse);

	return(rv);
}

/* ---------------------------------------------------------------------------
** for (;;)
*/

asr_t *eval_for(asr_t *node) {
	asr_t *bool = node->asr_kids[1];
	asr_t *rv = NULL;
	asr_t *thingy = node->asr_kids[2];

	int oldloop = Looping;

#ifdef TRACE_CALLS
	Dprintf("eval_for(%p)\n", node);
#endif

	if(node->asr_kids[0]) {
		rv = eval_expr(node->asr_kids[0]);
		if (rv) alloc_downref (rv);
		rv = NULL;
	}

	for(;;) {
		/* catch break and continue */

		Looping = IN_LOOP;

		if(bool && !asr_getint(bool)) 
			break;

		/* evaluate the statement block */

		if(node->asr_kids[3]) {
			rv = eval_statement(node->asr_kids[3]);
			if(rv) 
				break;
		}

		if(Looping == BREAK_LOOP)
			break;
		   
		/* evaluate the increment expressions */

		if(thingy) {
			rv = eval_expr(thingy);
			if (rv) 
				alloc_downref (rv);
			rv = NULL;
		}
	}

	Looping = oldloop;
	return(rv);
}

/* ---------------------------------------------------------------------------
** iterate through a list
*/

asr_t *eval_foreach(asr_t *node) {
	asr_t *name, *list, *statement, *n, *rv = 0;

	int oldloop = Looping;

#ifdef TRACE_CALLS
	Dprintf("eval_foreach(%p)\n", node);
#endif

	name = node->asr_kids[0];
	list = node->asr_kids[1];
	statement = node->asr_kids[2];	

	list = list_expand(list);

	if(st_get(name->asr_ident, Level)) {
		error(E_WARN, 
			"foreach instance variable \"%s\" replaces"
			" a global variable.\n", name->asr_ident);
	
		st_remove(name->asr_ident, Level);
	}

	/* catch breaks and continues */

	n = list;

	for(; n; n = n->asr_kids[1]) {
		asr_t *v = n->asr_kids[0];
		asr_t *x;

		Looping = IN_LOOP;

		if(!v)
			break;

		if(v->asr_type == N_ID) {
			x = st_get(v->asr_ident, Level);
       			v = x;
		}

		alloc_upref (v);
		st_insert(name->asr_ident, v, Level);
		
		if(statement) { 
			rv = eval_statement(statement);
		}

		st_remove(name->asr_ident, Level);
		alloc_downref (v);

		if(rv || Looping == BREAK_LOOP)
			break;
	}

	Looping = oldloop;
	return(rv);
}			

