%{

/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * Redistribution and use are governed by the terms detailed in the
 * license document ("LICENSE.TXT") included with this source.  If the
 * document is missing, it may be obtained from Network Associates, Inc.
 */

#define YYMAXDEPTH 100000

#include "casl.h"

void yyerror(char *);

asr_t *Top = NULL;

extern int casl_line;

/* This file contains the specification for the CASL language parser.
 * The purpose of the CASL parser is to turn a stream of input characters
 * into an abstract representation of a CASL script (the format for which 
 * is documented in "casl-asr.c") which can be easily interpreted by
 * the CASL interpreter.
 *
 * This YACC parser does two things, in addition to recognizing the 
 * CASL language and spouting errors: it builds the ASR trees for
 * the interpreter, using asr_*() functions, and it ensures that
 * function declarations are inserted into the symbol table before 
 * the ASR tree is evaluated (all other symbol table entries are done
 * dynamically, as the script is interpreted)
 *
 * The other portion of the CASL recognizer is the lexer, which is in
 * "casl-lex.l". All the lexer is doing is converting strings into
 * tokens, which can easily be dealt with in YACC. Nonterminals are
 * lowercase in this spec, terminals (tokens) are in uppercase.
 */

%}

%union {
	u_long int_c;
	char char_c;
	char *string_c;
	char *ident_c;
	struct _asr *asr_c;
}

/*
 * operators (low to high)
 *    assign				right
 *    or				left
 *    and				left
 *    logical or			left
 *    logical xor			left
 *    logical and			left
 *    compare eq,neq			left
 *    compare lt,le,gt,ge		left
 *    logical shift			left
 *    plus, minus			left
 *    multiply, divide, mod		left
 *    not, logical not, inc, dec	right (unary)
 *    increment, decrement		left (unary)
 */

%right T_NAME T_ASSIGN T_COMMA
%left T_OR
%left T_AND
%left T_BOR
%left T_BXOR
%left T_BAND
%left T_EQ T_NE
%left T_LT T_GT T_LE T_GE
%left T_BLSHIFT T_BRSHIFT
%left T_NEG T_PLUS T_MINUS 
%left T_MUL T_DIV T_MOD
%right T_COPY T_POP T_TAIL 
%right T_PUSH T_APPEND
%right T_UNARY T_NOT T_BNOT
%left T_INCR T_DECR
%left  T_DOT

%token T_C_PAREN T_O_PAREN T_C_BRACE T_O_BRACE T_SEMI T_PROC
%token T_WHILE T_IF T_FOR T_FOREACH T_RETURN T_ELSE T_DUMP
%token T_O_BRACKET T_C_BRACKET T_RANGE T_NEW T_DEBUG T_TRON T_TROFF
%token T_BIT T_BYTE T_WORD T_DWORD T_VARIABLE T_EXTRACT T_FROM
%token T_STRUCT T_COLON T_BREAK T_CONTINUE T_SCALE T_INCR T_DECR

%token <string_c> 	T_STRING_LIT 
%token <char_c> 	T_CHAR_LIT 
%token <int_c> 	T_INT_LIT 
%token <ident_c> 	T_ID 

%type <asr_c> program function statement id opt_function_args function_body
%type <asr_c> statements opt_expr expr opt_else structdef foreach size
%type <asr_c> range_spec opt_end_range struct_body field_decls field_decl
%type <asr_c> list call assign idextract opt_callargs opt_morelistargs
%type <asr_c> more_opt_callargs opt_morefunction_args string m_statements

%type <string_c> opt_morestring

%%

/* ---------------------------------------------------------------- */

program :					/*E*/
			{ 
				$$ = NULL; 
			}

		| statement program
			{ 
				$$ = Top = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}

		| function program
                        { 
				$$ = Top = asr_node(LIST, $1, $2);
				$$->asr_line = casl_line;
			}

		;

/* ---------------------------------------------------------------- */

function : 	T_PROC id T_O_PAREN opt_function_args T_C_PAREN T_O_BRACE
			function_body 
		T_C_BRACE
			{ 
				static char buf[8192];
				$$ = asr_node(N_FUNC, $2, $4, $7); 
				$$->asr_line = casl_line;
				snprintf(buf, 8192, "%s%s",
					CASL_FUNC_PREFIX, $2->asr_ident);
				buf[8191] = 0;
				st_insert(xstrdup(buf), $$, 0);
			}
		;

function_body :	statements
			{ 
				$$ = $1; 
			}
		;

/* ---------------------------------------------------------------- */

statements :		{ 
				$$ = NULL; 
		/*E*/	}

		| statement statements
			{ 
				$$ = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}
		;

m_statements : 	  statement statements
			{ 
				$$ = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}
		;
			
statement :	  opt_expr T_SEMI
			{ 
				$$ = asr_node(N_EXPR, $1); 
				$$->asr_line = casl_line;
			}

		| T_O_BRACE m_statements T_C_BRACE
			{ 	
				$$ = $2; 
			}

		| T_WHILE T_O_PAREN expr T_C_PAREN statement
			{ 
				$$ = asr_node(N_WHILE, $3, $5); 
				$$->asr_line = casl_line;
			}

		| T_IF T_O_PAREN expr T_C_PAREN statement opt_else
			{ 
				$$ = asr_node(N_ITE, $3, $5, $6); 
				$$->asr_line = casl_line;
			}

		| T_FOR T_O_PAREN opt_expr T_SEMI 
				  opt_expr T_SEMI 
			   	  opt_expr T_C_PAREN 
		  		statement
			{ 
				$$ = asr_node(N_FOR, $3, $5, $7, $9); 
				$$->asr_line = casl_line;
			}

		| structdef
			{ 
				$$ = $1; 
				$$->asr_line = casl_line;
			}

		| foreach
			{ 
				$$ = $1; 
				$$->asr_line = casl_line;
			}

		| T_CONTINUE T_SEMI
			{
				$$ = asr_node(N_CONTINUE);
				$$->asr_line = casl_line;
			}

		| T_BREAK T_SEMI
			{ 	
				$$ = asr_node(N_BREAK);
				$$->asr_line = casl_line;
			}

		| T_RETURN T_O_PAREN opt_expr T_C_PAREN T_SEMI
			{ 
				$$ = asr_node(N_RETURN, $3); 
				$$->asr_line = casl_line;
			}
		
		| T_DEBUG T_SEMI
			{ 	
				$$ = asr_node(N_DEBUG, NULL);
				$$->asr_line = casl_line;
			}
		
		| T_TRON T_SEMI
			{
				$$ = asr_node(N_TRON, NULL);
				$$->asr_line = casl_line;
			}

		| T_TROFF T_SEMI
			{
				$$ = asr_node(N_TROFF, NULL);
				$$->asr_line = casl_line;
			}

		| T_DUMP T_SEMI 
			{
				$$ = asr_node(N_DUMP, NULL);
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

idextract : 	  id T_DOT size range_spec 
			{ 
				$$ = asr_node(N_IDX, $1, $3, $4); 
				$$->asr_line = casl_line;
			}

		| id range_spec 
			{
				$$ = asr_node(N_IDX, $1, 
						     asr_sizet(S_DEF),
						     $2);
				$$->asr_line = casl_line;
			}

		| id T_DOT id 
			{ 
				$$ = asr_node(N_IDX, $1, $3); 
				$$->asr_line = casl_line;
			}
		;

range_spec :	  T_O_BRACKET expr T_C_BRACKET
			{ 
				$$ = asr_node(N_RANGE, $2, NULL);
				$$->asr_line = casl_line;
			}

		| T_O_BRACKET expr T_RANGE opt_end_range T_C_BRACKET 
			{ 
				$$ = asr_node(N_RANGE, $2, $4); 
				$$->asr_line = casl_line;
			}
		;

opt_end_range :  
		 	{ 
				$$ = asr_node(N_END, NULL); 
				$$->asr_line = casl_line;
		/*E*/	}	

		| expr
			{ 
				$$ = $1; 
			}
		;

/* ---------------------------------------------------------------- */

structdef : 	T_STRUCT id T_O_BRACE struct_body T_C_BRACE 
			{ 
				$$ = asr_node(N_STRUCT, $2, $4); 
				$$->asr_line = casl_line;
			}

struct_body :	field_decls
			{ 
				$$ = $1; 
			}
		;

field_decls : 		{ 
				$$ = NULL; 
		/*E*/	}	

		| field_decl field_decls
			{ 
				$$ = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}

		;

field_decl : 	  id T_COLON expr size T_SEMI
			{ 
				asr_t *n = asr_node(N_SIZESPEC, $3, $4);
				n->asr_line = casl_line;
			  	$$ = asr_node(N_STRUCTDEF, $1, n);
				$$->asr_line = casl_line;
			}

		| T_SCALE size T_SEMI
			{ 
				$$ = asr_node(N_SCALE, $2);
				$$->asr_line = casl_line;
			}

		;

/* ---------------------------------------------------------------- */

size : 		  T_BIT
			{
				$$ = asr_sizet(S_BIT); 
				$$->asr_line = casl_line;
			}

		| T_BYTE
			{ 
				$$ = asr_sizet(S_BYT); 
				$$->asr_line = casl_line;
			}

		| T_WORD
			{ 
				$$ = asr_sizet(S_WRD); 
				$$->asr_line = casl_line;
			}

		| T_DWORD
			{ 
				$$ = asr_sizet(S_DWD); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

foreach :	T_FOREACH id list statement
			{ 
				$$ = asr_node(N_FOREACH, $2, $3, $4); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

opt_else :		{ 
				$$ = NULL; 
		/*E*/	}	

		| T_ELSE statement
			{ 
				$$ = $2; 
			}
		;

/* ---------------------------------------------------------------- */

expr :		 
		  expr T_INCR %prec T_UNARY
			{
				$$ = asr_node(N_POSTINCR, $1);
				$$->asr_line = casl_line;
			}
		| expr T_DECR %prec T_UNARY
			{
				$$ = asr_node(N_POSTDECR, $1);
				$$->asr_line = casl_line;
			}
		| T_INCR expr %prec T_UNARY
			{
				$$ = asr_node(N_PREINCR, $2);
				$$->asr_line = casl_line;
			}
		| T_DECR expr %prec T_UNARY
			{
				$$ = asr_node(N_PREDECR, $2);
				$$->asr_line = casl_line;
			}
		| T_NEG expr %prec T_UNARY
			{ 
				$$ = asr_node(N_NOT, $2); 
				$$->asr_line = casl_line;
			} 

		| T_BNOT expr %prec T_UNARY
			{ 
				$$ = asr_node(N_BNOT, $2);
				$$->asr_line = casl_line;
			}

		| T_O_PAREN expr T_C_PAREN
			{ 
				$$ = $2; 
			}

		| expr T_AND expr %prec T_AND
			{ 
				$$ = asr_node(N_AND, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_OR expr %prec T_OR
			{ 	
				$$ = asr_node(N_OR, $1, $3); 
				$$->asr_line = casl_line;
			}

		| idextract
			{ 
				$$ = $1; 
			}

		| expr T_PLUS expr
			{ 
				$$ = asr_node(N_PLUS, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_MINUS expr
			{ 
				$$ = asr_node(N_MINUS, $1, $3); 
				$$->asr_line = casl_line;
			}

		| T_MINUS expr %prec T_UNARY
			{
				$$ = asr_node(N_MINUS, asr_int (0, casl_line), $2);
				$$->asr_line = casl_line;
			}

		| expr T_MUL expr
			{ 
				$$ = asr_node(N_MUL, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_DIV expr
			{ 
				$$ = asr_node(N_DIV, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_MOD expr
			{
				$$ = asr_node(N_MOD, $1, $3);
				$$->asr_line = casl_line;
			}

		| expr T_BLSHIFT expr
			{
				$$ = asr_node(N_BLSHIFT, $1, $3);
				$$->asr_line = casl_line;
			}

		| expr T_BRSHIFT expr
			{
				$$ = asr_node(N_BRSHIFT, $1, $3);
				$$->asr_line = casl_line;
			}

		| expr T_BOR expr
			{
				$$ = asr_node(N_BOR, $1, $3);
				$$->asr_line = casl_line;
			}
	
		| expr T_BAND expr
			{
				$$ = asr_node(N_BAND, $1, $3);
				$$->asr_line = casl_line;
			}
	
		| expr T_BXOR expr
			{
				$$ = asr_node(N_BXOR, $1, $3);
				$$->asr_line = casl_line;
			}

		| expr T_PUSH expr
			{ 
				$$ = asr_node(N_PLIST, $1, $3);
				$$->asr_line = casl_line;
			}

		| expr T_APPEND expr
			{
				$$ = asr_node(N_PTAIL, $1, $3);
				$$->asr_line = casl_line;
			}

		| T_EXTRACT id T_FROM id
			{ 
				$$ = asr_node(N_EXTRACT, $2, $4); 
				$$->asr_line = casl_line;
			}
 
		| T_EXTRACT expr T_BYTE T_FROM id
			{ 
				$$ = asr_node(N_EXTRACT2, $2, $5);
				$$->asr_line = casl_line;
			}

		| expr T_EQ expr
			{ 
				$$ = asr_node(N_EQ, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_NE expr
			{ 
				$$ = asr_node(N_NE, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_GT expr
			{
				$$ = asr_node(N_GT, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_LT expr
			{ 
				$$ = asr_node(N_LT, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_GE expr
			{ 
				$$ = asr_node(N_GE, $1, $3); 
				$$->asr_line = casl_line;
			}

		| expr T_LE expr
			{ 
				$$ = asr_node(N_LE, $1, $3); 
				$$->asr_line = casl_line;
			}

		| assign
			{ 
				$$ = $1; 
			}

		| call
			{ 
				$$ = $1; 
			}

		| list
			{ 
				$$ = $1; 
			}

		| id
			{ 
				$$ = $1; 
			}

		| T_NAME expr
			{ 	
				$$ = asr_node(N_SNAME, $2);
				$$->asr_line = casl_line;
			}
            
		| T_COPY expr
			{
				$$ = asr_node(N_COPY, $2);
				$$->asr_line = casl_line;
			}

		| T_POP expr
			{ 
				$$ = asr_node(N_POP, $2); 
				$$->asr_line = casl_line;
			}

		| T_TAIL expr
			{
				$$ = asr_node(N_TAIL, $2);
				$$->asr_line = casl_line;
			}
	
		| T_INT_LIT
			{ 
				$$ = asr_int(yylval.int_c, casl_line); 
				$$->asr_line = casl_line;
			}

		| T_CHAR_LIT
			{ 
				$$ = asr_char(yylval.char_c, casl_line); 
				$$->asr_line = casl_line;
			}
		
		| string
			{
				$$ = $1;
			}

		;

/* ---------------------------------------------------------------- */

assign :	  id T_ASSIGN expr
			{ 
				$$ = asr_node(N_ASSIGN, $1, $3); 
				$$->asr_line = casl_line;
			}

		| id T_ASSIGN T_NEW id
			{ 
				$$ = asr_node(N_NEW, $1, $4); 
				$$->asr_line = casl_line;
			}

		| idextract T_ASSIGN expr
			{ 
				$$ = asr_node(N_ASSIGN, $1, $3); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

call :		id T_O_PAREN opt_callargs T_C_PAREN
			{ 
				$$ = asr_node(N_CALL, $1, $3); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

list :		T_O_BRACKET opt_expr opt_morelistargs T_C_BRACKET
			{ 
				$$ = asr_node(N_CREATELIST,
					asr_node(LIST, $2, $3));
				$$->asr_line = casl_line;
			} 
		;

/* ---------------------------------------------------------------- */

opt_morelistargs : 	{ 
				$$ = NULL; 
		/*E*/	} 

		| T_COMMA expr opt_morelistargs
			{ 
				$$ = asr_node(LIST, $2, $3); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

opt_callargs :		{ 
				$$ = NULL; 
		/*E*/	} 	

		| expr more_opt_callargs
			{ 
				$$ = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

more_opt_callargs : 	{ 
				$$ = NULL; 
		/*E*/	}	

		| T_COMMA expr more_opt_callargs
			{ 
				$$ = asr_node(LIST, $2, $3); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

opt_function_args :	{ 
				$$ = NULL; 
		/*E*/	}	

		| id opt_morefunction_args
			{ 
				$$ = asr_node(LIST, $1, $2); 
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

opt_morefunction_args :	{ 
				$$ = NULL; 
		/*E*/	}	

		| T_COMMA id opt_morefunction_args
			{ 
				$$ = asr_node(LIST, $2, $3); 
				$$->asr_line = casl_line;
			}	
		;

/* ---------------------------------------------------------------- */

opt_expr :		{ 
				$$ = NULL; 
		/*E*/	}	

		| expr
			{ 
				$$ = $1; 
			}
		;

/* ---------------------------------------------------------------- */

id : 		T_ID
			{ 
				$$ = asr_id(yylval.ident_c, casl_line); 
			}
		;

/* ---------------------------------------------------------------- */

string : 	T_STRING_LIT opt_morestring
			{
				char *cp = strap(xstrdup($1), $2);
				asr_t *ap = asr_string(cp, casl_line);
				$$ = asr_node(N_COPY, ap);
				$$->asr_line = casl_line;
			}
		;

/* ---------------------------------------------------------------- */

opt_morestring:	/*E*/
			{
				$$ = NULL;
			}	

		| T_STRING_LIT opt_morestring
 			{
				$$ = strap(xstrdup($1), $2);
			}
		;

/* ---------------------------------------------------------------- */


