/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/* Linux specefic code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <net/route.h>

#include "nettab.h"

#define MAX_IFACES      100
#define IPADDR(sockaddr)        \
        ((struct sockaddr_in *)(sockaddr))->sin_addr.s_addr
#define ROUNDUP(x)      (((x) + (sizeof(long) - 1)) & ~(sizeof(long) - 1))

/* ----------------------------------------------------------------------------
** Go to the next interface record.
*/

static struct ifreq *
next_ifr(struct ifreq *ifr)
{
#ifdef IFR_VARSIZE
    int delta;

    if(ifr->ifr_addr.sa_len)
        delta = ifr->ifr_addr.sa_len - sizeof(struct sockaddr);
    else
        delta = 0;

    ifr = (struct ifreq *)((char *)ifr + delta);
#endif
    ifr ++;
    return ifr;
}

/* ----------------------------------------------------------------------------
** Return a list of interface data.
*/

struct iflist *
get_iflist()
{
    struct ifreq *ifr, *end, ifreq;
    struct ifconf ifc;
    struct iflist *head, *tail = 0, *new;
    char confbuf[MAX_IFACES * sizeof(struct ifreq)];
    int sock, addr, mask, flags;

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if(sock < 0)
        return 0;

    ifc.ifc_buf = confbuf;
    ifc.ifc_len = sizeof confbuf;
    if(ioctl(sock, SIOCGIFCONF, &ifc) == -1) {
        close(sock);
        return 0;
    }

    end = (struct ifreq *)(confbuf + ifc.ifc_len);
    head = 0;
    for(ifr = ifc.ifc_req; ifr < end; ifr = next_ifr(ifr)) {
        if(ifr->ifr_addr.sa_family != AF_INET)
            continue;
        addr = IPADDR(&ifr->ifr_addr);

        ifreq = *ifr;
        if(ioctl(sock, SIOCGIFNETMASK, &ifreq) == -1)
            continue;
        mask = IPADDR(&ifreq.ifr_addr);
        if(ioctl(sock, SIOCGIFFLAGS, &ifreq) == -1)
            continue;
        flags = ifreq.ifr_flags;

        if((flags & IFF_UP) == 0)
            continue;

        new = malloc(sizeof *new);
        memset(new, 0, sizeof *new);
        if(!new) {
            close(sock);
            destroy_iflist(head);
            return 0;
        }
        new->name = strdup(ifr->ifr_name);
        new->ipaddr = addr;
        new->netmask = mask;

        if(ioctl(sock, SIOCGIFHWADDR, &ifreq) == 0)
            memcpy(new->macaddr, ifreq.ifr_hwaddr.sa_data, 6);

        if(head)
            tail->next = new;
        else
            head = new;
        tail = new;
    }
    close(sock);
    return head;
}

/* ----------------------------------------------------------------------------
** Get a list of routes.
*/

struct routelist *
get_routelist(struct iflist *iflist)
{
    char buf[150], ifname[16];
    struct iflist *t;
    struct routelist *head, **tailp, *new;
    FILE *fp;
    int flags, metric, addr, gwaddr, maskaddr;

    fp = fopen("/proc/net/route", "r");
    if(!fp)
        return 0;

    head = 0;
    tailp = &head;
    while(fgets(buf, sizeof buf - 2, fp)) {
        if(sscanf(buf, "%s %x %x %x %*d %*d %d %x %*d %*d %*d\n",
                  ifname, &addr, &gwaddr, &flags, &metric, &maskaddr) != 6)
            continue;

        if(!(flags & RTF_GATEWAY))
            continue;

        for(t = iflist; t; t = t->next) {
            if(strcmp(ifname, t->name) == 0)
                break;
        }
        if(!t)
            continue;

        new = malloc(sizeof *new);
        if(!new) {
            fclose(fp);
            destroy_routelist(head);
            return 0;
        }
        memset(new, 0, sizeof *new);
        new->ipaddr = addr;
        new->netmask = maskaddr;
        new->gateway = gwaddr;
        new->metric = metric;
        new->iface = t;

        *tailp = new;
        tailp = &new->next;
    }
    fclose(fp);
    return head;
}
