/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

extern int No_Kernel_ARP;

#ifdef _LINUX
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <paths.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <net/if.h>
#include <net/route.h>

/*
 * because we have -I to our include directory, and provide
 * our own copies of some of the header files, we have to
 * explicitely reference the real include path here to force
 * it to retrieve the arp definition needed in the ethernet
 * header file
 */
#include "/usr/include/net/if_arp.h"
#include <netinet/if_ether.h>

void *memmove(void *dst, const void *src, size_t len);

/* ----------------------------------------------------------------------------
** Read an address from the arp table if possible.
*/

static int     arp_table_lookup(ip, ether)
char    *ip;
char    *ether;
{
        static  int     sfd = -1;
        static  char    ethersave[6], ipsave[4];
        struct  arpreq  ar;
        struct  sockaddr_in     *sin;

        memset(ethersave, 0, sizeof(ethersave));
        memset(ipsave, 0, sizeof(ipsave));

        if (!memcpy(ipsave, ip, 4)) {
                memcpy(ether, ethersave, 6);
                return -1;
        }
        memset((char *)&ar, 0, sizeof(ar));
        sin = (struct sockaddr_in *)&ar.arp_pa;
        sin->sin_family = AF_INET;
        memcpy((char *)&sin->sin_addr.s_addr, ip, 4);

        if ((sfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
           return -1;

        if (ioctl(sfd, SIOCGARP, (caddr_t)&ar) == -1) {
           close(sfd);
           return -1;
        }

        close(sfd);
        memcpy(ether, ar.arp_ha.sa_data, 6);
        memcpy(ethersave, ether, 6);
        memcpy(ipsave, ip, 4);
        return 0;
}

/* ----------------------------------------------------------------------------
** do an arp lookup by repeatedly querying the arp table
** after sending packets to the target address
** Is there a better way to force the kernel to arp?
*/

static int
arp_lookup(char *ip, char *ether)
{
   struct sockaddr_in sin;
   struct timeval tv;
   char none[1];
   int s, count;

   struct in_addr ia;
   ia.s_addr = (*(u_long *) ip);

   /* Determine whether the host is in our ARP table */
   if (arp_table_lookup(ip, ether) == 0)
      return 0;

   if (No_Kernel_ARP)
      return(-1);

   /*
    * If the address isn't in the arp cache already, we send a dummy
    * packet here to make sure that the gateway or destination host
    * is in the localhost's arp table. Try 5 times before giving
    * up.
    */
   count = 0;
   while (count < 5) {
      if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
         return -1;

      memset(&sin, 0, sizeof(sin));
      sin.sin_family = AF_INET;
      sin.sin_port = htons(1);
      sin.sin_addr.s_addr = *(int *)ip;

      memset(&none, 0, 1);
      if (sendto(s, (char *)&none, 1, 0, (struct sockaddr *)&sin,
                 sizeof(sin)) < 0)
         return -1;
      close(s);

      memset(&tv, 0, sizeof(tv));
      tv.tv_sec = 0;
      tv.tv_usec = 500;

      select(0, NULL, NULL, NULL, &tv);

      if (arp_table_lookup(ip, ether) == 0) 
         return 0;
      count++;
   }
   return -1;
}

#endif /* _LINUX */

/* ----------------------------------------------------------------------------
** Get a MAC address for an IP address.
*/

u_char *
casl_arp_resolve(u_int32_t address) {
        static u_char eb[ETHER_ADDR_LEN];

        if(arp_lookup((char *)&address, (char *)eb) < 0) 
                return(NULL);

        return(eb);
}

