/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/* OS independant code - OS dependant code goes in os/$OS/ostab.c */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "nettab.h"

static struct iflist *ifaces = 0;
static struct routelist *routes = 0;
static int init = 0;

/* ---------------------------------------------------------------------------
** Initialize routing table information.
*/

static int
route_init()
{
    if(init && (!ifaces || !routes))
        return -1;
    if(init)
        return 0;

    init = 1;
    ifaces = get_iflist();
    if(!ifaces)
        return -1;
    routes = get_routelist(ifaces);
    if(!routes)
        return -1;
    return 0;
}

/* ---------------------------------------------------------------------------
** Lookup the interface needed to directly reach an ipaddr.
*/

int
iface_lookup(int ipaddr, struct iflist **ifacep)
{
    struct iflist *iface;

    if(route_init() == -1)
        return -1;

    for(iface = ifaces; iface; iface = iface->next) {
        if((ipaddr & iface->netmask) == (iface->ipaddr & iface->netmask)) {
            *ifacep = iface;
            return 0;
        }
    }
    return -1;
}

/* ---------------------------------------------------------------------------
** Lookup the route for an IP address.
*/
int
route_lookup(int ipaddr, int *nexthopp, struct iflist **ifacep)
{
    struct iflist *iface;
    struct routelist *route;
    unsigned int bestmask, curmask;
    int match, maskedip, bestmetric;

    if(route_init() == -1)
        return -1;

    bestmask = 0;
    bestmetric = 0x7fffffff;
    match = 0;

    /* find best match in the route table */
    for(route = routes; route; route = route->next) {
        /* route has to match */
        maskedip = ipaddr & route->netmask;
        if(maskedip != (route->ipaddr & route->netmask))
            continue;

        /* cant be worse than our current best */
        if(route->metric > bestmetric)
            continue;

        /* cant be less specific than our current best */
        curmask = ntohl(maskedip);
        if(curmask < bestmask)
            continue;

        match = 1;
        *nexthopp = route->gateway;
        bestmask = curmask;
        bestmetric = route->metric;
        *ifacep = route->iface;
    }

    /* find best match in the iface table (directly connected) */
    for(iface = ifaces; iface; iface = iface->next) {
        maskedip = ipaddr & iface->netmask;
        if(maskedip != (iface->ipaddr & iface->netmask))
            continue;

        curmask = ntohl(maskedip);
        if(curmask < bestmask)
            continue;

        match = 1;
        *nexthopp = ipaddr;
        bestmask = curmask;
        bestmetric = 1;
        *ifacep = iface;
    }

    if(match)
        return 0;
    return -1;
}

/* ---------------------------------------------------------------------------
** Clean up the interface list.
*/

void
destroy_iflist(struct iflist *t)
{
    struct iflist *next;

    while(t) {
       next = t->next;
       free(t->name);
       free(t);
       t = next;
    }
    return;
}

/* ---------------------------------------------------------------------------
** Clean up the route list.
*/
void
destroy_routelist(struct routelist *rlist)
{
    struct routelist *next;

    while(rlist) {
        next = rlist->next;
        free(rlist);
        rlist = next;
    }
    return;
}
