/*
 * Parts of this file are copyright David R. Hanson, from
 * "C Interfaces and Implementations"
 *
 *
 * The author of this software is David R. Hanson.
 * 
 * Copyright (c) 1994,1995,1996,1997 by David R. Hanson. All Rights Reserved.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, subject to the provisions described below, without fee is
 * hereby granted, provided that this entire notice is included in all
 * copies of any software that is or includes a copy or modification of
 * this software and in all copies of the supporting documentation for
 * such software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY. IN PARTICULAR, THE AUTHOR DOES MAKE ANY REPRESENTATION OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 * 
 * David Hanson / drh@microsoft.com / http://www.research.microsoft.com/~drh/
 */

#include "table.h"
#include "utils.h"
#include <assert.h>

int primes[] = { 509, 1021, 2053, 4093, 8191, 16381, 
			32771, 65521, INT_MAX };

#define MAX_BUCKETS 2048
static struct bucket *bucket_cache = 0;
static int bucket_cachec = 0;

/* ----------------------------------------------------------------------------
** Allocate a bucket struct.  If there are buckets available in the cache, use
** it!  Otherwise, use malloc as a last resort.
*/

static inline struct bucket *alloc_bucket ()
{
	struct bucket *p = bucket_cache;
	if (p) {
		bucket_cachec--;
		bucket_cache = bucket_cache->next;
	} else
		p = (struct bucket *) malloc (sizeof (struct bucket));
	return p;
}

/* ----------------------------------------------------------------------------
** Free a bucket.  If there is space in the cache, store it there.  Only free
** it if things start to get too crowded.
*/

static inline void free_bucket (struct bucket *entry)
{
	if (bucket_cachec < MAX_BUCKETS) {
		entry->next = bucket_cache;
		bucket_cache = entry;
		bucket_cachec++;
	} else
		free (entry);
}

/* ----------------------------------------------------------------------------
** Allocate a new hash table.
*/

struct table *t_new(hint, compare, hash)
int hint;
int compare(const void *x, const void *y);
unsigned hash(const void *key);
{
  struct table *table;
  int i;
  
  assert(hint >= 0);
  
  /* find adequate table size */
  for(i = 1; primes[i] < hint; i++);
  i--;

  /* allocate the table structure */
  table = (struct table *) malloc(sizeof(struct table));
  if(!table) 
    return(NULL);

  /* initialize the table */
  table->size    = primes[i];
  table->compare = compare ? compare : tableCompareStrings;
  table->hash    = hash    ? hash    : tableHashStrings;
  
  /* allocate the hash table */
  table->buckets = (struct bucket **) malloc(primes[i] *
					      sizeof(struct bucket *));
  if(!table->buckets)
    return(NULL);
  
  /* initialize the hash table */
  for(i = 0; i < table->size; i++) 
    table->buckets[i] = NULL;
  
  table->length = 0;
  table->timestamp = 0;
  
  return(table);
}

/* ----------------------------------------------------------------------------
** Find a key in the table.  Return it's value.
*/

void *t_get(struct table *table, const void *key) {
  unsigned i;
  struct bucket *b;
  
  assert(table);
  assert(key);
  
  /* hash */
  i = table->hash(key) % table->size;

  /* check the bucket, check for collisions */
  for(b = table->buckets[i]; b; b = b->next)
    if(!(table->compare(key, b->key))) 
      break;
  
  return(b ? b->value : NULL);
}

/* ----------------------------------------------------------------------------
** Add a new key/value pair.
*/

void *t_put(struct table *table, const void *key, void *value) {
  unsigned i;
  struct bucket *b;
  void *prev;
  
  assert(table);
  assert(key);
  
  /* hash */
  i = table->hash(key) % table->size;
  
  /* find matching key */
  for(b = table->buckets[i]; b; b = b->next) 
    if(!(table->compare(key, b->key)))
      break;
    
  /* no such entry exists in the table */
  if(!b) {
    b = alloc_bucket();
    assert(b);
    
    /* initialize new entry */
    b->key  = key;
    b->next = table->buckets[i];
    table->buckets[i] = b;
    table->length++;
    prev = NULL;
  
  } else
    prev = b->value;
  
  b->value = value;
  table->timestamp++;
  
  /* if we updated a key, return the old value so we know */
  return(prev);
}

/* ----------------------------------------------------------------------------
** Return the number of entries in the table.
*/

int t_length(struct table *table) {
  assert(table);
  
  return(table->length);
}

/* ----------------------------------------------------------------------------
** A warning to all who come this way with intent to change that which they
** seek to know.  Such actions will surely cause an awful death.  Those
** who'se code is pure may seek the truth of all the keys to unlock all the
** values once held in the mystic hash.
*/

void t_map(table, apply, closure) 
struct table *table;
void apply(const void *key, void **value, void *closure);
void *closure;
{
  int i;

  /* to verify that the table hasn't been modified */
  unsigned stamp;

  struct bucket *b;
  
  assert(table);
  assert(apply);
  
  stamp = table->timestamp;
  
  for(i = 0; i < table->size; i++) {
    for(b = table->buckets[i]; b; b = b->next) {
      apply(b->key, &b->value, closure);

      /* apply function cannot modify the table */
      assert(table->timestamp == stamp);
    }
  }

  return;
}

/* ----------------------------------------------------------------------------
** Remove a key/value pair from the table.
*/

void *t_remove(struct table *table, const void *key) {
  unsigned i;
  struct bucket **bb;
  
  assert(table);
  assert(key);
  
  table->timestamp++;
  
  i = table->hash(key) % table->size;
  
  for(bb = &table->buckets[i]; *bb; bb = &(*bb)->next) {
    if(!(table->compare(key, (*bb)->key))) {
     struct bucket *b = *bb;
     void *value = b->value;
     *bb = b->next;
     free_bucket(b);
     table->length--;
     return(value);
    }
  }
  
  return(NULL);
}

/* ----------------------------------------------------------------------------
** Return an array of key/value pairs based on the list.
*/

void **t_2array(struct table *table, void *end) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  /* allocate a pointer to key and value for each entry */
  array = malloc( (2 * (table->length + 1) ) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) {
    for(b = table->buckets[i]; b; b = b->next) {
      array[j++] = (void *) b->key;
      array[j++] = (void *) b->value;
    }
  }
   
  array[j] = end;
  
  return(array);
}

/* ----------------------------------------------------------------------------
** Return an array of just the keys.
*/

void **t_keys(struct table *table) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  array = malloc( (table->length + 1) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) 
    for(b = table->buckets[i]; b; b = b->next) 
      array[j++] = (void *) b->key;
   
  array[j] = NULL;
  
  return(array);
}

/* ----------------------------------------------------------------------------
** Return an array of just the values.
*/

void **t_values(struct table *table) {
  int i;
  int j = 0;
  
  void **array;
  
  struct bucket *b;
  
  assert(table);
  
  array = malloc( (2 * (table->length + 1) ) * sizeof(*array));
  if(!array)
    return(NULL);
  
  for(i = 0; i < table->size; i++) 
    for(b = table->buckets[i]; b; b = b->next) 
        array[j++] = (void *) b->value;
       
  array[j] = NULL;
  
  return(array);
}

/* ----------------------------------------------------------------------------
** Free an entire whole table.
*/

void table_free(struct table **table) {
  table_t *t = *table;

  assert(table);
  assert(*table);
  
  if((*table)->length) {
    int i;
    struct bucket *p, *q;
    for(i = 0; i < (*table)->size; i++) {
      for(p = (*table)->buckets[i]; p; p = q) {
	q = p->next;
	free_bucket(p);
      }
    }
  }

  free(t->buckets);
  free(*table);
  
  return;
}

/* ----------------------------------------------------------------------------
** A very complicated and involved functions few will understand that takes two
** pointers and iterates each in parallal comparing the byte value pointed to
** by each until either their is a difference or the value being compared is
** not posative or negative.
*/

int tableCompareStrings(const void *x, const void *y) {
	return(strcmp((char *)x, (char *)y));
}

/* ----------------------------------------------------------------------------
** Turn a string into a hash value.
*/

unsigned tableHashStrings(const void *key) {
	unsigned h = 0;
	unsigned g;

	char *name = (char *) key;

	while(*name) {
		h = (h << 4) + *name++;
		if((g = (h & 0xf0000000)))
			h ^= g >> 24;
	
		h &= ~g;
	}

	return(h);
}	
