/*
 * snprintf.c
 *	Contains snprintf for systems that don't have their own
 *
 * This file has sections for different snprintf functions for 
 * different systems.  The sections in this file are:
 *   _IRIX
 *   _SUNOS5
 *   _WIN32
 *
 * systems that have their own snprintf and are:
 *   _LINUX
 *   _FREEBSD
 *   _OPENBSD
 *   _BSDI
 */



/* ------------------------------------------------------------------ */
/* snprintf() for irix */
#if defined(_IRIX)

#include <sys/types.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef _IOSTRG
#define STRFLAG	(_IOSTRG|_IOWRT)	/* no _IOWRT: avoid stdio bug */
#else
#define STRFLAG	(_IOREAD)		/* XXX: Assume svr4 stdio */
#endif

void _doprnt(char *, va_list, FILE *);

int
vsnprintf(s, n, fmt, args)
	char *s, *fmt;
	size_t n;
	va_list args;
{
	FILE fakebuf;

	fakebuf._flag = STRFLAG;
	fakebuf._ptr = (void *) s;
	fakebuf._cnt = n-1;
	_doprnt(fmt, args, &fakebuf);
	fakebuf._cnt++;
	putc('\0', &fakebuf);
	if (fakebuf._cnt<0)
	    fakebuf._cnt = 0;
	return (n-fakebuf._cnt-1);
}

int
#ifdef __STDC__
snprintf(char *s, size_t ss, const char *fmt, ...)
#else
snprintf(va_alist)
	va_dcl
#endif
{
	va_list ap;
	int n;

#ifdef __STDC__
	va_start(ap, fmt);
#else
	char *s, *fmt;
	size_t ss;

	va_start(ap);
	s = va_arg(ap, char *);
	ss = va_arg(ap, size_t);
	fmt = va_arg(ap, char *);
#endif

	n = vsnprintf(s, ss, fmt, ap);

	va_end(ap);
	return n;
}
#endif /* irix */




/* ------------------------------------------------------------------ */
/* snprintf() for solaris */
#if defined(_SUNOS) && !defined(_SOLARIS26)
/*
 * Copyright (c) 1997 Theo de Raadt
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <signal.h>
#include <stdio.h>
#if __STDC__
#include <stdarg.h>
#include <stdlib.h>
#else
#include <varargs.h>
#endif
#include <setjmp.h>
#include <sys/sysmacros.h>

static int pgsize;
static char *curobj;
static int caught;
static jmp_buf bail;

static char *
msetup(str, n)
        char *str;
        size_t n;
{
        char *e;

        if (pgsize == 0)
                pgsize = getpagesize();

        if (n == 0) {
                *str = '\0';
                return 0;
        }
        curobj = (char *)malloc(n + pgsize * 2);
        if (curobj == NULL)
                return NULL;
        e = curobj + n;
        e = (char *)roundup((unsigned long)e, pgsize);
        if (mprotect(e, pgsize, PROT_NONE) == -1) {
                free(curobj);
                curobj = NULL;
                return NULL;
        }
        return (e - n - 2);
}

static void
mcatch()
{
        longjmp(bail, 1);
}

static void
mcleanup(str, n, p)
        char *str;
        size_t n;
        char *p;
{
        strncpy(str, p, n-1);
        str[n-1] = '\0';
        free(curobj);
        if (mprotect((caddr_t)(p + n), pgsize,
            PROT_READ|PROT_WRITE|PROT_EXEC) == 0)
                return;
        mprotect((caddr_t)(p + n), pgsize,
            PROT_READ|PROT_WRITE);
}


int
#if __STDC__
snprintf(char *str, size_t n, char const *fmt, ...)
#else
snprintf(str, n, fmt, va_alist)
        char *str;
        size_t n;
        char *fmt;
        va_dcl
#endif
{
        va_list ap;
        int ret;
        char *p;
        void (*sigsegv)() = NULL;

#if __STDC__
        va_start(ap, fmt);
#else
        va_start(ap);
#endif

	ret = vsprintf(p, fmt, ap);
	va_end(ap);
        return(ret);

        p = msetup(str, n);
        if (p == NULL)
                return 0;
        if (setjmp(bail) == 0) {
                sigsegv = signal(SIGSEGV, mcatch);
                ret = vsprintf(p, fmt, ap);
        }
        va_end(ap);
        mcleanup(str, n, p);
        (void) signal(SIGSEGV, sigsegv);
        return (ret);
}
#endif /* sunos5 */




/* ------------------------------------------------------------------ */
/* snprintf for bsdi and win32 */
#if defined(_WIN32)
#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)snprintf.c	8.1 (Berkeley) 6/4/93";
#endif /* LIBC_SCCS and not lint */

#include <stdio.h>
#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#if __STDC__
int snprintf(char *str, size_t n, char const *fmt, ...)
#else
int snprintf(str, n, fmt, va_alist)
	char *str;
	size_t n;
	char *fmt;
	va_dcl
#endif
{
	int ret;
	va_list ap;
	FILE f;

	if ((int)n < 1)
		return (EOF);
#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	f._flags = __SWR | __SSTR;
	f._bf._base = f._p = (unsigned char *)str;
	f._bf._size = f._w = n - 1;
#ifdef _WIN32
	f._data = _REENT;
#endif
	ret = vfprintf(&f, fmt, ap);
	*f._p = 0;
	va_end(ap);
	return (ret);
}
#endif /* bsdi, win32 */


