%{

/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * Redistribution and use are governed by the terms detailed in the
 * license document ("LICENSE.TXT") included with this source.  If the
 * document is missing, it may be obtained from Network Associates, Inc.
 */

#include "casl.h"
#include "y.tab.h"


#ifdef LEX_DEBUG
#define return_token(x) \
	printf("--> %s\t\t\t%s\n", #x, yytext); \
	return(x)
#else
#define return_token(x) \
	return(x)
#endif

#ifdef LEX_DEBUG
#define lineup() \
	casl_line++; \
	printf("-----------------[%4d]---------------------\n", casl_line)
#else
#define lineup() \
	casl_line++; 
#endif

int casl_line = 1;
char *casl_file = NULL;

char *xstrdup(char *);

extern void error(int, char *, ...);
extern char *CASL_Script;

 void multi_comment() {
	 char c;

	 for(;;) {
		 while((c = input()) != '*' && c != 0)
			 if(c == '\n')
				lineup();

		 while((c = input()) == '*' && c != 0)
			/**/ ;

		 if(c == 0) 
			 error(E_USER, "comment continues past end of file");
		 
		 if(c == '/');
			 break;
	 }
	 
	 return;
 }
	 
void single_comment() {
	char c;

	while((c = input()) != '\n' && c)
		 /**/ ;

	if(c)
		lineup();

	return;
}

void cpp_control() {
	char *file;
	char *cp;

	/* ditch non-line-numbers */

	if(atoi(yytext + 1) == 0) 
		return;

	/* don't include the current line as an actual line number */

	casl_line = atoi(yytext + 1) - 1;

	file = strchr(yytext, '\"');
	if(!file)
		return;
	file += 1;

	cp = strchr(file, '\\');
	if(!cp)
		cp = strchr(file, '\"');
	if(!cp)
		return;

	*cp = 0;

	if(!*file)
		casl_file = CASL_Script;
	else if(isprint(*file))
		casl_file = xstrdup(file);

	return;
}

char *escape(char *ptr) {
	char *cp, *rp;
	char buf[3];

	for(cp = rp = ptr; *cp; cp++) {
		if(*cp != '\\') 
			*rp++ = *cp;
		else if(!*(cp++)) 
			error(E_USER, "unterminated string literal");
		else {
			switch(*cp) {
			case 'n':
				*rp++ = '\n';
				break;

			case 'x': 
				buf[0] = cp[1];
				buf[1] = cp[2];
				buf[2] = 0;

				if(!buf[0] || !buf[1])
					error(E_USER,
					"malformed hexidecimal string");

				*rp++ = (int) strtoul(buf, NULL, 16);
				
				cp = &cp[2];
				break;
			
			case 't':
				*rp++ = '\t';
				break;

			case 'r':
				*rp++ = '\r';
				break;

			case '"':
				*rp++ = '"';
				break;

			case '\'':
				*rp++ = '\'';
				break;

			case '\\':
				*rp++ = '\\';
				break;

			case '\n':
				break;

			default:
				error(E_USER, "unrecognized escape character %c", 
					*cp);

				break;
			}
		}
	}

	if(*rp)
		*rp = 0;

	return(ptr);
}

%}

%option noyywrap

alpha		[a-zA-Z_]
num		[0-9]
hexnum		[0-9a-fA-F]
alphanum	[a-zA-Z_0-9]
text		[^\n\r"]
char		[^\n\r']
white		[ \t]
any		.

%%

^"#".*"\n"	{ cpp_control(); 	}
"/*"		{ multi_comment(); 	}
"//"		{ single_comment(); }
">="		{ return_token(T_GE); }
"<="		{ return_token(T_LE); }
"=="		{ return_token(T_EQ); }
"!="		{ return_token(T_NE); }
">"		{ return_token(T_GT); }
"<"		{ return_token(T_LT); }
"&&"		{ return_token(T_AND); }
"||"		{ return_token(T_OR); }

"["		{ return_token(T_O_BRACKET); }
"]"		{ return_token(T_C_BRACKET); }

"="		{ return_token(T_ASSIGN); }
";"		{ return_token(T_SEMI); }

"!"		{ return_token(T_NEG); }
"+"		{ return_token(T_PLUS); }
"-"		{ return_token(T_MINUS); }
"*"		{ return_token(T_MUL); }
"/"		{ return_token(T_DIV); }
"%"		{ return_token(T_MOD); }
"<<"		{ return_token(T_BLSHIFT); }	
">>"		{ return_token(T_BRSHIFT); }
"|"		{ return_token(T_BOR); }
"&"		{ return_token(T_BAND); }
"^"		{ return_token(T_BXOR); }
"~"		{ return_token(T_BNOT); }
"++"		{ return_token(T_INCR); }
"--"		{ return_token(T_DECR); }

"("		{ return_token(T_O_PAREN); }
")"		{ return_token(T_C_PAREN); }
"{"		{ return_token(T_O_BRACE); }
"}"		{ return_token(T_C_BRACE); }

"." 		{ return_token(T_DOT); }
":"		{ return_token(T_COLON); }
".."		{ return_token(T_RANGE); }

","		{ return_token(T_COMMA); }

"debug"		{ return_token(T_DEBUG); }
"dump"		{ return_token(T_DUMP); }
"tron"		{ return_token(T_TRON); }
"troff"		{ return_token(T_TROFF); }

"new" 		{ return_token(T_NEW); }
"copy"		{ return_token(T_COPY); }
"extract"	{ return_token(T_EXTRACT); }
"nameof"	{ return_token(T_NAME); }
"from"		{ return_token(T_FROM); }
"if"		{ return_token(T_IF); }
"else"		{ return_token(T_ELSE); }
"while"		{ return_token(T_WHILE); }
"foreach"	{ return_token(T_FOREACH); }
"for"		{ return_token(T_FOR); }
"return"	{ return_token(T_RETURN); }
"proc"		{ return_token(T_PROC); }
"push"		{ return_token(T_PUSH); }
"append"	{ return_token(T_APPEND); }
"prepend"	{ return_token(T_PUSH); }
"pop"		{ return_token(T_POP); }
"head"		{ return_token(T_POP); }
"tail"		{ return_token(T_TAIL); }
"break"		{ return_token(T_BREAK); }
"continue"	{ return_token(T_CONTINUE); }
"define" 	{ return_token(T_STRUCT); }

"bit"		{ return_token(T_BIT); }
"bits"		{ return_token(T_BIT); }
"byte"		{ return_token(T_BYTE); }
"bytes"		{ return_token(T_BYTE); }
"word"		{ return_token(T_WORD); }
"words"		{ return_token(T_WORD); }
"dword"		{ return_token(T_DWORD); }
"dwords"	{ return_token(T_DWORD); }
"variable"	{ return_token(T_VARIABLE); }
"scale"		{ return_token(T_SCALE); }

"'"({char}|{char}{char})"'"	{ 
					char *cp;
		
					cp = strrchr(yytext, '\'');
					if(cp)
						*cp = 0;
	
					cp = escape(&yytext[1]);	

					yylval.char_c = *cp;	

					return_token(T_CHAR_LIT); 
				}

"\""({text}|"\\\"")*"\""	{ 
					char *cp = strrchr(yytext, '"');
					if(cp)
						*cp = '\0';

					yylval.string_c = escape(yytext + 1);

					return_token(T_STRING_LIT); 
				}

{num}*				{ yylval.int_c = strtoul(yytext, NULL, 10);
				  return_token(T_INT_LIT);
				}

"0x"{hexnum}*			{ yylval.int_c = strtoul(yytext, NULL, 16);
				  return_token(T_INT_LIT); 
				}

{num}+"\."{num}+"\."{num}+"\."{num}+	{
						yylval.int_c = 
						ntohl(inet_addr(yytext));
						return(T_INT_LIT);
					}

{alpha}{alphanum}*		{ yylval.ident_c = yytext; 
				  return_token(T_ID); }

"\n"				{ lineup(); }
{white}*			;

{any}				{ return_token(yytext[0]); }

%%
