/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

static u_int32_t cidr_to_netmask(char *nm);

int IP_Initialized = 0;

/* Global CAPE IP engine state stuff -- all of it belongs here */

char *CASL_Default_Interface = NULL;
u_int32_t CASL_Default_Gateway = INADDR_NONE;
u_int32_t CASL_IP_Address = INADDR_NONE;
u_int32_t CASL_Netmask = INADDR_NONE;
struct ether_addr *CASL_Gateway_MAC = NULL;
struct ether_addr *CASL_Local_MAC = NULL;

extern int AlwaysArp;

int Fallback_to_Gateway = 1;
int No_Kernel_ARP = 0;

/* ----------------------------------------------------------------------------
** Internet Protocol checksum, accumulated version; takes value of previous
** checksum, or ACCUM_ZERO (0xffff) for new checksum, and performs checksum
** on "len" bytes of "buf"
*/

u_short
cksum_accum(u_short accum, void *buf, int len) {
	int count, sum;
	u_short *sp;

	sum 	= (u_short)~accum;
	sp 	= (u_short *)buf;

	count = len / 2;

	while(count--) 
		sum += *sp++;

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum >> 16);

	return(~sum);
}

/* ----------------------------------------------------------------------------
** Take a list of asr nodes, and turn it into a buffer.  Return the length.
*/

static int iovlist(asr_t *list, struct iovec *ivp, struct iovec *end);

int list_coagulate(asr_t *list, u_char **buf) {
	struct iovec iovs[CAPE_MAXIOVS];
	int i, l;
	
	i = iovlist(list, iovs, &iovs[CAPE_MAXIOVS - 1]);
	*buf = coagulate(iovs, i, &l);
	return(l);	
}

/* ----------------------------------------------------------------------------
** Take a list of asr_nodes and turn it into an iovec list of it's members.
** The members are expanded recursively.
*/

static int iovlist(asr_t *list, struct iovec *ivp, struct iovec *end) {
	asr_t *ide;

	int i = 0;

	for(; list; list = list->asr_kids[1]) {
		if(&ivp[i] == end)
			error(E_USER, "Too many elements in list");

		if(list->asr_kids[0]->asr_type == N_ID)
			ide = eval_id(list->asr_kids[0]);
		else
			ide = list->asr_kids[0];

		/* handle arbitrarily nested lists */

		if(ide->asr_type == LIST) {
			i += iovlist(ide, &ivp[i], end);
			continue;
		}

		ivp[i].iov_len = asr_tobuf(ide, (u_char **) &ivp[i].iov_base);
		i += 1;
	}

	return(i);
}

/* ----------------------------------------------------------------------------
** return a buffer containing all the data from the array of 'n' iovecs at
** 'iovs', returning the total length of this data through 'l'.
*/

u_char *coagulate(struct iovec *iovs, int n, int *l) {
	u_char *Buf = NULL;
	int Bufsiz;

	int i, c;

	if(!Buf) {
		Bufsiz = CAPEBUFSIZ;
		Buf = xcalloc(1, Bufsiz);
	}

	for(i = c = 0; i < n; i++) {
		c += iovs[i].iov_len;
		if(c > Bufsiz) {
			u_char *nb = xcalloc(1, (c * 2));
			u_char *sb = Buf;
			Buf = nb;

			memcpy(Buf, sb, Bufsiz);

			Bufsiz = (c * 2);

			free(sb);
		}

		memcpy(Buf + (c - iovs[i].iov_len), iovs[i].iov_base, iovs[i].iov_len);
	}

	*l = c;
	return(Buf);
}


/* ----------------------------------------------------------------------------
** Perform Internet checksum on CASL data element, referenced by list element
** 2, and accumulate using list element 2, returning the current accumulated
** checksum as an integer.
*/

asr_t *casl_checksum(asr_t *list) {
	asr_t *buf;
	asr_t *accum;
	asr_t *ap = 0;
	u_char *p;
	u_short a;
	int i;

	if(!list || !list->asr_kids[1]) 
		error(E_USER, "too few arguments for checksum()");  

	if (list->asr_kids[1]->asr_kids[1]) {
		error (E_USER, "too many arguments for checksum()");
	}

	accum = list->asr_kids[0];
	buf = list->asr_kids[1]->asr_kids[0];

	i = asr_getint(accum);
	if(i < 0) 
		error(E_USER, "Unable to accumulate into non-numeric value (bad arg 1)");

	a = i;

	alloc_upref (buf);
	if (!asr_basetype(buf)) {
		buf = eval_expr (buf);
	}

	/* i wrote this? */
	/* you wrote the rest of it, too */

	i = asr_tobuf(asr_basetype(buf) ? buf : (ap = eval_expr(buf)), &p);
	if(i < 0) 
		error(E_USER, "Unable to evaluate buffer as raw data (bad arg 2)");

	alloc_downref (buf);

	a = cksum_accum(a, p, i);

	return(asr_int(a, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the casl ip_range() function.
*/

asr_t *casl_ip_range(asr_t *list) {
	asr_t *range;
	asr_t *head;
	asr_t **lp = &head;
	list_t *a, *c; 
	char *str;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for ip_range()");

	if (list->asr_kids[1]) {
		error (E_USER, "too many arguments for ip_range()");
	}

	range = list->asr_kids[0];

	if(range->asr_type != N_BUFFER)
		error(E_USER, "Argument to ip_range() must be a string.\n");

	str = buf2asciiz (range);
	a = inet_addrs(str);
	xfree ((void **)&str);
	if(!a)
		error(E_USER, "Malformed argument to ip_range()\n");

	for(c = a; c; c = c->next) {
		u_int32_t addr = (*(u_int32_t *)c->data);
		asr_t *n = asr_int(ntohl(addr), 0);
		*lp = asr_node(LIST, NULL, NULL);
		(*lp)->asr_kids[0] = n;
		lp = &(*lp)->asr_kids[1];
	}

	for(c = a; c; c = c->next)
		xfree((void **)&c->data);

	l_free(&a);

	return(head);
}

/* ----------------------------------------------------------------------------
** Builtin hook for the casl ip_filter() function.
*/

asr_t *casl_ip_filter(asr_t *list) {
	char *filter;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for ip_filter");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for ip_filter");

	filter = list_to_string(list->asr_kids[0]);

	if(!IP_Initialized) {
		ip_init(NULL);
		IP_Initialized = 1;;
	}

	ip_filter(filter);

	return(asr_int(0, 0));
}

/* ----------------------------------------------------------------------------
** Initialize the IP code.
*/

extern int Print_Status;
extern void print_stat(void);

void ip_init(void *v) {
	char ebuf[256];
	char *interface;
	u_int32_t address;
	u_int32_t dest = 0;
	char *d;

	if((d = getenv(CC_DESTINATION))) 
		dest = resolve(d);

	/* GET THE LOCAL INTERFACE */

	if(CASL_Default_Interface)
		interface = CASL_Default_Interface;
	else {
		interface = lookup_string(CC_INTERFACE);

		if(!strcasecmp(interface, CC_AUTO)) {
			struct iflist *ifp = NULL;
			asr_t *gg = NULL;

			if(route_lookup(dest, &address, &ifp) < 0) 
				error(E_USER,
				"Unable to automatically determine "
				"local interface name.");

			gg = asr_string(ifp->name, 0);

			xfree ((void **)&interface);

			st_replace(CC_INTERFACE, gg, 0);		

			interface = xstrdup (ifp->name);	
		}
		CASL_Default_Interface = interface;
	}

	/* GET THE ADDRESS OF THAT INTERFACE */

	address = get_address_from_interface(interface);
	CASL_IP_Address = address;

	/* OPEN THE INTERFACE */

	Packet_Context = pcap_open_live(interface, 1500, 1, 100, ebuf);
	if(!Packet_Context) 
		error(E_USER, "Error accessing interface %s: %s.\n", 
		      interface, ebuf);

	/* GET THE DEFAULT GATEWAY */

	if(CASL_Default_Gateway == INADDR_NONE) {
		asr_t *gg = st_get(CC_GATEWAY, 0);
		if(!gg) 
			error(E_USER, "CC_Gateway unset in config file.");

		if(gg->asr_type == N_INT || gg->asr_type == N_NEGINT)
			CASL_Default_Gateway = htonl(lookup_num(CC_GATEWAY));
		else 
			special_cc_gateway(gg);

	}

	/* GET THE NETMASK */

	if(CASL_Netmask == INADDR_NONE) {
		asr_t *a = st_get(CC_NETMASK, 0);
		if(!a)
			error(E_USER, "Undefined identifier \"%s\"", CC_NETMASK);

		a = lookup(a);
		if(a->asr_type == N_BUFFER) {
			char *str = buf2asciiz (a);
			CASL_Netmask = cidr_to_netmask(str);
			xfree ((void **)&str);
			if(CASL_Netmask == INADDR_NONE)
				error(E_USER, "Malformed CIDR netmask");
		} else 
			CASL_Netmask = htonl(asr_getint(a));
	}

	/* GET THE GATEWAY MAC ADDRESS */

	if(!CASL_Gateway_MAC) {
       		char *gwm = lookup_string(CC_GATEWAY_MAC);

		if(!strcasecmp(CC_GATEWAY_MAC_NONE, gwm) ||
		   getenv(CC_GATEWAY_MAC_NONE_ENV)) {
			u_char b[6] = { 0,0,0,0,0,0 };
			
			CASL_Gateway_MAC = (struct ether_addr *)
				xbufdup(b, ETHER_ADDR_LEN);
		} else if(!strcasecmp(CC_GATEWAY_MAC_AUTO, gwm)) {
			u_char *b = casl_arp_resolve(CASL_Default_Gateway);
			if(!b) 
			       error(E_USER, "Unable to resolve gateway MAC address"
				     " (%s)\n", addr_ntoa(CASL_Default_Gateway));
			
			CASL_Gateway_MAC = (struct ether_addr *) 
				xbufdup(b, ETHER_ADDR_LEN);
		} else {
			CASL_Gateway_MAC = ether_aton(gwm);
			if(!CASL_Gateway_MAC)
				error(E_USER, "Malformed MAC address (%s) for gateway",
				      gwm);

			CASL_Gateway_MAC = (struct ether_addr *)
				xbufdup((u_char *) CASL_Gateway_MAC,
						   ETHER_ADDR_LEN);
		}
		xfree ((void **)&gwm);
	}

	/* GET THE LOCAL MAC ADDRESS */

	if(!CASL_Local_MAC) {
		char *lm = lookup_string(CC_LOCAL_MAC);
		if(!strcasecmp(CC_LOCAL_MAC_AUTO, lm)) {

			/* XXX */

			u_char *buf = xcalloc(1, ETHER_ADDR_LEN);
			CASL_Local_MAC = (struct ether_addr *) xbufdup(buf, 
							  ETHER_ADDR_LEN);
		} else {
			CASL_Local_MAC = ether_aton(lm);
			if(!CASL_Local_MAC) 
				error(E_USER, "Malformed MAC address (%s) for local",
				      lm);
			
			CASL_Local_MAC = (struct ether_addr *)
				xbufdup((u_char *) CASL_Local_MAC, 
					ETHER_ADDR_LEN);
		}
		xfree ((void **)&lm);
	}

	AlwaysArp = lookup_num(CC_ALWAYSARP);
	
	return;
}

/* ----------------------------------------------------------------------------
** Get an IP address for a named interface.
*/

u_int32_t get_address_from_interface(char *name) {
	struct iflist *list;

	for (list = get_iflist (); list; list = list->next) {
		if (!strcmp (name, list->name)) {
			return (list->ipaddr);
		}
	}

	return (INADDR_NONE);
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_Interface.
*/

void special_cc_interface(asr_t *a) {
	struct iflist *ifp;
	char *str;

       	a = lookup(a);

	if(a->asr_type != N_BUFFER)
		error(E_USER, "CC_Interface must be a string");

	str = buf2asciiz (a);

	if(!strcasecmp(str, CC_AUTO)) {
		if(iface_lookup(0, &ifp) < 0) 
			error(E_USER,
				"Unable to automatically determine "
				"gateway IP address.");

		a = asr_string(ifp->name, 0);
	}

	st_replace(CC_INTERFACE, a, 0);

	if (CASL_Default_Interface) {
		xfree ((void **)&CASL_Default_Interface);
	}
	CASL_Default_Interface = str;

	if(Packet_Context) 
		pcap_close(Packet_Context);
	
	ip_init(NULL);

	return;
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_Gateway.
*/

void special_cc_gateway(asr_t *a) {
	u_int32_t dest = 0;
	char *d;
	a = lookup(a);
	
	if((d = getenv(CC_DESTINATION))) 
		dest = resolve(d);

	if(a->asr_type == N_BUFFER) {
		u_int32_t ip;
		struct iflist *ifp;
		char *str = buf2asciiz (a);

		if(!strcasecmp(str, CC_AUTO)) {
			if(route_lookup(dest, &ip, &ifp) < 0) 
				error(E_USER,
					"Unable to automatically determine "
					"gateway IP address.");

			a = asr_int(htonl(ip), 0);

			st_replace(CC_GATEWAY, a, 0);

			CASL_Default_Gateway = ip;
			xfree ((void **)&str);
			return;
		}			
		xfree ((void **)&str);
	}

	if(a->asr_type != N_INT && a->asr_type != N_NEGINT)
		error(E_USER, "CC_Gateway must be an IP address or \"auto\"");

	st_replace(CC_GATEWAY, a, 0);

	CASL_Default_Gateway = htonl(a->asr_integer);

	return;
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_Gateway_MAC.
*/

void special_cc_gateway_mac(asr_t *a) {
	char *str;
	a = lookup(a);
	if(a->asr_type != N_BUFFER)
		error(E_USER, "CC_Gateway_MAC must be a string");

	st_replace(CC_GATEWAY_MAC, a, 0);

	if(CASL_Gateway_MAC)
		xfree((void **)&CASL_Gateway_MAC);

	str = buf2asciiz(a);
	CASL_Gateway_MAC = ether_aton(str);
	xfree ((void **)&str);
	if(!CASL_Gateway_MAC)
		error(E_USER, "Malformed MAC address for gateway");

	CASL_Gateway_MAC = (struct ether_addr *)
		xbufdup((u_char *)CASL_Gateway_MAC, ETHER_ADDR_LEN);

	return;
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_Local_MAC.
*/

void special_cc_local_mac(asr_t *a) {
	char *str;
	a = lookup(a);
	if(a->asr_type != N_BUFFER)
		error(E_USER, "CC_Local_MAC must be a string");

	st_replace(CC_LOCAL_MAC, a, 0);

	if(CASL_Local_MAC)
		xfree((void **)&CASL_Local_MAC);

	str = buf2asciiz (a);
	CASL_Local_MAC = ether_aton(str);
	xfree ((void **)&str);
	if(!CASL_Local_MAC)
		error(E_USER, "Malformed MAC address for local");
	
	CASL_Local_MAC = (struct ether_addr *)
		xbufdup((u_char *)CASL_Local_MAC, ETHER_ADDR_LEN);

	return;
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_Netmask.
*/

void special_cc_netmask(asr_t *a) {
	u_int32_t m;

	a = lookup(a);
	if(a->asr_type == N_BUFFER) {
		char *str = buf2asciiz (a);
		m = cidr_to_netmask(str);
		xfree ((void **)&str);

		if(m == INADDR_NONE)
			error(E_USER, "Malformed CIDR Netmask");
	} else 
		m = asr_getint(a);

	CASL_Netmask = htonl(m);

	a = asr_int(m, 0);
	st_replace(CC_NETMASK, a, 0);

	return;
}
       	
/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_AlwaysArp.
*/

void special_cc_alwaysarp(asr_t *a) {

	AlwaysArp = asr_getint(a);

	a = asr_int(AlwaysArp, 0);

	st_replace(CC_ALWAYSARP, a, 0);

	return;
}

/* ----------------------------------------------------------------------------
** Handle setting the special variable CC_NoKernalArp.
*/

void special_cc_nokernelarp(asr_t *a) {

	No_Kernel_ARP = asr_getint(a);

	a = asr_int(No_Kernel_ARP, 0);

	st_replace(CC_NOKERNELARP, a, 0);
}

/* ----------------------------------------------------------------------------
** Handle setting special variable CC_ArpFallBack.
*/

void special_cc_arpfallback(asr_t *a) {
	
	Fallback_to_Gateway = asr_getint(a);

	a = asr_int(No_Kernel_ARP, 0);

	st_replace(CC_ARPFALLBACK, a, 0);
}

/* ----------------------------------------------------------------------------
** Return if an IP address is on a network based on an IP on that network and
** the netmask.
*/

int isinnetwork(u_int32_t addr, u_int32_t network, u_int32_t mask) {
	u_int32_t base, top;
        
	base = network & mask;
	top = base + ~(mask);

	base = ntohl(base);
	top  = ntohl(top);
	addr = ntohl(addr);

	if(addr < base || addr > top)
		return(0);

	return(1);
}

/* ----------------------------------------------------------------------------
** Take a CIDR network specification (IE change /29 to 255.255.255.248).
*/

static u_int32_t cidr_to_netmask(char *nm) {
        	u_int32_t base = 0xFFFFFFFF;
	int bits;

	if(*(nm++) != '/')
		return(INADDR_NONE);

	bits = atoi(nm);
	if(!bits || bits > 32)
		return(0);
        
	bits = 32 - bits;
	if(!bits)
		return(0xFFFFFFFF);
        
	base <<= bits;
        
	return(ntohl(base));
}


