/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

/*
 * inet_addrs() - parse a range of IP addresses with wildcards and 
 * 		  CIDR netmasks into a list of all component addresses
 */

#define NUL '\0';

list_t *inet_addrs(char *addrstring);

static list_t *explode(char *addrstring);
static list_t *explode_numbers(char *string);
static list_t *explode_cidr(char *string);

/* ----------------------------------------------------------------------------
** Parse a string into a list of IP addresses.
*/

list_t *inet_addrs(char *addrstring) {
	list_t *addresses;
	list_t *node;
	char *cp;

	list_t *a = NULL;

	if(!strchr(addrstring, '.'))
		return(NULL);

	while((cp = strsep(&addrstring, ";"))) {
		int i = 0;
		char *ccp;

		if(!strlen(cp)) 
			continue;

		for(ccp = cp; *ccp; ccp++)
			if(*ccp == '.') 
				i++;

		if(i > 3)
			return(NULL);

		if(strchr((char *) cp, '/')) 
			addresses = explode_cidr(cp);
		else
			addresses = explode(cp);

		for(node = addresses; node; node = node->next) 
			a = l_push(a, node->data);

		l_free(&addresses);
	}

	return(a);
} 	

/* ----------------------------------------------------------------------------
** take a 1.2.3-4.* spec and return all possible ulong addresses
*/

static list_t *explode(char *addrstring) {
	char *cp;
	list_t *numbers[4];	   /* one octet each 	   */
	list_t *n[4];
	list_t *addresses = NULL;  /* ulong IP addresses   */
	list_t *node;

	int i;

	for(i = 0; i < 4; i++)
		numbers[i] = NULL;

	if(!addrstring || !strlen(addrstring)) 
		return(NULL);

	i = 0;

	while((cp = strsep(&addrstring, "."))) {
		numbers[i] = explode_numbers(cp);

		if(!numbers[i]) 
			return(NULL);;

		i++;
	}

	/* blank octets are wildcarded. 206.54.252 = 206.54.252.* */

	for(i = 0; i < 4; i++) {
		if(!numbers[i]) 
			numbers[i] = explode_numbers("*");

		numbers[i] = l_reverse(numbers[i]);
	}

	/* weeeeeeee */

	for(n[0] = numbers[0]; 
	    n[0]; 
            n[0] = n[0]->next) {
	   for(n[1] = numbers[1]; 
	       n[1]; 
	       n[1] = n[1]->next) {
	      for(n[2] = numbers[2];
	          n[2]; 
	          n[2] = n[2]->next) { 
	         for(n[3] = numbers[3];
		     n[3]; 
		     n[3] = n[3]->next) { 

			/* whew; we're now working with a specific
		 	 * IP address in our set, represented as 
			 * an array of 4 characters.
			 */

			unsigned char octs[4];

			/* mallocing each address is silly. But then
			 * again, so is the rest of this code. 
			 */

			unsigned long *xad = calloc(1, sizeof(unsigned long));

			octs[0] = (unsigned char)(*(int *) n[0]->data);
			octs[1] = (unsigned char)(*(int *) n[1]->data);
			octs[2] = (unsigned char)(*(int *) n[2]->data);
			octs[3] = (unsigned char)(*(int *) n[3]->data);

			/* build IP address */

			*xad = (octs[0] << 24)
			     + (octs[1] << 16)
			     + (octs[2] << 8)
			     +  octs[3];

			/* save it for later */

			*xad = htonl(*xad);

			addresses = l_push(addresses, xad);
		      }
	         } 
	    } 
	}	

	/* throw out our leftovers */

	for(i = 0; i < 4; i++) {
		for(node = numbers[i]; node; node = node->next)
			free(node->data);

		l_free(&numbers[i]);
	}

	/* return the addresses */

	return(addresses);
}

/* ----------------------------------------------------------------------------
** Do the work for the cases of * and -
*/

static list_t *explode_numbers(char *string) {
	int begin;
	int end;
	char *cp;
	char *ep = NULL;
	int *number;
	int i;

	list_t *numbers = NULL;

	/* ignore empty strings */

	if(!string)
		return(NULL);

	for(cp = string; *cp && isspace((int)*cp); cp++)
		/* do nothing */;

	if(!*cp)
		return(NULL);

	string = cp;

	if(strchr(string, ',')) {
		while((cp = strsep(&string, ","))) {
			list_t *l = explode_numbers(cp);
			if(l)
				numbers = l_append(numbers, l);
		}

		return(numbers);
	}		

	/* expand '-'-denoted ranges */

	if((cp = strchr(string, '-'))) {
		*cp++ = '\0';

		begin = (int) strtoul(string, &ep, 10);
		end   = (int) strtoul(cp, &ep, 10);

	/* expand '*' wildcards */

	} else if((cp = strchr(string, '*'))) {
		begin = 0;
		end   = 255;

	/* otherwise, treat the token as an individual number */

	} else {
		begin = (int) strtoul(string, &ep, 10);
		end   = (int) strtoul(string, &ep, 10);
	}

	if(ep && *ep) 
		error(E_USER, "Invalid range passed to ip_range()");

	if(begin > 255 ||
	   end   > 255 ||
	   begin < 0   ||
	   end   < 0   ||
	   begin > end)
		return(NULL);

	/* ranges and wildcards give us an upper and lower bound
 	 * on which numbers we're using; expand this out to a list
  	 * of all those numbers 
 	 */

	for(i = begin; i <= end; i++) {
		number  = (int *) calloc(1, sizeof(int));
		*number = i;			
		numbers = l_push(numbers, number);
	}

	return(numbers);
}

/* ----------------------------------------------------------------------------
** take a CIDR address specification (a.b.c.d/netbits) and
** return a list of all possible addresses in that network.
** Thanks, al.
*/

static list_t *explode_cidr(char *string) {
	list_t *addrs = NULL;

    	u_long ip, base, mask, top;
    	int bits;
   	char *cp;
	
	if(!string) 
		return(NULL);

	/* recover netmask bits */

	if(!(cp = strchr(string, '/')))
		bits = 24;
	else {
		*cp++ = '\0';
		bits = atoi(cp);
	}

	/* build IP address */

	ip = ntohl(inet_addr(string));
    
	/* we specify the number of net bits, want host bits */

	bits = 32 - bits;

	/* shift off host bits */

	mask = ~((1 << bits) - 1);

	/* blot out host bits from specified IP address */

	base = ip & mask;
    	top  = base + ~mask;

	/* explode */

	for(ip = base; ip <= top; ip++) { 
		u_long *b = calloc(1, sizeof(u_long));
		*b = htonl(ip);

		addrs = l_push(addrs, b);
	}

	return(addrs);
}

