/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

/*
 * This code is the output half of CAPE, which is the packet generation extension for
 * CASL. 
 */

extern int AlwaysArp;

extern int pcap_write(pcap_t *, u_char *, int);

static int ip_output(asr_t *olist);

static void fixup(u_char *p, int i);
static u_char *ethernet_frame(int *len);
static int ip_packet_type(u_char *p, int i);

static void fix_ip_checksum(u_char *p, int i);
static void fix_ip_length(u_char *p, int i);
static void fix_udp_checksum(u_char *p, size_t i);
static void fix_udp_length(u_char *p, size_t i);
static void fix_tcp_checksum(u_char *p, size_t i);
static void fix_tcp_offset(u_char *p, size_t i);
static void fix_icmp_checksum(u_char *p, size_t i);

static void fix_ether_frame(u_char *ether, u_char *ip);

pcap_t *Packet_Context = NULL;

struct psuedo {
       	u_int32_t src;
	u_int32_t dst;
	u_char zero;
	u_char proto;
	u_int16_t len;
};

static u_short cksum_psuedo(u_short *addr, int len, struct psuedo *ph);
static u_short in_cksum(u_short *addr, int len);

/* ---------------------------------------------------------------------------
** CASL builtin stub for ip output routine, does almost nothing except for
** ensuring that the IP stuff is initialized before continuing.
*/

asr_t *casl_ip_output(asr_t *list) {
	if(!list || !list->asr_kids[0]) 
		error(E_USER, "too few arguments for ip_output()");

	if (list->asr_kids[1])
		error (E_USER, "too many arguments for ip_output()");

       	if(!IP_Initialized) {
		ip_init(NULL);
		IP_Initialized = 1;;
	}

	return(asr_int(ip_output(list), 0));
}

/* ---------------------------------------------------------------------------
** Output an IP packet, passed in as a list of data elements. Prepends the 
** Ethernet frame header, and optionally fixes protocol lengths and 
** checksums.
*/

static int ip_output(asr_t *olist) {
	int i = 0, j;
	asr_t *list;
	asr_t *buf;
	u_char *up;

	/* prepend Ethernet frame header */

	up = ethernet_frame(&i);

	buf = asr_buffer(i, "ethernet");
	memcpy(buf->asr_buf, up, i);

	list = olist->asr_kids[0];

	if(list->asr_type != LIST) {
		list = asr_node(LIST, buf, asr_node(LIST, list, NULL));      
		j = 1;
	} else {
		list = asr_node(LIST, buf, list);
		j = 0;
	}

	i = list_coagulate(list, &up);

	/* make sure things like checksums match */
	
	fixup(up, i);

	/* write it to the network */
	
	i = pcap_write(Packet_Context, (up + 2), (i - 2));

	/* clean up */

	if(j) {
		asr_sfree(&list->asr_kids[1]);
	}

	asr_sfree(&list);
	asr_sfree(&buf);

	return(i);

}

/* ---------------------------------------------------------------------------
** Where protocol-specific fixups go to die. The convention for fixups here
** is that they can all be explicitly turned off by unsetting a global variable, 
** and are skipped if the value they're setting is already present (assuming
** then that the user set the value that way on purpose).
*/

static void fixup(u_char *p, int i) {
	u_char *etherframe;
	struct ip *ih;

	if(!lookup_num("Fixups") || i < ETHER_FRAME_SIZE + IP_HEADER_SIZE)
		return;

	etherframe = p + 2;

	p  += ETHER_FRAME_SIZE + 2;
	i  -= ETHER_FRAME_SIZE + 2;

       	fix_ip_length(p, i);
       	fix_ip_checksum(p, i);

	switch(ip_packet_type(p, i)) {
	case IPPROTO_UDP:
		fix_udp_length(p, i);
		fix_udp_checksum(p, i);
		break;

	case IPPROTO_TCP:
		fix_tcp_offset(p, i);
		fix_tcp_checksum(p, i);
		break;

	case IPPROTO_ICMP:
		fix_icmp_checksum(p, i);
		break;

	default:
		break;
	}

	ih = (struct ip *) p;

	if(AlwaysArp 
	|| isinnetwork(ih->ip_dst.s_addr, CASL_IP_Address, CASL_Netmask))
 		fix_ether_frame(etherframe, p);

	return;
}

/* ---------------------------------------------------------------------------
** Return an ethernet header, sending this frame to the gateway with our
** real source address. Forging source address is impossible on some archs, 
** directing frames locally is tricky and usually not necessary, but
** XXX should be fixed at some point.
**
** I'm prepending 2 bytes to the ethernet frame to combat alignment
** problem on SPARC.
*/


static u_char *ethernet_frame(int *len) {
	static u_char frame[16];
	static u_char buf[ETHER_FRAME_SIZE];
	struct ether_header *eh = (struct ether_header *) buf;
	u_char *p = frame;
	
   	memcpy(eh->ether_shost, CASL_Local_MAC, ETHER_ADDR_LEN);
  	memcpy(eh->ether_dhost, CASL_Gateway_MAC, ETHER_ADDR_LEN);

	eh->ether_type = htons(ETHERTYPE_IP);

	*len = ETHER_FRAME_SIZE + 2;
	
	memcpy(p + 2, buf, ETHER_FRAME_SIZE);

	return frame;
}

/* ---------------------------------------------------------------------------
** Fix the IP checksum.
*/

static void fix_ip_checksum(u_char *p, int i) {
	struct ip *ih = (struct ip *) p;
	int l = 0;

	if(!lookup_num("Fixup_IP_Checksum") || ih->ip_sum)
		return;

	if(ih->ip_hl)
		l = ih->ip_hl << 2;
	else
		l = sizeof(*ih);

	ih->ip_sum = cksum_accum(ACCUM_ZERO, p, l);
}

/* ---------------------------------------------------------------------------
** Fix the IP length.
*/

static void fix_ip_length(u_char *p, int i) {
	struct ip *ih = (struct ip *) p;

	if(!lookup_num("Fixup_IP_Length") || ih->ip_len)
		return;

	ih->ip_len = htons(i);
       
	if(!ih->ip_hl)
		ih->ip_hl = sizeof(*ih) >> 2;

	return;
}

/* ---------------------------------------------------------------------------
** Find the packet type.
*/

static int ip_packet_type(u_char *p, int i) {
	struct ip *ih = (struct ip *) p;

	/*
         * If the offset field is set then the packet is not
         * fixed up .. ie. no checksum or length. This still allows
         * either or/and IP_DF, IP_RF set. 
         */ 
        if(ih->ip_off && (htons(ih->ip_off) & (IP_MF | IP_RF)))
                return(-1);

	return(ih->ip_p);
}

/* ---------------------------------------------------------------------------
** Fixup the UDP checksum.
*/

static void fix_udp_checksum(u_char *p, size_t i) {
	struct ip *ih = (struct ip *) p;
	struct udphdr *uh;
	struct psuedo ph;
	size_t l;

	if(i < sizeof(struct ip) + sizeof(struct udphdr))
		return;

	l = ih->ip_hl << 2;
	if(l > i)
		return;

	uh = (struct udphdr *) (p + l);

	if(!lookup_num("Fixup_UDP_Checksum") || uh->uh_sum)
		return;

	ph.src = ih->ip_src.s_addr;
	ph.dst = ih->ip_dst.s_addr;
	ph.proto = ih->ip_p;
	ph.len = htons(i - l);
	ph.zero = 0;

	uh->uh_sum = cksum_psuedo((u_short *) uh, i - l, &ph);

	return;
}

/* ---------------------------------------------------------------------------
** Fixup UDP length.
*/

static void fix_udp_length(u_char *p, size_t i) {
	struct ip *ih = (struct ip *) p;
	struct udphdr *uh;
	size_t l;

	if(i < sizeof(struct ip) + sizeof(struct udphdr))
		return;

	l = ih->ip_hl << 2;
	if(l > i)
		return;

	uh = (struct udphdr *) (p + l);

	if(!lookup_num("Fixup_UDP_Length") || uh->uh_ulen)
		return;

	uh->uh_ulen = htons(i - l);

	return;
}

/* ---------------------------------------------------------------------------
** Fixup the TCP check sum.
*/

static void fix_tcp_checksum(u_char *p, size_t i) {
	struct ip *ih = (struct ip *) p;
	struct tcphdr *th;
	struct psuedo ph;
	size_t l;

	if(i < sizeof(struct ip) + sizeof(struct tcphdr))
		return;

	l = ih->ip_hl << 2;
	if(l > i)
		return;

	th = (struct tcphdr *) (p + l);

	if(!lookup_num("Fixup_TCP_Checksum") || th->th_sum)
		return;

	ph.src = ih->ip_src.s_addr;
	ph.dst = ih->ip_dst.s_addr;
	ph.proto = ih->ip_p;
	ph.len = htons(i - l);
	ph.zero = 0;

	th->th_sum = cksum_psuedo((u_short *) th, i - l, &ph);

	return;
}

/* ---------------------------------------------------------------------------
** Fixup the TCP offset.
*/

static void fix_tcp_offset(u_char *p, size_t i) {
	struct ip *ih = (struct ip *) p;
	struct tcphdr *th;
	size_t l;

	if(i < sizeof(struct ip) ||
  	   i < (ih->ip_hl * 4) + sizeof(struct tcphdr))
		return;

	l = ih->ip_hl << 2;
	if(l > i)
		return;

	th = (struct tcphdr *) (p + l);

	if(!lookup_num("Fixup_TCP_Offset") || th->th_off)
		return;

	th->th_off = sizeof(*th) >> 2;

	return;
}

/* ---------------------------------------------------------------------------
** Fixup the ICMP checksum.
*/

static void fix_icmp_checksum(u_char *p, size_t i) {
	struct ip *ih = (struct ip *) p;
	size_t l, s;
	u_char *sum;

	if(i < sizeof(struct ip) ||
  	   i < ((size_t) (ih->ip_hl * 4) + 4))
		return;

	l = ih->ip_hl << 2;
	if(l > i)
		return;

	s = ntohs(ih->ip_len);
	if(s > i)
		return;

	p += l;
	sum = p + 2;
	(*(u_short *) sum) = in_cksum((u_short *) p, s - l);
	
	return;
}      

/* ---------------------------------------------------------------------------
** checksum a packet in-place, but include the IP psuedo-header in the checksum.
*/

static u_short
cksum_psuedo(u_short *addr, int len, struct psuedo *ph) {
	register int nleft = len;
	register u_short *w;
	register int sum = 0;
	u_short answer = 0;
	int i;

	w = (u_short *) ph;
	for(i = 0; i < 6; i++) 
		sum += *w++;

	w = addr;
	while(nleft > 1) {
		sum += *w++;
		nleft -= 2;
	}

	if(nleft) {
		*(u_char *)(&answer) = *(u_char *)w;
		sum += answer;
	}

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum >> 16);
	answer = ~sum;
	return(answer);
}

/* ------------------------------------------------------------------- */

static u_short in_cksum(u_short *addr, int len) {
	register int nleft = len;
	register u_short *w = addr;
	register int sum = 0;
	u_short answer = 0;

	while(nleft > 1) {
		sum += *w++;
		nleft -= 2;
	}

	if(nleft) {
		*(u_char *)(&answer) = *(u_char *)w;
		sum += answer;
	}

	/* fold 32 bits into 16 */

	sum = (sum >> 16) + (sum & 0xffff);
	sum += (sum >> 16);

	/* 1's complement of result */

	answer = ~sum;

	return(answer);
}

/* ---------------------------------------------------------------------------
** do explicit ARP for this host, ignore gateway
*/

static void fix_ether_frame(u_char *ether, u_char *ip) {
	struct ether_header *eh = (struct ether_header *) ether;
	struct ip *ih = (struct ip *) ip;
	u_char *arp;

	arp = casl_arp_resolve(ih->ip_dst.s_addr);
	if(!arp) 
		return;		

	memcpy(eh->ether_dhost, arp, ETHER_ADDR_LEN);

	return;
}

/* ---------------------------------------------------------------------------
** explicitly "fix up" an IP packet - useful for IP fragmentation, when we
** might want to create a completely valid packet before breaking it into frags
*/

asr_t *casl_ip_fixup(asr_t *buf) {
	asr_t *ether;
	u_char *up;
	int l, i, extranode;

       	if(!IP_Initialized) {
		ip_init(NULL);
		IP_Initialized = 1;
	}

	if(!buf || !buf->asr_kids[0])
		error(E_USER, "too few arguments for ip_fixup()");
	if (buf->asr_kids[1]) 
		error (E_USER, "too many arguments for ip_fixup()");

	buf = buf->asr_kids[0];

	up = ethernet_frame(&i);
	ether = asr_buffer(i, "ethernet");
	memcpy(ether->asr_buf, up, i);

	if(buf->asr_type != LIST) {
		buf = asr_node(LIST, ether, asr_node(LIST, buf, NULL));
		extranode = 1;
	} else {
		buf = asr_node(LIST, ether, buf);
		extranode = 0;
	}

	l = list_coagulate(buf, &up);

	if (extranode) {
		asr_sfree (&buf->asr_kids[1]);
	}
	asr_sfree (&buf->asr_kids[0]);

	fixup(up, l);
	
	l -= i;
	up += i;

	ether = asr_buffer(l, "INTERNAL");
	memcpy(ether->asr_buf, up, l);

	return(ether);
}

/* ------------------------------------------------------------------- */

