/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

int ip_filter(char *filter);
asr_t *casl_ip_input(asr_t *list);

static int ip_input(u_char **buf, struct bpf_program *bf, int ms);
static void read_packet(u_char *arg1, const struct pcap_pkthdr *ph, const u_char *pd);
static void timeout_off(void);
static void timeout_on(int ms);
static struct bpf_program *compile_filter(char *filter);

struct capiov {
	u_char *iov_base;
	int iov_len;
	int iov_wirelen;
};

/* ---------------------------------------------------------------------------
** This code is the input portion of CAPE, the packet generation extension
** of CASL.
*/

asr_t *casl_ip_input(asr_t *list) {
	struct bpf_program *bf = 0;
	
	asr_t *timeout;
	asr_t *filter = NULL;
	asr_t *ret;

	int ms = 0;
	char *string = NULL;

	u_char *buf = NULL;
	int len;

	if(list) {
		timeout = list->asr_kids[0];
		if(timeout) 
			ms = asr_getint_strict(timeout);
		
		if(list->asr_kids[1]) {
			filter = list->asr_kids[1]->asr_kids[0];
			if(filter)
				string = list_to_string(filter);
		}
		if (list->asr_kids[1]->asr_kids[1]) {
			error (E_USER, "too many arguments for ip_input()");
		}
	}

	if(!IP_Initialized) {
		ip_init(NULL);
		IP_Initialized = 1;	          
	}

	if(filter)
		bf = compile_filter(string);

	len = ip_input(&buf, filter ? bf : NULL, ms);
	if(!len)
		return(asr_int(0, 0));

	ret = asr_buffer(len, "IP Input Buffer");
	memcpy(ret->asr_buf, buf, len);

	return(ret);
}
	
/* ---------------------------------------------------------------------------
** Takes a filter and compiles it to bpf byte-code.
*/

static struct bpf_program *compile_filter(char *filter) {
	static struct bpf_program bf;
	bpf_u_int32 mask;

	char errbuf[PCAP_ERRBUF_SIZE];
	bpf_u_int32 net;

	if(pcap_lookupnet(CASL_Default_Interface, &net, &mask, errbuf) < 0) 
		error(E_INTERNAL, "Unable to compile filter: %s", errbuf);

	assert(filter);

       	if(pcap_compile(Packet_Context, &bf, filter, 1, mask) < 0) 
		error(E_USER, "Unable to parse filter program"
		      ": %s. Problematic filter:\n"
		      "\"%s\"", pcap_geterr(Packet_Context), filter);	

	return(&bf);
}

/* ---------------------------------------------------------------------------
** Sets the global bpf filter.
*/

int ip_filter(char *filter) {
	struct bpf_program *bp;

	bp = compile_filter(filter);
	
	if(pcap_setfilter(Packet_Context, bp) < 0) 
		error(E_INTERNAL, "Unable to re-set interface filter for read: %s",
		      pcap_geterr(Packet_Context));

	return(0);
}

/* ---------------------------------------------------------------------------
** Read a packet.
*/

static jmp_buf Timeout_Kludge;
static void spatula_of_death(int sig);

static int ip_input(u_char **buf, struct bpf_program *bp, int ms) {
	static u_char Packet_Buffer[8192];
	struct capiov iv;

	if(sigsetjmp(Timeout_Kludge, 1)) {
		*buf = NULL;
		timeout_off();
		return(0);
	}

	timeout_on(ms);

	for(;;) {
		memset(&iv, 0, sizeof(iv));
		iv.iov_base = Packet_Buffer;

		if(pcap_dispatch(Packet_Context, 1, read_packet, (u_char *) &iv) != 1) {       
			*buf = NULL;
			timeout_off();
			return(0);
		}

		if(!bp)
			break;

		if(bpf_filter(bp->bf_insns, iv.iov_base, iv.iov_wirelen, iv.iov_len))
			break;
	}       

	timeout_off();

	*buf = Packet_Buffer + ETHER_FRAME_SIZE;

	return(iv.iov_len - ETHER_FRAME_SIZE);
}

/* ---------------------------------------------------------------------------
** Yeah.
*/

static void read_packet(u_char *arg1, const struct pcap_pkthdr *ph, const u_char *pd) {
	const u_char *packet = pd;
	int size = ph->caplen;
	struct capiov *iv = (struct capiov *) arg1;

	memcpy(iv->iov_base, packet, size);
	iv->iov_len = size;
	iv->iov_wirelen = ph->len;

	return;
} 

/* ---------------------------------------------------------------------------
** Turn on the timeout.  There has got to be a better way to do this.
*/

static void timeout_on(int ms) {
	struct itimerval it;

	extern int setitimer();

	it.it_value.tv_sec = ms / 1000;
	it.it_value.tv_usec = (ms * 1000) % 1000000;
	it.it_interval.tv_sec = 0;
	it.it_interval.tv_usec = 0;

	if(signal(SIGALRM, spatula_of_death) < 0) 
		error(E_INTERNAL, "Unable to set timeout handler: %s",
			strerror(errno));

	if(setitimer(ITIMER_REAL, &it, NULL) < 0)
		error(E_INTERNAL, "Unable to start timer: %s",
			strerror(errno));
	return;
}

/* ---------------------------------------------------------------------------
** Disable the timeout.
*/

static void timeout_off() {
	struct itimerval it; 

	extern int setitimer();

	memset(&it, 0, sizeof(it));

	setitimer(ITIMER_REAL, &it, NULL);

	return;
}

/* ---------------------------------------------------------------------------
** Aptly named, considering the use of longjump.
*/

static void spatula_of_death(int sig) {
	siglongjmp(Timeout_Kludge, 1);
	assert(0);
}
