/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"
#include <stdarg.h>

/* This file defines the actual structure of the parsed CASL script code.
 *
 * The YACC parser turns a CASL script into a collection of trees, one
 * for each function (including the global-scope function body). These
 * trees comprise the abstract syntax representation of a CASL program;
 * the interpreter walks the tree and acts on the contents of each node.
 * 
 * An abstract syntax representation tree (ASR) node consists of a 
 * a type and the data associated with it. 
 *
 * There are 3 kinds of ASR nodes in this code. Most nodes consist of 
 * a type and a variable number (depending on the type) of child nodes.
 * None of these nodes contain any data; their purpose is to control the
 * flow of program execution. 
 *
 * Next, we have leaf nodes. Leaf nodes don't contain child pointers;
 * their purpose is to encapsulate the data a CASL script is manipulating.
 * The interpreter code extracts and manipulates the data in a leaf
 * node depending on it's type.
 * 
 * Finally, there are "list" nodes. List nodes have two children, one
 * of which is (probably) a leaf node containing data, and one of which
 * is a pointer to the next list node. The first child is always the data
 * pointer, the second is always the next pointer.
 *
 * Leaf nodes are special-cased, because each type of node handles it's
 * data differently. All of the other nodes, including lists, are built
 * with a single function called "asr_node".
 */

/* allow us to look up a node and find out how many children it has */

extern struct arity arity_table[];

extern char *casl_file;

/* These definitions are node constructors for each base type. */

/* ----------------------------------------------------------------------------
** Return an ID (Variable reference) node.
*/

asr_t *asr_id(char *id, int line) {
	asr_t *ap = asr_new(0);

#ifdef TRACE_CALLS
	Dprintf("asr_id(%s, %d)\n", id, line);
#endif

	ap->asr_size	= sizeof(ap->asr_ident);
	ap->asr_type 	= N_ID;
	ap->asr_line	= line;
	ap->asr_ident	= xstrdup(id);
	assert(ap->asr_ident);

	return(ap);
}

/* ----------------------------------------------------------------------------
** Return a buffer node.
*/

asr_t *asr_buffer(int size, char *structtype) {
	asr_t *ap = asr_new(0);

#ifdef TRACE_CALLS
	Dprintf("asr_buffer(%d, %s)\n", size, structtype);
#endif

	ap->asr_size	= size;
	ap->asr_type 	= N_BUFFER;
	ap->asr_line	= 0;
	ap->asr_buf	= size ? (u_char *) xcalloc(1, size) : NULL;

	ap->asr_extra = structtype;

	return(ap);
}

/* ----------------------------------------------------------------------------
** Return the structure name attached to this buffer.
*/

char *asr_buffer_struct(asr_t *ap) {

#ifdef TRACE_CALLS
	Dprintf("asr_buffer_struct(%p)\n", ap);
#endif

	if(ap->asr_type != N_BUFFER)
		return(NULL);

	if(ap->asr_extra)
		return((char *) ap->asr_extra);

	return("Anonymous buffer");
}

/* ----------------------------------------------------------------------------
** Set the stucture name associated with this buffer.  The names are assumed
** to be static.
*/

void asr_buffer_struct_set(asr_t *ap, char *name) {
	assert(ap->asr_type == N_BUFFER);

	ap->asr_extra = name;

	return;
}

/* ----------------------------------------------------------------------------
** Return the size of a buffer.
*/

int asr_buf_size(asr_t *ap) {

#ifdef TRACE_CALLS
	Dprintf("asr_buf_size(%p)\n", ap);
#endif

	return(ap->asr_size);
}

/* ----------------------------------------------------------------------------
** Change the size of a buffer.
*/

void asr_buf_resize(asr_t *ap, int add) {
	u_char *buf;
	int s = add + ap->asr_size;
	
#ifdef TRACE_CALLS
	Dprintf("asr_buf_resize(%p, %d)\n", ap, add);
#endif

	buf = (u_char *) xalloc(s);
	memcpy(buf, ap->asr_buf, ap->asr_size);
	memset(&buf[ap->asr_size], 0, s - ap->asr_size);

	if(ap->asr_base)
		xfree((void **) &ap->asr_base);
	else if(ap->asr_buf)
		xfree((void **) &ap->asr_buf);

	ap->asr_buf = ap->asr_base = buf;
	ap->asr_size = s;

	return;
}
	
/* ----------------------------------------------------------------------------
** Define a sizet node.  This defines the scaling factor for subvalue
** reference.
*/

asr_t *asr_sizet(int i) {
	asr_t *ap = asr_new(0);

#ifdef TRACE_CALLS
	Dprintf("asr_sizet(%d)\n", i);
#endif

	ap->asr_size	= 0;
	ap->asr_type 	= N_SIZET;
	ap->asr_line	= 0;
	ap->asr_integer = i;

	return(ap);
}

/* ----------------------------------------------------------------------------
** Return an integer node.
*/

asr_t *asr_int(u_long i, int line) {
	asr_t *ap = asr_new(0);

#ifdef TRACE_CALLS
	Dprintf("asr_int(%lu, %d)\n", i, line);
#endif

	ap->asr_size	= sizeof(i);
	ap->asr_type 	= N_INT;
	ap->asr_line 	= line;
	ap->asr_integer = i;

	return(ap);
}

/* ----------------------------------------------------------------------------
** Return a string node.  This really just returns an initialized buffer.
*/

asr_t *asr_string(char *string, int line) {
	asr_t *ap = asr_new(0);

#ifdef TRACE_CALLS
	Dprintf("asr_string(%s, %d)\n", string, line);
#endif

	ap->asr_size	= string? strlen(string): 0;
	ap->asr_type	= N_BUFFER;
	ap->asr_line	= line;
	ap->asr_buf	= ap->asr_size? xstrdup(string): 0;
	ap->asr_extra  	= "__string";

	return(ap);
}

/* ----------------------------------------------------------------------------
** Returns a character node.
*/

asr_t *asr_char(char c, int line) {
	asr_t *ap = asr_new(0);
	
#ifdef TRACE_CALLS
	Dprintf("asr_char(%c, %d)\n", c, line);
#endif

	ap->asr_size	= 1;
	ap->asr_type 	= N_CHAR;
	ap->asr_line	= line;
	ap->asr_character = c;

	return(ap);
}

/* ----------------------------------------------------------------------------
** Create a node and initialize any children it has.  This is what it all
** comes down to for execution of the CASL program.
*/

asr_t *asr_node(int type, ...) {
	va_list ap;
	asr_t *asr;
	int arity = arity_table[type].arity;
	int i;

#ifdef TRACE_CALLS
	Dprintf("asr_node(%d)\n", type);
#endif

	asr = (asr_t *) asr_new(arity);

	asr->asr_size = 0;
	asr->asr_type = type;
	asr->asr_line = 0;

	va_start(ap, type);

	for(i = 0; i < arity; i++) 
		asr->asr_kids[i] = va_arg(ap, asr_t *);
	
	va_end(ap);

	return(asr);
}

/* ----------------------------------------------------------------------------
** Set the start of the buffer.  This is used to avoid unnecessary malloc/free
** calls for extract.
*/

void asr_buf_slide(asr_t *buf, int count) {
	
#ifdef TRACE_CALLS
	Dprintf("asr_buf_slide(%p, %d)\n", buf, count);
#endif

	if(buf->asr_type != N_BUFFER) 
		error(E_INTERNAL, "Attempt to manipulate buffer data of non-buffer"
		      " type.");
	
	if(count > buf->asr_size) {
		buf->asr_size = 0;
		
		buf->asr_buf  = NULL;
		return;
	}

	buf->asr_size -= count;

	if(!buf->asr_base)
		buf->asr_base = buf->asr_buf;

	buf->asr_buf += count;

	return;
}

/* ----------------------------------------------------------------------------
** Manipulate buffer nodes.
*/

void asr_buf_move(asr_t *from, asr_t *to, int count) {
	char *cp = NULL;

#ifdef TRACE_CALLS
	Dprintf("asr_buf_move(%p. %p, %d)\n", from, to, count);
#endif

	if(from->asr_type != N_BUFFER || 
	    to->asr_type != N_BUFFER)
		error(E_INTERNAL, "Attempt to manipulate buffer data of non-buffer"
		      " type.");

	if(to->asr_size < count)
		count = to->asr_size;

	if(from->asr_size < count) {
		u_char *cp;
		memcpy(to->asr_buf, from->asr_buf, from->asr_size);
		cp = to->asr_buf + from->asr_size;
		memset(cp, 0, count - from->asr_size);
	} else
		memcpy(to->asr_buf, from->asr_buf, count);

 	if((cp = asr_buffer_struct(from))) {
		to->asr_extra = cp;
	}

	return;
}
	
/* ----------------------------------------------------------------------------
** See if this node contains data.
*/

int asr_basetype(asr_t *ap) {
	assert(ap);

#ifdef TRACE_CALLS
	Dprintf("asr_basetype(%p)\n", ap);
#endif

	switch(ap->asr_type) {
	case N_INT:
	case N_NEGINT:
	case N_ID:
	case N_CHAR:
	case N_BUFFER:
		return(1);

	default:
		return(0);
	}

	/*NOTREACHED*/
}

/* ----------------------------------------------------------------------------
** Get a string representation of an ASR node.  This function is indirectly
** recursive by calling list_to_string for lists, which in turn calls
** asr_strep for each item.
*/

int asr_strep(asr_t *ap, char **p) {
	static char buf[100];


#ifdef TRACE_CALLS
	Dprintf("asr_strep(%p, %p)\n", ap, p);
#endif

	if(ap->asr_type == LIST) {
		char *cp = list_to_string(ap);
		*p = xstrdup(cp);
		return(strlen(cp));
	}

	if(!asr_basetype(ap))
		error(E_USER, "attempt to extract meaning from internal language symbol");
       
	switch(ap->asr_type) {
	case N_CHAR:
		sprintf(buf, "%c", ap->asr_character);		
		*p = xstrdup(buf);
		return(1);

	case N_INT:
		sprintf(buf, "%lu", ap->asr_integer);
		buf[99] = '\0';
		*p = xstrdup(buf);
		return(strlen(buf));

	case N_NEGINT:
		sprintf (buf, "%ld", ap->asr_sinteger);
		buf[99] = '\0';
		*p = xstrdup (buf);
		return (strlen (buf));
	
	case N_ID: {
		asr_t *tmp = ap;
		int l;
		
		ap = st_get(ap->asr_ident, Level);
		if(!ap) 
			error(E_USER, "undefined symbol %s", tmp->asr_ident);

		l = asr_strep(ap, p);
		return(l);
	}

	/* XXX copout */

	case N_BUFFER: 
		if(!ap->asr_size) 
			error(E_USER, "Attempt to obtain strep of empty buffer");

		*p = buf2asciiz (ap);

		return(ap->asr_size);

	default:
		assert(0);
	}
	
	return(0);
}

/* ----------------------------------------------------------------------------
** Grab a pointer to the data referenced by an asr_node and return it's size.
** This function needs to be re-written.  Sometmes it allocates the data,
** sometimes not.
*/

int asr_tobuf(asr_t *ap, u_char **pp) {
#ifdef TRACE_CALLS
	Dprintf("asr_tobuf(%p, %p)\n", ap, pp);
#endif

	if(!asr_basetype(ap)) 
		error(E_USER, "attempt to extract meaning from internal language symbol");

	switch(ap->asr_type) {
	case N_INT: 
	case N_NEGINT:
		*pp = xcalloc(1, 4);
		*(*(u_int32_t **) pp) = htonl(ap->asr_integer);
		break;
		       
	case N_CHAR:
		(*pp) = (u_char *) &ap->asr_character;
		break;

	case N_ID: {
		ap = st_get(ap->asr_ident, Level);
		if(!ap) 
			error(E_USER, "Undefined symbol %s", ap->asr_ident);
		
		return (asr_tobuf(ap, pp));
	}

	case N_BUFFER:
		*pp = ap->asr_buf;
		break;
		
	default:
		/* something's wrong with basetype */

		assert(0);
	}

	return(ap->asr_size);
}

/* ----------------------------------------------------------------------------
** Get the integer value of a node.  Do not try to cast non integer types to
** their boolean value.  This function borrows a reference count from the
** caller.
*/

unsigned long asr_getint_strict(asr_t *ap) {
	unsigned long result = 0;
#ifdef TRACE_CALLS
	Dprintf("asr_getint_strict(%p)\n", ap);
#endif

	assert(ap);

	alloc_upref (ap);

	if(ap->asr_type == N_ID) {
		asr_t *new;
		new = lookup(ap);
		alloc_upref (new);
		alloc_downref (ap);
		ap = new;
	}

	if(!asr_basetype(ap))
		ap = eval_expr(ap);

	if(ap->asr_type == N_CHAR) 
		result = ((unsigned long) ap->asr_character);
	else if(ap->asr_type == N_INT || ap->asr_type == N_NEGINT)
		result = (ap->asr_integer);
	else 
		error(E_USER, "Attempt to take integer value from nonscalar.");

	alloc_downref (ap);
	return (result);
}

/* ----------------------------------------------------------------------------
** Get the integer value of an expression.  If the value is not an integer,
** get the boolean value.  This function borrows a reference from the caller.
*/

unsigned long asr_getint(asr_t *ap) {
	unsigned long ul = 0;

#ifdef TRACE_CALLS
	Dprintf("asr_getint(%p)\n", ap);
#endif

	assert(ap);

	alloc_upref (ap);
	if(ap->asr_type != N_CHAR && ap->asr_type != N_INT && ap->asr_type != N_NEGINT) {
		ap = eval_expr(ap);
	}

	switch(ap->asr_type) {
	case N_BUFFER:
		ul = (ap->asr_size ? 1 : 0);
		break;
	
	case LIST:
		ul = (list_items(ap) ? 1 : 0);
		break;

	case N_CHAR:
		ul = ap->asr_character;
		break;

	case N_INT:
	case N_NEGINT:
		ul = ap->asr_integer;
		break;

	default:
	error(E_USER, "Attempt to reference non-numeric value as an integer");
	}

	alloc_downref (ap);
	return(ul);
}

/* ----------------------------------------------------------------------------
** Print a dump of an asr_node tree.  This is not intended to be pretty.
*/

void asr_print(asr_t *ap) {
	int arity;
	int i;

	if(!ap)
		return;

	arity = arity_table[ap->asr_type].arity;

	switch(ap->asr_type) {
	case N_ID: 
		printf("ID: %s\n", ap->asr_ident);

		{
			extern int Level;
			extern asr_t *st_get();
			asr_t *m = st_get(ap->asr_ident, Level);
			asr_print(m);
		}

		break;	

	case N_INT:
		printf("INT: %lu\n", ap->asr_integer);
		break;	

	case N_NEGINT:
		printf ("INT: %ld\n", ap->asr_sinteger);
		break;

	case N_CHAR:
		printf("CHR: %c\n", ap->asr_character);
		break;	

	case LIST:
		printf("LIST\n");
		break;	

	case N_CALL:
		printf("CALL\n");
		break;	

	case N_ASSIGN:
		printf("ASSIGN\n");
		break;	

	case N_IDX:
		printf("IDX\n");
		break;	

	case N_FIELD:
		printf("FIELD\n");
		break;	

	case N_RANGE:
		printf("RANGE\n");
		break;	

	case N_ITE:
		printf("ITE\n");
		break;	

	case N_FOR:
		printf("FOR\n");
		break;	

	case N_WHILE:
		printf("WHILE\n");
		break;	

	case N_FOREACH:
		printf("FOREACH\n");
		break;	

	case N_NEG :
		printf("NEG\n");
		break;

	case N_POP :
		printf("POP\n");
		break;

	case N_PLUS :
	case N_MINUS :
	case N_MUL :
	case N_DIV :
	case N_EQ :
	case N_NE :
	case N_GT :
	case N_LT :
	case N_GE :
	case N_LE :
	case N_AND :
	case N_OR :
		printf("AROP %d\n", ap->asr_type);
		break;

	case N_FUNC :
		printf("FUNCDEC\n");
		break;

	case N_BODY :
		printf("FUNCBODY\n");
		break;

	case N_STRUCT :
		printf("STRUCTDEC\n");
		break;

	case N_STRUCTDEF :
		printf("STRUCTFIELD\n");
		break;

	case N_RETURN :
		printf("RETURN\n");
		break;

	case N_SIZET :
		printf("SIZET\n");
		break;

	case N_SIZESPEC :
		printf("SIZESPEC\n");
		break;

	case N_EXPR :
		printf("EXPR\n");
		break;

	case N_NEW :
		printf("NEW\n");
		break;

	case N_EXTRACT :
		printf("EXTRACT\n");
		break;

	case N_BUFFER :
		printf("BUFFER\n");
		break;

	default: 
		printf("(internal)\n");
		break;
	}

	for(i = 0; i < arity; i++) 
		asr_print(ap->asr_kids[i]);	

	return;
}
