#-------------------------------------------------------------------------------------------------
#   Main IP protocol
#-------------------------------------------------------------------------------------------------

protocol ip = {
	spec = "RFC 791 + RFC 1700 (Assigned Numbers)"
    size = '4*%field(-headerlength)'
	nextlayer = '%string(-protocol)'

    field -version @ (0, 0.4) = {
        type = integer
        default = 4
        help = "version of IP"
    }

    field -headerlength @ (0.4, 0.4) = {
        type = integer
        default = 5
        help = "length of header in DWORD"
    }

    field -tos @ (1, 1) = {
        type = record {
                   list(0.3) { 7 "netctrl" "network control"
                               6 "inetctrl" "internetwork control"
                               5 "critic" "CRITIC/ECP"
                               4 "flashover" "flash override"
                               3 "flash" "flash"
                               2 "immediate" "immediate"
                               1 "priority" "priority"
                               0 "routine" "routine" }
                   flags(0.5) { 0x10 "delay"       "normal delay" "low delay"
                                0x08 "throughput"  "normal throughput" "high throughput"
                                0x04 "reliability" "normal reliability" "high reliability"
                                0x02 "res1"        "reserved" "<illegal>"
                                0x01 "res2"        "reserved" "<illegal>" }
               }
        default = 0x00
        help = "type of service - (precedence, flags)"
    }

    field -length @ (2, 2) = {
        type = length
        default = 0
        help = "total length"
    }

    field -identification @ (4, 2) = {
        type = integer
        default = 0
        help = "identification set by sender"
    }

    field -flags @ (6, 0.3) = {
        type = flags { 0x04 "res" "reserved" "<illegal>"
                       0x02 "df"  "may fragment" "don't fragment"
                       0x01 "mf"  "last fragment" "more fragment" }
        default = 0x02
        help = "flags (fragmentation)"
		read-only
    }

    field -fragoffset @ (6.3, 0.13) = {
        type = integer
        default = 0
        help = "offset of the fragment in the datagram (in offset of 8 bytes)"
		read-only
    }

    field -ttl @ (8, 1) = {
        type = integer
        default = 128
        help = "time to live"
    }

    field -protocol @ (9, 1) = {
        type = list {  
                   1 "icmp" "Internet Control Message Protocol"
                   2 "igmp" "Internet Group Management or Multicast Protocol"
                   3 "ggp" "Gateway-to-Gateway Protocol"
                   4 "ip2" "IP in IP (encapsulation)"
                   5 "st" "Stream"
                   6 "tcp" "Transmission Control Protocol"
                   7 "ucl" "UCL"
                   8 "egp" "Exterior Gateway Protocol"
                   9 "igp" "Any private Interior Gateway Protocol"
                   10 "bbn-rcc-mon" "BBN RCC Monitoring"
                   11 "nvp" "Network Voice Protocol"
                   12 "pup" "PARC Universal packet Protocol"
                   13 "argus" "ARGUS"
                   14 "emcon" "EMCON"
                   15 "xnet" "Cross Net Debugger"
                   16 "chaos" "Chaos"
                   17 "udp" "User Datagram Protocol"
                   18 "mux" "Multiplexing"
                   19 "dcn-meas" "DCN Measurement Subsystems"
                   20 "hmp" "Host Monitoring Protocol"
                   21 "prm" "Packet Radio Measurement"
                   22 "xns-idp" "Xerox NS IDP"
                   23 "trunk-1" "Trunk-1"
                   24 "trunk-2" "Trunk-2"
                   25 "leaf-1" "Leaf-1"
                   26 "leaf-2" "Leaf-2"
                   27 "rdp" "Reliable Data Protocol"
                   28 "irtp" "Internet Reliable Transaction Protocol"
                   29 "iso-tp4" "ISO Transport Protocol Class 4"
                   30 "netblt" "Bulk Data Transfer Protocol"
                   31 "mfe-nsp" "MFE Network Services Protocol"
                   32 "merit-inp" "MERIT Internodal Protocol"
                   33 "sep" "Sequential Exchange Protocol"
                   34 "3pc" "Third Party Connect Protocol"
                   35 "idpr" "Inter-Domain Policy Routing Protocol"
                   36 "xtp" "XTP"
                   37 "ddp" "Datagram Delivery Protocol"
                   38 "idpr-cmtp" "IDPR Control Message Transport Protocol"
                   39 "tp++" "TP++ Transport Protocol"
                   40 "il" "IL Transport Protocol"
                   41 "ipv6" "IPv6"
                   42 "sdrp" "Source Demand Routing Protocol"
                   43 "ipv6-route" "Routing Header for IPv6"
                   44 "ipv6-frag" "Fragment Header for IPv6"
                   45 "idrp" "Inter-Domain Routing Protocol"
                   46 "rsvp" "Reservation Protocol"
                   47 "gre" "General Routing Encapsulation"
                   48 "mhrp" "Mobile Host Routing Protocol"
                   49 "bna" "BNA"
                   50 "esp" "Encap Security Payload for IPv6"
                   51 "ah" "Authentication Header for IPv6"
                   52 "i-nlsp" "Integrated Net Layer Security TUBA"
                   53 "swipe" "IP with Encryption"
                   54 "nhrp" "NBMA Next Hop Resolution Protocol"
                   55 "mobile" "IP Mobility"
                   56 "tlsp" "Trans. Layer Sec. Prot. using Kryptonet key mgmt"
                   57 "skip" "SKIP"
                   58 "ipv6-icmp" "ICMP for IPv6"
                   59 "ipv6-nonxt" "No Next Header for IPv6"
                   60 "ipv6-opts" "Destination Options for IPv6"
                   61 "hip" "Any Host Internal Protocol"
                   62 "cftp" "CFTP"
                   63 "ln" "Any Local Network"
                   64 "sat-expak" "SATNET and Backroom EXPAK"
                   65 "kryptolan" "Kryptolan"
                   66 "rvd" "MIT Remote Virtual Disk Protocol"
                   67 "ippc" "Internet Pluribus Packet Core"
                   68 "dfs" "Any Distributed File System"
                   69 "set-mon" "SATNET Monitoring"
                   70 "visa" "VISA Protocol"
                   71 "ipcv" "Internet Packet Core Utility"
                   72 "cpnx" "Computer Protocol Network Executive"
                   73 "cphb" "Computer Protocol Heart Beat"
                   74 "wsn" "Wang Span Network"
                   75 "pvp" "Packet Video Protocol"
                   76 "br-sat-mon" "Backroom SATNET Monitoring"
                   77 "sun-nd" "SUN ND PROTOCOL-Temporary"
                   78 "wb-mon" "WIDEBAND Monitoring"
                   79 "wb-expak" "WIDEBAND EXPAK"
                   80 "iso-ip" "ISO Internet Protocol"
                   81 "vmtp" "VMTP"
                   82 "secure-vmtp" "SECURE-VMTP"
                   83 "vines" "VINES"
                   84 "ttp" "TTP"
                   85 "nsfnet-igp" "NSFNET-IGP"
                   86 "dgp" "Dissimilar Gateway Protocol"
                   87 "tcf" "TCF"
                   88 "eigrp" "Interior Gateway Routing Protocol"
                   89 "ospfigp" "OSPFIGP"
                   90 "sprite-rpc" "Sprite RPC Protocol"
                   91 "larp" "Locus Address Resolution Protocol"
                   92 "mtp" "Multicast Transport Protocol"
                   93 "ax.25" "AX.25 Frames"
                   94 "ipip" "IP-within-IP Encapsulation Protocol"
                   95 "micp" "Mobile Internetworking Control Protocol"
                   96 "scc-sp" "Semaphore Communications Secure Protocol"
                   97 "etherip" "Ethernet-within-IP Encapsulation"
                   98 "encap" "Encapsulation Header"
                   99 "pes" "Any Private Encryption Scheme"
                   100 "gmtp" "GMTP"
                   101 "ifmp" "Ipsilon Flow Management Protocol"
                   102 "pnni" "PNNI over IP"
                   103 "pim" "Protocol Independent Multicast"
               }
        default = 6
        help = "next level protocol used in the data portion of the IP datagram"
    }

    field -checksum @ (10, 2) = {
        type = checksum(ip)
        help = "checksum (for the header only)"
    }

    field -src @ (12, 4) = {
        type = address_ip
        default = '%address_ip'
        help = "source address"
		read-only
    }

    field -dest @ (16, 4) = {
        type = address_ip
        default = '127.0.0.1'
        help = "destination address"
    }

	fieldvariablearray -options @ (20,
				'((%field(-headerlength)>=5?%field(-headerlength):5)-5)*4') = {
		type = mapper(ip_option)
		help = "options"
		read-only
	}
}


#-------------------------------------------------------------------------------------------------
#   IP Options
#-------------------------------------------------------------------------------------------------

protocol ip_option = {
    size = '(%data(0,1)==0||%data(0,1)==1) ? 1 : %data(1,1)'
	read-only

    field -type @ (0, 1) = {
        type = sparelist {
				   0 	"eol" "end of option list"
                   1 	"nop" "no operation"
                   130 	"sec" "security"
                   131	"lsrr" "loose source et record route"
                   68	"time" "internet timestamp"
                   7	"rr" "record route"
                   136	"sid" "stream id"
                   137	"ssrr" "strict source and record route"
                   18	"tr" "trace route (rfc 1393)"
			   }
        help = "the option itself (including flags)"
		default = 0
    }

    field -typeflags @ (0, 0.3) = {
        type = record {
                   flags(0.1) {
					   0x01 "copy" "not copied" "copied"
				   }
                   list(0.2) {
					   0 "ctrl" "control"
                       1 "res1" "reserved for future use"
                       2 "debug" "debugging and measurement"
                       3 "res2" "reserved for future use"
				   }
               }
        help = "information about the option"
		default = 0
    }

	field -data @ (1, '%data(1,1)-1') = {	# -1 because option code already in header
		type = switch('%field(-type)') {
				   0,1			none
				   131,7,137	mapper(ip_route)
				   default		mapper(ip_nodecode)
			   }
		help = "specific data depending on the type"
	}
}

#-------------------------------------------------------------------------------------------------

protocol ip_nodecode = {
    size = '%data(0,1)-1'	# -1 because option code already in header
	read-only

    field -length @ (0, 1) = {
        type = integer
        help = "length of the option"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------

protocol ip_route = {
    size = '%data(0,1)-1'	# -1 because option code already in header
	read-only

    field -length @ (0, 1) = {
        type = integer
        help = "length of the option"
		default = 0
    }

    field -pointer @ (1, 1) = {
        type = integer
        help = "pointer to the next area where to store data (if > length then no more space)"
		default = 0
    }

    fieldarray -addresses @ (2, 4) = {
        nbelements = '(%field(-length)-3)/4'
        type = address_ip
        help = "addresses recorded"
        default = 0
    }
}
