#-------------------------------------------------------------------------------------------------
#   ICMP - Internet Control Message Protocol
#-------------------------------------------------------------------------------------------------

protocol icmp = {
	spec = "RFC 792 + RFC 1700 (Assigned Numbers)"
    size = '%computedsize'

    field -type @ (0, 1) = {
		type = list {
                   0 "echo-reply" "Echo Reply"
                   3 "destination-unreachable" "Destination Unreachable"
                   4 "source-quench" "Source Quench"
                   5 "redirect" "Redirect"
                   6 "alternate-address" "Alternate Host Address"
                   8 "echo" "Echo"
                   9 "router-advertisement" "Router Advertisement"
                   10 "router-solicitation" "Router Selection"
                   11 "time-exceeded" "Time Exceeded"
                   12 "parameter-problem" "Parameter Problem"
                   13 "timestamp-request" "Timestamp Request"
                   14 "timestamp-reply" "Timestamp Reply"
                   15 "information-request" "Information Request"
                   16 "information-reply" "Information Reply"
                   17 "mask-request" "Address Mask Request"
                   18 "mask-reply" "Address Mask Reply"
                   30 "traceroute" "Traceroute"
                   31 "conversion-error" "Datagram Conversion Error"
                   32 "mobile-redirect" "Mobile Host Redirect"
                   33 "ipv6-where-are-you" "IPv6 Where-Are-You"
                   34 "ipv6-i-am-here" "IPv6 I-Am-Here"
                   35 "mobile-reg-request" "Mobile Registration Request"
                   36 "mobile-reg-reply" "Mobile Registration Reply"
                   37 "domain-name-request" "Domain Name Request"
                   38 "domain-name-reply" "Domain Name Reply"
                   39 "skip" "SKIP"
                   40 "photuris" "Photuris"
			   }
		help = "class of the message"
		default = 0
    }

    field -code @ (0, 2) = {
		type = sparelist {
                   0x0000 "echo-reply" "Echo Reply Message"
                   0x0300 "net-unreachable" "Net Unreachable"
                   0x0301 "host-unreachable" "Host Unreachable"
                   0x0302 "protocol-unreachable" "Protocol Unreachable"
                   0x0303 "port-unreachable" "Port Unreachable"
                   0x0304 "fragment-set" "Fragmentation Needed and Don't Fragment was Set"
                   0x0305 "source-route-failed" "Source Route Failed"
                   0x0306 "network-unknown" "Destination Network Unknown"
                   0x0307 "host-unknown" "Destination Host Unknown"
                   0x0308 "host-isolated" "Source Host Isolated"
                   0x0309 "net-adm-prohibited" 
						  "Communication with Destination Network is Administratively Prohibited"
                   0x030a "host-adm-prohibited"
						  "Communication with Destination Host is Administratively Prohibited"
                   0x030b "net-tos-unreachable" "Destination Network Unreachable for Type of Service"
                   0x030c "host-tos-unreachable" "Destination Host Unreachable for Type of Service"
                   0x030d "administratively-prohibited" "Communication Administratively Prohibited"
                   0x030e "host-precedence-unreachable" "Host Precedence Violation"
                   0x030f "precedence-unreachable" "Precedence cutoff in effect"
 	               0x0400 "source-quench" "Source Quench"
                   0x0500 "net-redirect" "Redirect Datagram for the Network (or subnet)"
                   0x0501 "host-redirect" "Redirect Datagram for the Host"
                   0x0502 "net-tos-redirect" "Redirect Datagram for the Type of Service and Network"
                   0x0503 "host-tos-redirect" "Redirect Datagram for the Type of Service and Host"
                   0x0600 "alt-address-host" "Alternate Address for Host"
                   0x0800 "echo" "Echo Message"
                   0x0900 "router-advertisement" "Router Advertisement"
                   0x0a00 "router-solicitation" "Router Selection"
                   0x0b00 "ttl-exceeded" "Time to Live exceeded in Transit"
                   0x0b01 "reassembly-timeout" "Fragment Reassembly Time Exceeded"
                   0x0c00 "pointer-error" "Pointer indicates the error"
                   0x0c01 "option-missing" "Missing a Required Option"
                   0x0c02 "packet-too-big" "Bad Length"
                   0x0d00 "timestamp-request" "Timestamp Request"
                   0x0e00 "timestamp-reply" "Timestamp Reply"
                   0x0f00 "information-request" "Information Request"
                   0x1000 "information-reply" "Information Reply"
                   0x1100 "mask-request" "Address Mask Request"
                   0x1200 "mask-reply" "Address Mask Reply"
                   0x2801 "unknown-sec-param" "Uunknown security parameters index"
                   0x2802 "auth-failed" "Valid security parameters, but authentication failed"
                   0x2803 "decryp-failed" "Valid security parameters, but decryption failed"
			   }
		help = "type of the message (depends on the type)"
		default = 0
    }

    field -checksum @ (2, 2) = {
        type = checksum(ip)
        help = "the checksum (0 means no checksum generated)"
		default = 0
    }

	field -data @ (4, '%variable') = {
		type = switch('%field(-type)') {
				   0, 8			mapper(icmp_echo)
				   3, 4			mapper(icmp_unreachable)
				   5			mapper(icmp_redirect)
				   10			mapper(icmp_router)
				   12			mapper(icmp_parameter)
				   13, 14		mapper(icmp_timestamp)
				   15, 16		mapper(icmp_information)
				   17, 18		mapper(icmp_addressmask)
				   default		mapper(icmp_empty)
			   }
		help = "specific data according to type"
	}
}

#-------------------------------------------------------------------------------------------------
# ICMP - Message with no special options

protocol icmp_empty = {
	spec = "RFC 792"
	size = 4

    field -unused @ (0, 4) = {
        type = integer
        help = "unused DWORD"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Mapper to hold header when original IP embedded in answer

protocol icmp_ip = {
	spec = "RFC 792"
	size = '%computedsize'

    field -ip @ (0, '%variable') = {
        type = mapper(ip)
        help = "IP header of packet generating the error"
    }

    field -src_port @ ('%follows', 2) = {
        type = integer
        help = "source port (TCP or UDP)"
		default = 0
    }

    field -dest_port @ ('%follows', 2) = {
        type = integer
        help = "destination port (TCP or UDP)"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Destination Unreachable

protocol icmp_unreachable = {
	spec = "RFC 792"
	size = '%computedsize'

    field -unused @ (0, 2) = {
        type = integer
        help = "unused DWORD"
		default = 0
    }

    field -mtu @ (2, 2) = {
        type = integer
        help = "MTU to used (when fragmentation required, i.e. code = 0x0304)"
		default = 0
    }

    field -header @ (4, '%variable') = {
        type = mapper(icmp_ip)
        help = "embedded header IP (with options) + 8 bytes of data"
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Parameter Problem Message

protocol icmp_parameter = {
	spec = "RFC 792"
	size = 4

    field -pointer @ (0, 1) = {
        type = integer
        help = "pointer to the problem"
		default = 0
    }

    field -unused @ (1, 3) = {
        type = integer
        help = "unused DWORD"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Redirect Message

protocol icmp_redirect = {
	spec = "RFC 792"
	size = '%computedsize'

    field -gatewayaddress @ (0, 4) = {
        type = address_ip
        help = "Address of the gateway to which traffic should be sent"
		default = '127.0.0.1'
    }

    field -header @ (4, '%variable') = {
        type = mapper(icmp_ip)
        help = "embedded header IP (with options) + 8 bytes of data"
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Echo or Echo Reply Message

protocol icmp_echo = {
	spec = "RFC 792"
	size = 4

    field -identifier @ (0, 2) = {
        type = integer
        help = "an identifier to aid in matching echos and replies, may be zero"
		default = 0
    }

    field -sequence @ (2, 2) = {
        type = integer
        help = "a sequence number to aid in matching echos and replies, may be zero"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Router Selection

protocol icmp_router = {
	spec = "RFC 792"
	size = '4+8*%field(-nbaddresses)'

    field -nbaddresses @ (0, 1) = {
        type = integer
        help = "number of addresses"
		default = 0
    }

    field -entrysize @ (1, 1) = {
        type = integer
        help = "address entry size (always 2)"
		default = 2
    }

    field -lifetime @ (2, 2) = {
        type = integer
        help = "lifetime"
		default = 0
    }

    fieldarray -address_pref @ (4, 8) = {
		nbelements = '%field(-nbaddresses)'
        type = record {
				   address_ip(4)
				   integer(4)
			   }
        help = "tuple of address and preference level"
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Timestamp or Timestamp Reply Message

protocol icmp_timestamp = {
	spec = "RFC 792"
	size = 16

    field -identifier @ (0, 2) = {
        type = integer
        help = "an identifier to aid in matching timestamps and replies, may be zero"
		default = 0
    }

    field -sequence @ (2, 2) = {
        type = integer
        help = "a sequence number to aid in matching timestamps and replies, may be zero"
		default = 0
    }

    field -originate @ (4, 4) = {
        type = integer
        help = "originate timestamp"
		default = 0
    }

    field -receive @ (8, 4) = {
        type = integer
        help = "receive timestamp"
		default = 0
    }

    field -trasmit @ (12, 4) = {
        type = integer
        help = "transmit timestamp"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Address Mask Request or Reply

protocol icmp_addressmask = {
	spec = "RFC 792"
	size = 8

    field -identifier @ (0, 2) = {
        type = integer
        help = "an identifier to aid in matching requests and replies, may be zero"
		default = 0
    }

    field -sequence @ (2, 2) = {
        type = integer
        help = "a sequence number to aid in matching requests and replies, may be zero"
		default = 0
    }

    field -mask @ (4, 4) = {
        type = integer
        help = "network address mask"
		default = 0
    }
}

#-------------------------------------------------------------------------------------------------
# ICMP - Information Request or Information Reply Message

protocol icmp_information = {
	spec = "RFC 792"
	size = 4

    field -identifier @ (0, 2) = {
        type = integer
        help = "an identifier to aid in matching requests and replies, may be zero"
		default = 0
    }

    field -sequence @ (2, 2) = {
        type = integer
        help = "a sequence number to aid in matching requests and replies, may be zero"
		default = 0
    }
}
