/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "util.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <sys/types.h>

#include "table.h"

/* generic silly utility routines, wrappers, etc */

/* ----------------------------------------------------------------------------
** this is the focal point for all allocation within casl. Anything you do that 
** ever allocates memory must decay to a call to this routine (this enables
** us to swap allocators in two or three lines of code, and also allows for
** allocation tracing.
*/

void *xalloc(const int size) {
	void *ptr = (void *) malloc(size);

	if(!ptr) {
		fprintf(stdout, "Memory allocation failed\n"
				"Aborting!\n");
#ifdef DEBUG
		assert(0);
#else
		exit(-1);
#endif
	}

#ifdef ALLOCATION_TRACE
	printf("\t\t\t\t\tALLOC: CALL FOR %d BYTES\n", size);
#endif

	return(ptr);
}

/* ----------------------------------------------------------------------------
** Wrap calloc().
*/

void *xcalloc(const int num, const int size) {
	void *ptr = (void *) xalloc(num * size);
	if(!ptr) {
		fprintf(stdout, "Memory allocation failed\n"
				"Aborting!\n");
#ifdef DEBUG
		assert(0);
#else
		exit(-1);
#endif
	}

	memset(ptr, 0, num * size);

	return(ptr);
}

/* ----------------------------------------------------------------------------
** Wrap realloc().
*/

void *xrealloc(void *ptr, size_t size) {
	ptr = realloc(ptr, size);
	assert(ptr);

	return(ptr);
}

/* ----------------------------------------------------------------------------
** Wrap strdup().
*/

char *xstrdup(const char *str) {
	int len;
	char *s;

	assert(str);

	len = strlen(str) + 1;
	s = (char *) xalloc(len);
	if(!s) {
		fprintf(stdout, "Memory allocation failed\n"
				"Aborting!\n");
#ifdef DEBUG
		assert(0);
#else
		exit(-1);
#endif
	}
	
	memcpy(s, str, len);

	return(s);
}

/* ----------------------------------------------------------------------------
** Like strdup, but does it with an arbritrarily sized buffer.
*/

u_char *xbufdup(u_char *b, int size) {
	u_char *n;
	
	assert(b);

	n = xalloc(size);
	memcpy(n, b, size);

	return(n);
}

/* ----------------------------------------------------------------------------
** Like strdup, but does it with an unsigned long integer.
*/

u_int32_t *xulongdup(u_int32_t a) {
	return((u_int32_t *)xbufdup((u_char *) &a, sizeof(a)));
}

/* ----------------------------------------------------------------------------
** Wrap free().
*/

void xfree(void **p) {

#ifdef ALLOCATION_TRACE
	static int fc = 0;

	printf("\t\t[%d] FREE\n", fc++);
#endif

	free(*p);
	*p = NULL;

	return;
}
