/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"
#include <stdarg.h>

int SourceLine;
int QuietFlag = 0;

extern int casl_line;
extern char *casl_file;

extern list_t *FunctionStack;


/* Various utility routines used by CASL. */

/* ---------------------------------------------------------------------------
** Print an error message.
*/

int AlwaysWarn = 0;
int DebugOnError = 0;

void error(int level, char *format, ...) {
	char *cp;
	va_list ap;

	if(QuietFlag && level == E_WARN)
		return;

	if(FunctionStack)
		cp = (char *) FunctionStack->data;
	else
		cp = NULL;

	if(level == E_INTERNAL) 
		fprintf(stderr, "(INTERNAL ERROR) possibly line %d file \"%s\"\n\t ",
			casl_line, casl_file);
	else
		fprintf(stderr, "In file \"%s\", procedure \"%s\" near line %d:\n\t", 
			casl_file ? casl_file : "this script", 
			cp ? cp : "<global>",
			casl_line);

	va_start(ap, format);

	vfprintf(stderr, format, ap);

	putc('\n', stderr);

	if(DebugOnError)
		debugger();

	if(level == E_WARN)
		return;

	if(AlwaysWarn) {
		/* We're in sxs. Can't just ignore the error, since
		 * errors are just like panic statements.
		 */

		extern jmp_buf SxSbuf;

		longjmp(SxSbuf, 1);
	}

	exit(1);
}

/* ---------------------------------------------------------------------------
** Copy an ASR base type node
*/

asr_t *asr_copy(asr_t *exn) {
	asr_t *val = NULL;
	
	switch(exn->asr_type) {
	case N_INT:
		val = asr_int(exn->asr_integer, 0);
		break;

	case N_NEGINT:
		val = asr_int(exn->asr_integer, 0);
		val->asr_type = N_NEGINT;
		break;

	case N_CHAR:
		val = asr_char(exn->asr_character, 0);
		break;

	case N_BUFFER:
		val = asr_buffer(exn->asr_size, asr_buffer_struct(exn));
		asr_buf_move(exn, val, exn->asr_size);
		break;

	case LIST:
		val = list_copy(exn);
		break;

	default:
		error(E_INTERNAL, "Unable to create copy of expression.");

	}

	return(val);
}

/* ---------------------------------------------------------------------------
** look up a variable name in the symbol table, returning the actual object it
** refers to. Can be called on non-ID variables for cases where it's unknown
** whether the node is an ID (simplifies calling code).
*/

asr_t *lookup(asr_t *sym) {
	asr_t *tmp = sym;

	/* If we're not passed an identifier, pass back the argument */

	if(sym->asr_type != N_ID)
		return(sym);

	/* otherwise, look it up in the symbol table */

	if(sym->asr_type == N_ID) {
		sym = st_get(sym->asr_ident, Level);
		if(!sym)
			error(E_USER, "undefined identifier \"%s\"",
			      tmp->asr_ident);
	
		assert(sym->asr_type != N_ID);
	}
		
	return(sym);
}

/* ---------------------------------------------------------------------------
** Lookup a variable and return an integer.
*/

u_long lookup_num(char *symname) {
	asr_t *a;

	if(!(a = st_get(symname, Level)))
		error(E_USER, "undefined identifier \"%s\"",
		      symname);

	return(asr_getint(a));
}

/* ---------------------------------------------------------------------------
** Lookup a variable and return it as a string.
*/

char *lookup_string(char *symname) {
	asr_t *a;

	if(!(a = st_get(symname, Level)))
		error(E_USER, "undefined identifier \"%s\"",
		      symname);

	if(a->asr_type != N_BUFFER)
		error(E_USER, "CASL expects variable \"%s\" to be a string",
		      symname);

	return(buf2asciiz (a));
}

/* ---------------------------------------------------------------------------
** Convert a list to a string.
*/

char *list_to_string(asr_t *list) {
	char string[8192];
	char *cp = string;
	asr_t *cur;

	int left = 8192;

	memset(string, 0, 8192);

	if(list->asr_type != LIST)
		list = asr_node(LIST, list, NULL);
	
	for(cur = list; cur; cur = cur->asr_kids[1]) {
		if(cur->asr_kids[0]) {
			char *ap;
			int l = asr_strep(cur->asr_kids[0], &ap);

			if (ap) {
				strncpy(cp,  ap, l > left ? left : l);
				xfree ((void **)&ap);
			}
			cp += l > left ? left : l;
			left -= l > left ? left : l;
			if(!left) {
				string[8191] = '\0';
				return(xstrdup(string));
			}
		}
	}

	return(xstrdup(string));
}

/* ---------------------------------------------------------------------------
** What it says.
*/

u_long string_to_ulong(char *string) {
	if(!string) return 0;
	if(strlen(string) < 3 || string[1] != 'x')
		return(strtoul(string, NULL, 10));

       	return(strtoul(string, NULL, 16));
}

/* ---------------------------------------------------------------------------
** convert a string to an integer, handling the hexidecimal case.
*/

int string_to_int(char *string) { 
	if(strlen(string) < 3) 
		return(atoi(string));

	if(string[1] == 'x')
		return(strtol(string, NULL, 16));

	return(atoi(string));
}

/* ---------------------------------------------------------------------------
** Invoke the mystical debugger.
*/

extern int RegressionTest; 
extern int StopFree;

void debugger() {

#ifdef RELEASE
	fprintf(stderr, "Internal debugging support is not available in"
			" this release of CASL.\n");
	exit(1);
#else

#ifndef __EXECUTABLE_FILE__ 
	fprintf(stderr, "Internal debugging support cannot be enabled"
			" without specifying the executable file name.\n");	
	
	abort();
	exit(1);
#else
	
#ifndef __DEBUGGER__
#define __DEBUGGER__ "/usr/bin/gdb"
#endif

#ifdef  __ASSERT_AUTO_UNSUSPEND__
#define __DEBUGGER_SCRIPT__	"/usr/local/lib/gdb.assert.commands"
#endif

	static int debugged = 0;
	char pid[10];
	int suspend = 1;

	if(RegressionTest) 
		abort();

	StopFree = 1;

	if(!debugged) 
		debugged++;
	else
		return;

	sprintf(pid,"%d", getpid());
	pid[9] = '\0';

	if(!fork()) {
		execl(__DEBUGGER__, "debugger", 
		      "-q",
#ifdef __DEBUGGER_SCRIPT__
		      "-x", __DEBUGGER_SCRIPT__,
#endif
		      __EXECUTABLE_FILE__, pid, NULL);
	}

	while(suspend);

	return;
#endif
#endif
}	

/* ---------------------------------------------------------------------------
** Convert an integer to a string representation of an IP address.
*/

char *addr_ntoa(u_int32_t address) {
	static char a[40];
	u_char *cp = (u_char *) &address;
	
	sprintf(a, "%d.%d.%d.%d", cp[0], cp[1], cp[2], cp[3]);
	return(a);
}

/* ---------------------------------------------------------------------------
** Resolve a name to an IP.
*/

u_int32_t resolve(char *name) {
	u_long addr;

	addr = inet_addr(name);
	if(addr == INADDR_NONE) {
		struct hostent *hp = gethostbyname(name);
		if(!hp)
			return(INADDR_NONE);

		memcpy(&addr, hp->h_addr, 4);
	}

	return(addr);
}
	
/* ---------------------------------------------------------------------------
** Tie the cargo down so it doesn't fall off.
*/

char *strap(char *a, char *b) {
	char *c;

	if(!b)
		return(a);

	c = xalloc(strlen(a) + strlen(b) + 1);
	strcpy(c, a);
	strcat(c, b);

	free(a);
	free(b);

	return(c);
}

/* ---------------------------------------------------------------------------
** Take a buffer node and return a null terminated version of it's data.
** Perhaps buffers should be modified to add an extra byte to eliminate this
** need?
*/

char *buf2asciiz(const asr_t *buf) {
	char *str;
	if (buf->asr_type != N_BUFFER)
		return NULL;

	str = xalloc (buf->asr_size + 1);
	strncpy (str, buf->asr_buf, buf->asr_size);
	str[buf->asr_size] = 0;
	return str;
}
/*  -------------------------------------------------------------------- */

