/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

/* deal with casl structures.
 *
 * The way I'm handling structures right now is to use a seperate
 * table for each structure; each structure table contains a 
 * (name, offset, length) tuple for each field. 
 *
 * A structure is used in a CASL script by assigning to an ID
 * using the NEW operator; a buffer of data big enough for the 
 * new structure is returned. A stucture can be extracted from 
 * a buffer of data with the EXTRACT operator.
 */

#ifdef SET_TEST
int LLL;
#endif

/* ---------------------------------------------------------------------------
** define a structure, which can later be extracted from a buffer of data
*/

void eval_struct(asr_t *node) {
	st *ss = NULL;
	char *id = node->asr_kids[0]->asr_ident;
	asr_t *fieldlist = node->asr_kids[1];
	asr_t *n;
	int boff = 0;
	int fields = 0;

#ifdef TRACE_CALLS
	Dprintf("eval_struct(%p)\n", node);
#endif

	/* create a name for the new structure */

	sd_new(id);
	ss = sd_get(id);
	ss->st_scale = S_BYT;

	/* fill out it's fields */

	for(n = fieldlist; n; n = n->asr_kids[1]) {
		sdef *s;

		/* define the index scale for this structure */

		if(n->asr_kids[0]->asr_type == N_SCALE) {
			ss->st_scale = n->asr_kids[0]->asr_kids[0]->asr_integer;
			continue;
		}
		
		/* otherwise define a field of this structure */

		s = eval_field(n->asr_kids[0], &boff);
		if(s) {
			if(!sd_addfield(id, s)) {
				error(E_INTERNAL, 
					"unable to add field"
					" \"%s\" to struct"
					" \"%s\"",
					s->sd_name, id);
			}
		} else
			error(E_INTERNAL, 
				"unable to evaluate field"
				" definition for table \"%s\"", id);

		fields += 1;
	}

	/* a structure with no field definitions is resizeable */

	if(!fields) 
		ss->st_flags |= S_RESIZEABLE;


	return;
}		

/* ---------------------------------------------------------------------------
** define the individual fields of a structure
*/

sdef *eval_field(asr_t *node, int *boff) {
	sdef *s = (sdef *) xcalloc(1, sizeof(*s));
	char *name;
	int i, mul = 0;

#ifdef TRACE_CALLS
	Dprintf("eval_field(%p, %p)\n", node, boff);
#endif

	name = node->asr_kids[0]->asr_ident;

	if(node->asr_kids[1]->asr_kids[1]->asr_integer == S_VAR) {
		s->sd_name = xstrdup(name);
		s->sd_type = S_BUF;
		s->sd_bitoffset = *boff;
		s->sd_bitlength = 0;
	}

	/* the field is an expression of the number of bits consumed
	 * by it in the structure.
	 */

	i = asr_getint(node->asr_kids[1]->asr_kids[0]);

	/* see if the size is in bits, bytes, or something else, and modify the count
	 * accordingly. 
	 */

	switch(node->asr_kids[1]->asr_kids[1]->asr_integer) {
	case S_BIT:
		mul = 1;
		break;

	case S_BYT:
		mul = 8;
		break;
		
	case S_WRD:
		mul = 16;
		break;

	case S_DWD:
		mul = 32;
		break;

	default:
		error(E_INTERNAL, "parsed non-recognized size operator %d", 
			node->asr_kids[1]->asr_kids[1]->asr_integer);
		break;			
	}

	i *= mul;

	s->sd_name = xstrdup(name);
	s->sd_type = S_VAL;
	s->sd_bitoffset = *boff;
	s->sd_bitlength = i;

	*boff += i;

	return(s);
}

/* ---------------------------------------------------------------------------
** create a new instance of a structure, or, more accurately, a new buffer of
** data from which to extract the structure's fields 
*/

asr_t *eval_new(asr_t *node) {
	char *sname;
	st *s;
	int length;
	asr_t *buf;

#ifdef TRACE_CALLS
	Dprintf("eval_new(%p)\n", node);
#endif

	sname = node->asr_kids[1]->asr_ident;

	/* find the structure definition */

	s = sd_get(sname);
	if(!s) 
		error(E_USER, 
			"attempt to reference undefined structure %s",
			sname);
		
	/* get the length of the structure in bytes */

	length = sd_len(sname);
	if(length < 8)
		length = 0;
	else
		length = (length / 8) + (length % 8 ? 1 : 0);

	/* create a buffer large enough for the structure */

	buf = asr_buffer(length, sname);

	/* put the new structure in the symbol table */

	alloc_upref(buf);

	st_replace(node->asr_kids[0]->asr_ident, buf, Level);
	return(buf);
}

/* ---------------------------------------------------------------------------
** get the name of a structure. We're going to need this to checksum
** lists of packet components (so we can fish out the IP and UDP 
** headers, etc).
*/

asr_t *eval_sname(asr_t *node) {
	char *sn;
	asr_t *st, *temp;

#ifdef TRACE_CALLS
	Dprintf("eval_sname(%p)\n", node);
#endif

	st = node->asr_kids[0];
	if(!st)
		error(E_INTERNAL, "missing argument to sname node");

	if(st->asr_type == N_ID) {
		temp = st;
		st = st_get(st->asr_ident, Level);
		if(!st)
			error(E_USER, "attempt to reference undefined"
				" id \"%s\"", temp->asr_ident);
		alloc_upref (st);
		alloc_downref (temp);
	}

	if(st->asr_type != N_BUFFER) 
		error(E_USER, "attempt to extract name from integral type");
	
	sn = asr_buffer_struct(st);
	if(!sn)
		sn = "undefined";

	return(asr_string(sn, 0));
}

/* ---------------------------------------------------------------------------
** ID (assign-to) ID (deref as) ID (buffer)  - extract a structure from a
** buffer
*/

asr_t *eval_extract(asr_t *node) {
	st *s;
	asr_t *tob, *fromb;

	char *as, *from;
	int i;

#ifdef TRACE_CALLS
	Dprintf("eval_extract(%p)\n", node);
#endif

	as   = node->asr_kids[0]->asr_ident;
	from = node->asr_kids[1]->asr_ident;

	/* get the structure definition */

	if(!(s = sd_get(as))) 
		error(E_USER, "attempt to extract undefined structure %s"
			 	" from %s", as, from);

	/* get the length of the structure */

	i = sd_len(as);
	if(i < 8)
		i = 8;

	i = (i / 8) + (i % 8 ? 1 : 0);

	/* find the old buffer */

	fromb = st_get(from, Level);
	if(!fromb) 
		error(E_USER, "attempt to extract from nonexistant packet %s", from);

	if(fromb->asr_type != N_BUFFER) 
		error(E_USER, "attempt to extract %s from non-packet %s",
				as, from);

	/* make sure the old buffer had enough data to copy from */

	if(fromb->asr_size < i)
		error(E_USER, "source buffer is not large enough to contain \"%s\"", as);

	/* create a new buffer for the structure */

	tob   = asr_buffer(i, as);

	asr_buf_move(fromb, tob, i);
	asr_buf_slide(fromb, i);

	asr_buffer_struct_set(tob, as);

	alloc_downref (node->asr_kids[0]);
	alloc_downref (node->asr_kids[1]);
	return(tob);
}

/* ---------------------------------------------------------------------------
** Like extract, but extracts just unnamed bytes.
*/

asr_t *eval_extract2(asr_t *node) {
	asr_t *tob, *fromb;

	char *from;
	int s;

#ifdef TRACE_CALLS
	Dprintf("eval_extract2(%p)\n", node);
#endif

	from = node->asr_kids[1]->asr_ident;

	s = asr_getint(node->asr_kids[0]);
	alloc_downref (node->asr_kids[0]);

	/* find the old buffer */

	fromb = st_get(from, Level);
	if(!fromb) 
		error(E_USER, "attempt to extract from nonexistant packet %s", from);
	      
	if(fromb->asr_type != N_BUFFER) 
		error(E_USER, "attempt to extract %d bytes from non-packet %s",
				s, from);

	/* make sure the old buffer had enough data to copy from */

	if(fromb->asr_size < s)
		error(E_USER, "source buffer is not large enough to"
			 	" contain %d bytes", s);

	/* create a new buffer for the structure */

	tob   = asr_buffer(s, NULL);

	asr_buf_move(fromb, tob, s);
	asr_buf_slide(fromb, s);

	return(tob);
}

/* ---------------------------------------------------------------------------
** assign a number to a bit/byte subset of a buffer. The vast majority of this
** code is sanity checking.
*/

asr_t *eval_assign_idx(asr_t *idx, asr_t *value) {
	extern table_t *Specials;
	asr_t *id, *buf, *range;
	int i, start, stop, resize;
	int scale;
	st *s = NULL;
	char *str;

#ifdef TRACE_CALLS
	Dprintf("eval_assign_idx(%p, %p)\n", idx, value);
#endif

	/* get the ID, make sure it exists, and make sure it's a buffer */

	id = idx->asr_kids[0];
	range = idx->asr_kids[2];

	if(id->asr_type != N_ID) 
		error(E_USER, "attempt to assign subvalue to integral type");

	if(t_get(Specials, id->asr_ident)) 
		error(E_USER, "attempt to dereference value from control variable");

	buf = st_get(id->asr_ident, Level);
	if(!buf) 
		error(E_USER, "attempt to assign subvalue to nonexistant variable");

#ifdef SET_TEST
	LLL = buf->asr_size;
#endif

	if(buf->asr_type != N_BUFFER) 
		error(E_USER, "attempt to assign subvalue to integral type");
	

	/* XXX catch struct field assignments here */

	if(idx->asr_kids[1]->asr_type == N_ID) 
		return(struct_assign(idx->asr_kids[0], idx->asr_kids[1], value));

	/* sanity check IDX children */

	if(idx->asr_kids[1]->asr_type != N_SIZET) 
		error(E_INTERNAL, "assign to nonrecognized size thingy robble robble");

	if(idx->asr_kids[2]->asr_type != N_RANGE) 
		error(E_INTERNAL, "assign to weird range robble robble");

	/* get and sanity check the start number */

	start = asr_getint_strict(range->asr_kids[0]);

	/* get the stop number. If the number is an END node, we're writing to the 
	 * end of the node's buffer.
	 */

	if(!range->asr_kids[1])
		stop = start + 1;
	else if(range->asr_kids[1]->asr_type == N_END) 
		stop = -1;	
	else 
		stop = asr_getint_strict(range->asr_kids[1]) + 1;
		
	/* get the name of the buffer and see if it's resizeable */

	str = asr_buffer_struct(buf);
	if(!str || strstr(str, "INTERNAL") || !(s = sd_get(str))) {
		resize = 0;
	} else {
		if(s->st_flags & S_RESIZEABLE) {
			resize = 1;
		} else
			resize = 0;
	}

	/* someone sometime is going to be stupid enough to do this */

	if(stop != -1 && stop - start < 0)
		error(E_USER, "attempt to assign to negative range");

	/* get the value to set */

	i = asr_getint(value);
	alloc_downref (value);

	/* do the actual data manipulation */

	if(idx->asr_kids[1]->asr_integer == S_DEF) {
		if(!s)
			scale = S_BYT;
		else
			scale = s->st_scale;
	} else
		scale = idx->asr_kids[1]->asr_integer;

	switch(scale) {
	case S_BIT:
		asr_buf_setbits(buf, start, stop, i, resize);
		break;

	case S_BYT:
		asr_buf_setbytes(buf, start, stop, i, resize);
		break;

	case S_WRD:
		asr_buf_setwords(buf, start, stop, i, resize);
		break;

	case S_DWD:
		asr_buf_setdwords(buf, start, stop, i, resize);
		break;

	default:
		error(E_INTERNAL, "unrecognized size specifier");
		break;
	}

	alloc_upref (buf);
	return(buf);
}

/* ---------------------------------------------------------------------------
** Set a range of bits in a buffer.
*/

void asr_buf_setbits(asr_t *buf, int start, int stop, u_long i, int r) {
	int b;

#ifdef TRACE_CALLS
	Dprintf("asr_buf_setbits(%p, %d, %d, %lu, %d)\n", buf, start, stop, i, r);
#endif

	if(!buf->asr_size) 
		asr_buf_resize(buf, 1);
	

	if(stop < 0) 
		stop = buf->asr_size * 8;

	b = stop / 8;
	if(b > buf->asr_size) {
		if(!r)
			error(E_USER, "range overflows size of buffer");

		asr_buf_resize(buf, b - buf->asr_size);
	}

	setbits(buf->asr_buf, start, stop - start, i);
	return;
}

/* ---------------------------------------------------------------------------
** Set a range of bytes in a buffer.
*/

void asr_buf_setbytes(asr_t *buf, int start, int stop, u_long i, int r) {
#ifdef TRACE_CALLS
	Dprintf("asr_buf_setbytes(%p, %d, %d, %lu, %d)\n", buf, start, stop, i, r);
#endif

	if(stop < 0) 
		stop = buf->asr_size;

	if(stop > buf->asr_size) {
		if(!r)
			error(E_USER, "range overflows size of buffer");
		
		asr_buf_resize(buf, stop - buf->asr_size);
	}

	setbits(buf->asr_buf, (start * 8), ((stop - start) * 8), i);
	return;
}

/* ---------------------------------------------------------------------------
** Set a range of words in a buffer.
*/

void asr_buf_setwords(asr_t *buf, int start, int stop, u_long i, int r) {
	int b;

#ifdef TRACE_CALLS
	Dprintf("asr_buf_setwords(%p, %d, %d, %lu, %d)\n", buf, start, stop, i, r);
#endif

	if(stop < 0) 
		stop = buf->asr_size / 2;

	b = stop * 2;
	if(b > buf->asr_size) {
		if(!r)
			error(E_USER, "range overflows size of buffer");
		
		asr_buf_resize(buf, b - buf->asr_size);
	}

	setbits(buf->asr_buf, ((start * 8) * 2), (((stop - start) * 8) * 2), i);
	return;
}

/* ---------------------------------------------------------------------------
** Set a range of dwords in a buffer.
*/

void asr_buf_setdwords(asr_t *buf, int start, int stop, u_long i, int r) {
	int b;

#ifdef TRACE_CALLS
	Dprintf("asr_buf_setdwords(%p, %d, %d, %lu, %d)\n", buf, start, stop, i, r);
#endif

	if(stop < 0) 
		stop = buf->asr_size / 4;

	b = stop * 4;
	if(b > buf->asr_size) {
		if(!r)
			error(E_USER, "range overflows size of buffer");

		asr_buf_resize(buf, b - buf->asr_size);
	}

	setbits(buf->asr_buf, ((start * 8) * 4), (((stop - start) * 8) * 4), i);
	return;
}

/* ---------------------------------------------------------------------------
** Uh......
*/

void 
asr_buf_string(asr_t *buf, int start, int stop, char *string, int r) {
	int b = stop - start;
	int s;

#ifdef TRACE_CALLS
	Dprintf("asr_buf_string(%p, %d, %d, %s, %d)\n", buf, start, stop, string, r);
#endif

	if(b > (buf->asr_size - start)) {
		if(!r) {
			error(E_USER, "range overflows size of buffer");
		} else {
			s = b - (buf->asr_size - start);
			asr_buf_resize(buf, s);
		}
	}

	strncpy(buf->asr_buf + start, string, b);
	return;
}

/* ---------------------------------------------------------------------------
** return a numeric value from a selection of bits/bytes, or the value of a
** struct's subfield. 
*/

asr_t *eval_idx(asr_t *idx) {
	st *s = NULL;
	asr_t *e1, *e2, *n;
	char *cp;
	int start, stop, i = 0;
	int scale;

#ifdef TRACE_CALLS
	Dprintf("asr_idx(%p)\n", idx);
#endif

	/* XXX catch struct field references here */

	if(idx->asr_kids[1]->asr_type == N_ID) 
		return(struct_get(idx->asr_kids[0], idx->asr_kids[1]));

	/* sanity checks (most of this is not necessary, yacc will catch it, but... */

	if(idx->asr_kids[1]->asr_type != N_SIZET) 
		error(E_INTERNAL, "robble robble wtf is that size supposed to be?");

	if(idx->asr_kids[2]->asr_type != N_RANGE) 
		error(E_INTERNAL, "robble robble wtf is that range supposed to be?");

	/* make sure the id we're referencing is valid and points at a buffer */

	n = lookup(idx->asr_kids[0]);

	if(n->asr_type != N_BUFFER) 
		error(E_USER, "attempt to set subvalue of integral type");

	/* grok the range we're given */

	e1 = idx->asr_kids[2]->asr_kids[0];
	e2 = idx->asr_kids[2]->asr_kids[1];
	
	start = asr_getint_strict(e1);

	if(!e2)
		stop = start + 1;
	else if(e2->asr_type == N_END) 
		stop = -1;	
	else {
		stop = asr_getint_strict(e2);
		stop++;
	}

	/* grab the actual value based on the size specifier */

	cp = asr_buffer_struct(n);
	if(!cp || strstr(cp, "INTERNAL") || !(s = sd_get(cp)))
		scale = S_BYT;
	else
		scale = s->st_scale;

	if(idx->asr_kids[1]->asr_integer == S_DEF) {
		if(!s)
			scale = S_BYT;
		else
			scale = s->st_scale;
	} else
		scale = idx->asr_kids[1]->asr_integer;

	switch(scale) {
	case S_BIT:
		i = asr_get_bits(n, start, stop);
		break;

	case S_BYT:
		i = asr_get_bytes(n, start, stop);
		break;

	case S_WRD:
		i = asr_get_words(n, start, stop);
		break;

	case S_DWD:
		i = asr_get_dwords(n, start, stop);
		break;
		
	default:
		error(E_INTERNAL, "invalid size specifier caught in idx reference");
		break;
	}

	return(asr_int(i, 0));
}

/* ---------------------------------------------------------------------------
** assign a numeric value to a subfield of a structure
*/

asr_t *struct_assign(asr_t *id, asr_t *field, asr_t *value) {
	asr_t *n;
	unsigned long i, start, stop;
	sdef *s;
	char *sname;

#ifdef TRACE_CALLS
	Dprintf("struct_assign(%p, %p, %p)\n", id, field, value);
#endif

	/* mess 'o sanity checking */

	n = st_get(id->asr_ident, Level);
	if(!n)
		error(E_USER, "attempt to reference invalid id %s", id->asr_ident);

	if(n->asr_type != N_BUFFER)
		error(E_USER, "attempt to assign to subvalue of integral type");

	sname = asr_buffer_struct(n);
	if(!sname)
		error(E_USER, "attempt to assign subfield to buffer not associated"
		      " with struct");

	/* look up the numbers for the field (and make sure it exists in the first place) */

	s = sd_getfield(sname, field->asr_ident);
	if(!s)
		error(E_USER, "attempt to reference undefined field %s", field->asr_ident);

	/* get the actual value to assign */

	i = asr_getint(value);
	alloc_downref (value);

	if(s->sd_type != S_VAL)
		error(E_USER, "field %s of struct %s is not assignable",
		      field->asr_ident, sname);

	start = s->sd_bitoffset;
	stop  = s->sd_bitoffset + s->sd_bitlength;

	asr_buf_setbits(n, start, stop, i, 0);
	alloc_upref (n);
	return(n);
}

/* ---------------------------------------------------------------------------
** retrieve a numeric value from the subfield of a structure
*/

asr_t *struct_get(asr_t *id, asr_t *field) {
	asr_t *n;
	sdef *s;
	char *sname;
	u_int32_t val;

#ifdef TRACE_CALLS
	Dprintf("struct_get(%p, %p)\n", id, field);
#endif

	/* mess 'o sanity checking (read the errors) */

	n = st_get(id->asr_ident, Level);
	alloc_downref (id);
	if(!n)
		error(E_USER, "attempt to reference nonexistant id %s", id->asr_ident);
	if(n->asr_type != N_BUFFER)
		error(E_USER, "attempt to reference subvalue of integral type");
	
	sname = asr_buffer_struct(n);
	if(!sname)
		error(E_USER, "attempt to reference subvalue of id not associated with struct");

	s = sd_getfield(sname, field->asr_ident);
	if(!s)
		error(E_USER, "attempt to reference undefined field %s of struct %s",
		      field->asr_ident, sname);

	if(s->sd_type != S_VAL)
		error(E_USER, "field %s of struct %s is not a retrievable field",
		      field->asr_ident, sname);

	/* get the actual info */

	val = asr_get_bits(n, s->sd_bitoffset, s->sd_bitoffset + s->sd_bitlength);
	n = asr_int(val, 0);

	return(n);
}

/* ---------------------------------------------------------------------------
** Get a range of bits from a buffer.
*/

u_long asr_get_bits(asr_t *buf, int start, int stop) {
	int b;
	u_long c;

#ifdef TRACE_CALLS
	Dprintf("asr_get_bits(%p, %d, %d)\n", buf, start, stop);
#endif

	if(stop < 0) 
		stop = buf->asr_size * 8;

	b = stop / 8 + (stop % 1 ? 1 : 0);
	if(b > buf->asr_size)
		error(E_USER, "range overflows size of buffer");
	
	c = getbits(buf->asr_buf, start, stop - start);

	return(c);

}

/* ---------------------------------------------------------------------------
** Get a range of bytes from a buffer.
*/

u_long asr_get_bytes(asr_t *buf, int start, int stop) {
	u_long c;

#ifdef TRACE_CALLS
	Dprintf("asr_get_bytes(%p, %d, %d)\n", buf, start, stop);
#endif

	if(stop < 0) 
		stop = buf->asr_size;

	if(stop > buf->asr_size) 
		error(E_USER, "range overflows size of buffer");

	start *= 8;
	stop *= 8;

	c = getbits(buf->asr_buf, start, stop - start);

	return(c);
}

/* ---------------------------------------------------------------------------
** Get a range of words from a buffer.
*/

u_long asr_get_words(asr_t *buf, int start, int stop) {
	u_long c;
	int b;

#ifdef TRACE_CALLS
	Dprintf("asr_get_words(%p, %d, %d)\n", buf, start, stop);
#endif

	if(stop < 0) 
		stop = buf->asr_size / 2;

	b = stop * 2;
	if(b > buf->asr_size)
		error(E_USER, "range overflows size of buffer");

	stop *= 8;
	stop *= 2;

	start *= 8;
	start *= 2;

	c = getbits(buf->asr_buf, start, stop - start);

	return(c);
}

/* ---------------------------------------------------------------------------
** Get a range of dwords from a buffer.
*/

u_long asr_get_dwords(asr_t *buf, int start, int stop) {
	u_long c;
	int b;

#ifdef TRACE_CALLS
	Dprintf("asr_get_dwords(%p, %d, %d)\n", buf, start, stop);
#endif

	if(stop < 0) 
		stop = buf->asr_size / 4;

	b = stop * 4;
	if(b > buf->asr_size)
		error(E_USER, "range overflows size of buffer");
	
	stop *= 8;
	stop *= 4;
	
	start *= 8;
	start *= 4;

	c = getbits(buf->asr_buf, start, stop - start);

	return(c);
}

/* -------------------------------------------------------------------- */

