/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"
#include "fileio.h"

typedef size_t (*afiofp)(int, const void *, size_t);

static int atomicio(afiofp fn, int fd, void *buf, size_t len);
static char *fdgets(char *buffer, int size, int fd);

/* ----------------------------------------------------------------------------
** Builtin hook for the casl open() function.
*/

asr_t *casl_file_open(asr_t *list) {
	int fd;
	char *str;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for open()\n");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for open()\n");

	if(list->asr_kids[0]->asr_type != N_BUFFER)
		error(E_USER, "argument to open() must be a string"
		      " representing a file name\n");

	str = buf2asciiz (list->asr_kids[0]);
	fd = open(str, O_RDWR | O_CREAT, 0644);
		
	if(fd < 0) {
		error(E_WARN, "open(%s) : %s", list->asr_kids[0]->asr_str,
		      strerror(errno));
		fd = 0;
	}
	xfree ((void **)&str);

	return(asr_int(fd, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the casl remove() function.
*/

asr_t *casl_file_remove(asr_t *list) {
	int fd;
	char *str;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for remove()\n");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for remove()\n");

	if(list->asr_kids[0]->asr_type != N_BUFFER)
		error(E_USER, "argument to remove() must be a string"
		      " representing a file name\n");

	str = buf2asciiz (list->asr_kids[0]);
	fd = unlink(str);
	if(fd < 0) {
		error(E_WARN, "remove(%s) : %s", list->asr_kids[0]->asr_buf,
		      strerror(errno));
		fd = 0;
	} else
		fd = 1;

	xfree ((void **)&str);

	return(asr_int(fd, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the casl write() function.
*/

asr_t *casl_file_write(asr_t *list) {
	asr_t *afd, *abuf;
	int l;
	int fd;

	afiofp fn = (afiofp) write;

	if(!list || !list->asr_kids[0] || !list->asr_kids[1] || !list->asr_kids[1]->asr_kids[0])
		error(E_USER, "too few arguments for write()\n");

	if(list->asr_kids[1]->asr_kids[1])
		error(E_USER, "too many arguments for write()\n");

	afd = list->asr_kids[0];
	fd = asr_getint_strict(afd);

	abuf = list->asr_kids[1]->asr_kids[0];
	alloc_upref (abuf);
	
	if (!asr_basetype (abuf) && abuf->asr_type != N_ID) {
		abuf = eval_expr (abuf);
	}

	if (abuf && abuf->asr_type != N_BUFFER) { 
		asr_t *old = abuf;
		extern asr_t *casl_tobuf(asr_t *);
		asr_t *l = asr_node(LIST, abuf, NULL);
		abuf = casl_tobuf(l);
		alloc_downref (old);
		asr_sfree(&l);
	}

       	l = atomicio(fn, fd, abuf->asr_buf, abuf->asr_size); 

       	if(l < 0) {
       		error(E_WARN, "write(%d) : %s", fd, strerror(errno));
       		l = 0;
       	}

	alloc_downref(abuf);

       	return(asr_int(l, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the CASL read() function.
*/

asr_t *casl_file_read(asr_t *list) {
	u_char *buf; 
	int fd;
	int count;
	int l;

	afiofp fn = (afiofp) read;

	if(!list || !list->asr_kids[0] || !list->asr_kids[1] || !list->asr_kids[1]->asr_kids[0])
		error(E_USER, "too few arguments for read()\n");

	if(list->asr_kids[1]->asr_kids[1])
		error(E_USER, "too many arguments for read()\n");

	fd = asr_getint_strict(list->asr_kids[0]);
	count = asr_getint_strict(list->asr_kids[1]->asr_kids[0]);

	buf = xcalloc(1, count);
	
	if((l = atomicio(fn, fd, buf, count)) < 0) 
		error(E_WARN, "read(%d, %d) : %s", fd, count, strerror(errno));

	list = asr_buffer(l, "read data");
	memcpy(list->asr_buf, buf, l);

	xfree((void **) &buf);

	return(list);
}
       
/* ----------------------------------------------------------------------------
** Builtin hook for the CASL fgets() function.
*/

asr_t *casl_file_fgets(asr_t *list) {
	char *buf; 
	int fd;
	int count;

	if(!list || !list->asr_kids[0] || !list->asr_kids[1] || !list->asr_kids[1]->asr_kids[0])
		error(E_USER, "too few arguments for fgets()\n");

	if(list->asr_kids[1]->asr_kids[1])
		error(E_USER, "too many arguments for fgets()\n");

	fd = asr_getint_strict(list->asr_kids[0]);
	count = asr_getint_strict(list->asr_kids[1]->asr_kids[0]);

	buf = xcalloc(1, count);

	if(!fdgets(buf, count, fd)) 
		return(asr_int(0, 0));

	list = asr_string(buf, 0);

	xfree((void **)&buf);

	return(list);
}

/* ----------------------------------------------------------------------------
** Builtin hook for the CASL rewind() function.
*/

asr_t *casl_file_rewind(asr_t *list) {
	int fd;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for rewind()\n");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for rewind()\n");

	fd = asr_getint_strict(list->asr_kids[0]);

	if(lseek(fd, 0, SEEK_SET) < 0)
		error(E_WARN, "rewind(%d) : %s", fd, strerror(errno));

	return(asr_int(0, 0));
}
		
/* ----------------------------------------------------------------------------
** Builtin hook for the CASL fastforward() function.
*/

asr_t *casl_file_fastforward(asr_t *list) {	
	int fd;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for fastforward()\n");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for fastforward()\n");

	fd = asr_getint_strict(list->asr_kids[0]);

	if(lseek(fd, 0, SEEK_END) < 0)
		error(E_WARN, "fastforward(%d) : %s", fd, strerror(errno));

	return(asr_int(0, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the CASL seek() function.
*/

asr_t *casl_file_seek(asr_t *list) {	
	int fd;
	off_t offset;
	int whence = SEEK_SET;

	if(!list || !list->asr_kids[1] || !list->asr_kids[1]->asr_kids[0])
		error(E_USER, "too few arguments for seek()\n");

	fd = asr_getint_strict(list->asr_kids[0]);
	list = list->asr_kids[1];
	offset = asr_getint_strict (list->asr_kids[0]);
	list = list->asr_kids[1];
	if (list) {
		if (list->asr_kids[1]) {
			error(E_USER, "too many arguments for seek()\n");
		}
		whence = asr_getint_strict (list->asr_kids[0]);
	}

	if(lseek(fd, offset, whence) < 0)
		if (list)
			error(E_WARN, "seek(%d, %d, %d) : %s", fd, offset, whence, strerror(errno));
		else
			error(E_WARN, "seek(%d, %d) : %s", fd, offset, strerror(errno));

	return(asr_int(0, 0));
}

/* ----------------------------------------------------------------------------
** Builtin hook for the CASL close() function.
*/

asr_t *casl_file_close(asr_t *list) {	
	int fd;

	if(!list || !list->asr_kids[0])
		error(E_USER, "too few arguments for close()\n");

	if(list->asr_kids[1])
		error(E_USER, "too many arguments for close()\n");

	fd = asr_getint_strict(list->asr_kids[0]);

	if(close(fd) < 0)
		error(E_WARN, "close(%d) : %s", fd, strerror(errno));

	return(asr_int(0, 0));
}

/* ----------------------------------------------------------------------------
** An equivelent to fgets() for descriptors.
*/

static char *___fdgets(char *buffer, int size, int fd, int chop);

static char *fdgets(char *buffer, int size, int fd) {
	return(___fdgets(buffer, size, fd, 0));
}

static char *___fdgets(char *buffer, int size, int fd, int chop) {
	char *cp = buffer;
	int c = size - 1;
	int e;

	while(c) {
		e = read(fd, cp, 1);
		if(e < 0 || *cp == EOF) {
			if(cp == buffer)
				return(NULL);
			else {
				*cp = '\0';
				return(buffer);
			}
		}

		c -= 1;

		if(*cp == '\n') {
			if(chop) {
				*cp = '\0';
				if(cp != buffer && *(cp - 1) == '\r')
					*(cp - 1) = '\0';		
			} else
				*(cp + 1) = '\0';

			return(buffer);
		}
	
		cp += 1;
	}

	*cp = '\0';
	return(buffer);
}

/* ----------------------------------------------------------------------------
** Preform a single IO function until all the data is handled.
*/

static int atomicio(afiofp fn, int fd, void *buf, size_t len) {
  int tot = 0;
  int n;
  
  while ((size_t) tot < len) {
    n = (fn)(fd, buf + tot, len - tot);
    switch(n) {
    case 0:
      return (tot);
    case -1:
      if (errno == EAGAIN || errno == EINTR)
	continue;
      return (-1);
    default:
      tot += n;
    }
  }
  return (tot);
}
