/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#define SPECIAL_DEFINED

#include <sys/param.h>		/*for win32 PATH_MAX */
#include "casl.h"

/* 
** Main. Start the ball rolling. 
*/

extern int casl_scan();
extern int yydebug;
extern FILE *caslin;
extern int casl_line;
extern char *casl_file;
extern int yydebug;
extern int casl_parse(void);

extern asr_t *Top;

char *CASL_BaseDir = CASL_BASE_DIR;
char *CASL_Script = "(this script)";

char *program = NULL;
int ErrorFound = 0;
int AlwaysArp = 0;

table_t *Specials = NULL;

extern int StopFree;

int Debug = 0;
int RegressionTest = 0;

static void fix_path(void);
static void setprogname(char *pn);
static char **getargs(char *prog, char *args, int *i);

/* ----------------------------------------------------------------------------
** This is very simplistic handling of errors.
*/

void yyerror(char *s) {
	printf("Parse error in file \"%s\" near line %d:\n\t%s\n", casl_file, casl_line, 
	       s ? s : "<generic parse error>");
	ErrorFound = 1;
}

/* ----------------------------------------------------------------------------
*/

int Print_Status = 0;

void args(char **av);
void print_resources(void);

static FILE *preprocess(char *filename, char *cppextra);
static void strip_hashbang(int fd);
static char *master_config(void);
static void print_interface_list(void);
static void initialize(int argc, char **argv);

#ifdef RELEASE
static void die_pleasantly(int sig);
#else
void debug_death(int sig);
#endif

/* ---------------------------------------------------------------------------
*/

/*
 * The special NULL variable 
 */

asr_t *null = NULL;

/* ---------------------------------------------------------------------------
** Start the dance; do some basic initialization, toss the input file through
** CPP, parse, and interpret.
*/

extern int coptind;
extern char *coptarg;
extern int copterr;

extern int DebugOnError;

int main(int argc, char **argv) {
	char *fn = NULL;
	char *cp = NULL;
	char *cppextra = NULL;
	FILE *fp;
	int c;
	/* jump the the debugger on any of these */

#ifdef RELEASE
	signal(SIGSEGV, die_pleasantly);
	signal(SIGBUS, die_pleasantly);
#else
	signal(SIGSEGV, debug_death);
	signal(SIGBUS, debug_death);
#endif
	
	signal(SIGCHLD, SIG_IGN);

	setprogname(argv[0]);
	
	if(argv[1] && !strncmp(argv[1], "kernel", 6)) {
		char *file = argv[2];
		argv[1] += sizeof("kernel");
		argv = getargs(program, argv[1], &argc);
		argv[argc] = file;
		argc++;
		argv[argc] = NULL;
	}

	while((c = casl_getopt(argc, argv, "XaAC:sDdIi:g:m:M:Rr")) != EOF) {
		switch(c) {
		case 'X': 
			DebugOnError = 1;
			break;

		case 'a':
			AlwaysArp = 1;
			break;

		case 'C':
			cppextra = xstrdup(coptarg);
			break;

		case 'r':
			(void)atexit(print_resources);
			break;

		case 'R':
			RegressionTest = 1;
			break;

		case 'f':
			fn = xstrdup(coptarg);
			break;

		case 's':
			Print_Status++;
			break;

		case 'd':      
			Debug = 1;
			break;

		case 'A':
			StopFree = 1;
			break;

		case 'D':
			yydebug = 1;
			break;

		case 'I':
			print_interface_list();
			exit(EXIT_SUCCESS);
	
		case 'i':
			CASL_Default_Interface = xstrdup(coptarg);		       
			break;

		case 'g': 
			CASL_Default_Gateway = resolve(coptarg);
			if(CASL_Default_Gateway == INADDR_NONE) {
				fprintf(stderr, "Unable to resolve default gateway.\n");
				exit(EXIT_FAILURE);
			}

			break;

		case 'm': 
			CASL_Gateway_MAC = ether_aton(coptarg);
			if(!CASL_Gateway_MAC) {
				fprintf(stderr, "Malformed MAC address for gateway.\n");
				exit(EXIT_FAILURE);
			}

			CASL_Gateway_MAC = (struct ether_addr *)
				xbufdup((u_char *)CASL_Gateway_MAC, 
						   ETHER_ADDR_LEN);

			break;
		

		case 'M': 
			CASL_Local_MAC = ether_aton(coptarg);
			if(!CASL_Local_MAC) {
				fprintf(stderr, "Malformed MAC address for local.\n");
				exit(EXIT_FAILURE);
			}

			CASL_Local_MAC = (struct ether_addr *)
				xbufdup((u_char *)CASL_Local_MAC, 
						   ETHER_ADDR_LEN);

			break;
	       

		default:
			fprintf(stdout, "??\n");
			exit(EXIT_FAILURE);
		}
	}
    
	argv += coptind;
	argc -= coptind;

	if(!*argv) {
		asr_t *casl_sxs(asr_t *);

		printf(
"WARNING. You are in CASL standalone mode. You do\n"
"not have access to CASL's networking functionality.\n"
"In order to access the network, run CASL via \"caslsh\"\n"
"instead of on it's own.\n\n");

		initialize(argc, argv);
		casl_sxs(NULL);
		exit(0);
	} else 
		fn = xstrdup(*argv);
	
	if(*argv)
		argv += 1;

	casl_file = fn;
	CASL_Script = xstrdup(fn);

	if((cp = getenv(CASL_BASEDIR_ENV)))
		CASL_BaseDir = xstrdup(cp);

	fix_path();

	/* grab input from CPP (and include config file) */

	fp = preprocess(fn, cppextra);
	if(!fp) {
		perror("lexer open");
		exit(EXIT_FAILURE);
	}

	/* tell LEX about it */

	caslin = fp;

	/* parse the program */

	while(casl_parse()) 
		/* PARSE PARSE PARSE */ ;

	/* can we execute ? */

	if(ErrorFound || !Top) {
		fprintf(stderr, "\n"
"The parser encountered errors and was unable to completely parse this script.\n"
"CASL is thus unable to run your program. Sorry.\n"); 

		exit(EXIT_FAILURE);
	}

	initialize(argc, argv);

	/* start executing */

	eval_statement(Top);
		
	st_globalfree ();

	exit(EXIT_SUCCESS);
}

/* ---------------------------------------------------------------------------
** Jump to the debugger.
*/

void debug_death(int sig) {
	if(RegressionTest) 
		abort();

	debugger();
}

#ifdef RELEASE

/* ---------------------------------------------------------------------------
** Die with a more or less pleasant error.
*/

static void die_pleasantly(int sig) {
	fprintf(stderr, "[ Caught Signal %d ]\n", sig);
	fputs(INTERNAL_ERROR, stderr);
	fflush(stderr);
	
	exit(EXIT_FAILURE);
}

#endif

/* ---------------------------------------------------------------------------
** Get the path to cpp.
*/

static char *make_cpppath(char *progname)
{
    char *retbuf, *p;
    int len;

    /* take the program name, and look for cpp in the same directory */
    if(progname && (p = strrchr(progname, '/'))) {
        len = p - progname;
        retbuf = malloc(len + sizeof "/cpp");
        if(retbuf) {
            sprintf(retbuf, "%.*s/cpp", len, progname);
            return retbuf;
        }
    }
    return "cpp";
}

/* ---------------------------------------------------------------------------
** open the input file and preprocess (strip control line, pass it through cpp)
*/

static FILE *preprocess(char *filename, char *cppextra) {
	char buffer[8192];

	char *cppargs[255];
	char *cppcmd;
	char *cpppath;
	char *cp;
	char *ptr;
	
	char *globalconfig;

	int count = 0;
	int to[2];
	int from[2];

	int fd;
	int l;

	/* make sure the file works */

	if(strcmp(filename, "-")) {
		fd = open(filename, O_RDONLY);
		if(fd < 0) {	
			fprintf(stderr, "open \"%s\": %s\n", filename, 
					strerror(errno));
			exit(EXIT_FAILURE);
		}
	} else
		fd = fileno(stdin);

	/* strip off hashbang */

	strip_hashbang(fd);

	/* figure out how we call CPP */
	
	l = 0;

	cppcmd = xstrdup(CPPCOMMAND);

	while((cp = strsep(&cppcmd, " \t"))) {
		cppargs[count++] = xstrdup(cp);
		if(++l == 1) {
			char path[PATH_MAX];
			
			sprintf(path, "-I%s/include", CASL_BaseDir);

			cppargs[count++] = xstrdup(path);

			if (cppextra)
				while ((ptr = strsep(&cppextra, " \t")))
					cppargs[count++] = xstrdup(ptr);
		}
	}

	free(cppcmd);

	/* execute CPP as a pipe */

	pipe(to);
	pipe(from);

	switch(fork()) {
	case -1:
		perror("fork");
		exit(EXIT_FAILURE);
		
	case 0:
		dup2(to[0], fileno(stdin));
		close(to[0]);
		close(to[1]);

		dup2(from[1], fileno(stdout));
		close(from[0]);
		close(from[1]);

		cpppath = make_cpppath(program);
		execvp(cpppath, cppargs);
		execvp(CPPFILE, cppargs);
		execvp("./cpp", cppargs);
		fprintf(stderr, "Unable to execute C Preprocessor at \"%s\": %s\n",
			CPPFILE, strerror(errno));

		exit(EXIT_FAILURE);

	default:
		close(to[0]);
		close(from[1]);

		/* fake an include for the global config if it exists */

		if((globalconfig = master_config())) {
			sprintf(buffer, "#include <%s>\n", globalconfig);
			write(to[1], buffer, strlen(buffer));
		}

		/* feed the input file to CPP */

		do {
			if((l = read(fd, buffer, 8192)) > 0)
				write(to[1], buffer, l);
		} while(l > 0);

		close(to[1]);
		
		/* grab the output */

		return(fdopen(from[0], "r"));
	}

	assert(0);
	return(NULL);
}

/* ---------------------------------------------------------------------------
** if the first line of the script is "#!"* then we're intended to be executed
** implicitly by the kernel (executable script). The kernel called the script
** with the args on this line, but did not strip the line off; cpp is going to
** give us grief for having it here, so strip the line off input before
** presenting the fd to the scanner.
*/

static void strip_hashbang(int fd) {
	char c[2];

	if(read(fd, c, 2) < 0) {
		perror("read");
		exit(EXIT_FAILURE);
	}

	if(c[0] != '#' || c[1] != '!') {
		lseek(fd, SEEK_SET, 0);
		return;
	}

	casl_line = 2;

	for(;;) {
		if(read(fd, c, 1) < 0) {
			perror("read");
			exit(EXIT_FAILURE);
		}
		
		if(c[0] == '\n')
			return;
	}
}
	
/* ---------------------------------------------------------------------------
** Print a debug message.
*/

void Dprintf(char *fmt, ...) {
	va_list ap;

	if(!Debug)
		return;

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);

	return;
}	       
	
/* ---------------------------------------------------------------------------
Convert the arguments to an asr node tree.
*/

void args(char **av) {
	int i;
	asr_t *head;
	asr_t **lp = &head;

	if(av[0]) {
		for(i = 0; av[i]; i++) {
			asr_t *str = asr_string(av[i], 1);
			*lp = asr_node(LIST, NULL, NULL);
			(*lp)->asr_kids[0] = str;
			lp = &(*lp)->asr_kids[1];
		}
		
		*lp = NULL;

		st_insert("args", head, 0);
		st_insert("argc", asr_int(i, 0), 0);
	} else {
		st_insert("args", 
			asr_node(LIST, asr_int(0, 0), NULL), 0);
		st_insert("argc", asr_int(0, 0), 0);
	}


	return;       
}

/* ---------------------------------------------------------------------------
** Print a list of interfaces on the machine.
*/

static void print_interface_list() {
	struct iflist *list = get_iflist ();

	if(!list) {
		fprintf(stderr, "Unable to list interfaces on the system.\n");
		exit(EXIT_FAILURE);
	}
	
	for(/**/; list; list = list->next) {
		fprintf(stdout, "%s (%s)\n", list->name, addr_ntoa(list->ipaddr));
	}
	
	return;
}

/* ---------------------------------------------------------------------------
** Print resource usage.  Currently only on BSD.
*/

#ifdef _BSD
#include <sys/resource.h>
#endif

void print_resources() {
#ifndef _BSD
	printf("Resource totals not available in this OS\n");
	return;
#else
	struct rusage ru;

	if(getrusage(RUSAGE_SELF, &ru) < 0) {
		perror("getrusage");
		return;
	}

	printf(
	       "\n"
	       "-------- USAGE STATISTICS ---------\n"
	       "%lu/%lu utime\n"
	       "%lu/%lu stime\n"
	       "--------------------------------\n"
	       "%ld max rss\n"
	       "%ld shared text\n"
	       "%ld unshared data\n"
	       "%ld unshared stack\n"
	       "--------------------------------\n"
	       "%ld page reclaims\n"
	       "%ld page faults\n"
	       "%ld swaps\n"
	       "--------------------------------\n"
	       "%ld blockin\n"
	       "%ld blockout\n"
	       "--------------------------------\n"
	       "%ld vswtch\n"
	       "%ld ivswtch\n",

	       ru.ru_utime.tv_sec, ru.ru_utime.tv_usec,
	       ru.ru_stime.tv_sec, ru.ru_stime.tv_usec,
	       ru.ru_maxrss, 
	       ru.ru_ixrss,
	       ru.ru_idrss,
	       ru.ru_isrss, 
	       ru.ru_minflt,
	       ru.ru_majflt,
	       ru.ru_nswap,
	       ru.ru_inblock,
	       ru.ru_oublock,
	       ru.ru_nvcsw,
	       ru.ru_nivcsw);

	return;
#endif
}

/* ---------------------------------------------------------------------------
** Return the full pathname to the master config.
*/

static char *master_config() {
	static char path[PATH_MAX];
	struct stat sb;

	sprintf(path, "%s/include/%s", CASL_BaseDir, MASTER_CONFIG);
	
	if(stat(path, &sb) < 0) {
		fprintf(stderr, "Warning: CASL configuration not present in %s\n", path);
		return(NULL);
	}

	strcpy(path, MASTER_CONFIG);
	return(path);     
}

/* ---------------------------------------------------------------------------
** Set the PATH environment.
*/

static void fix_path(void) {
	char buffer[8192];
	int i;

	char *path = getenv("PATH");
	
	if(path)
		strncpy(buffer, path, 8192);
	else
		path[0] = 0;

	i = strlen(buffer);

	snprintf(&buffer[i], 8192 - i, ":%s/bin", CASL_BaseDir);

	setenv("PATH", buffer, 1);

	return;
}

#define MAXARGS 	100

/* ---------------------------------------------------------------------------
** Return the list of arguments passed from the kernel to formal arguments.
*/

static char **getargs(char *prog, char *args, int *argc) {
	static char *argv[MAXARGS];
	int i = 0;
	char *cp;
	
	argv[i++] = prog;

	while((cp = strsep(&args, " \t")) && i < MAXARGS - 1) {
		if(!*cp) 
			continue;

		argv[i++] = cp;
	}
	
	argv[i] = NULL;
	*argc = i;

	return(argv);
}

/* ---------------------------------------------------------------------------
** Initialize various aspects of the interpreter.
*/

static void initialize(int argc, char **argv) {
	/* add special variables */

	Specials = t_new(50, NULL, NULL);
	if(!Specials) {
		fprintf(stderr, "Unable to create table (internal error).\n");
		exit(EXIT_FAILURE);
	} else {
		int si;

		for(si = 0; SpecialTab[si].name; si++) 
			t_put(Specials, SpecialTab[si].name, SpecialTab[si].handler);
	}       

	args(argv);

	st_insert("__CASL_INFO__", asr_string(CASL_INFO, 1), 0);

	/* add the NULL variable */

	null = asr_buffer(0, "NULL");
	alloc_upref (null);
	st_insert(NULL_VARIABLE, null, 0);

	/* add builtin (C-language) functions to the symbol table */

	add_builtins();

	return;
}

/* ----------------------------------------------------------------------------
** Set the program name.
*/

#ifdef _WIN32
char *__progname = "casl";
#endif

static void setprogname(char *pn) {
#ifdef _WIN32
	__progname = xstrdup(pn);
#endif

	program = pn;

	return;
}
