/*
 * Parts of this file are copyright David R. Hanson, from
 * "C Interfaces and Implementations"
 *
 *
 * The author of this software is David R. Hanson.
 * 
 * Copyright (c) 1994,1995,1996,1997 by David R. Hanson. All Rights Reserved.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, subject to the provisions described below, without fee is
 * hereby granted, provided that this entire notice is included in all
 * copies of any software that is or includes a copy or modification of
 * this software and in all copies of the supporting documentation for
 * such software.
 * 
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY. IN PARTICULAR, THE AUTHOR DOES MAKE ANY REPRESENTATION OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 * 
 * David Hanson / drh@microsoft.com / http://www.research.microsoft.com/~drh/
 */

#include "list.h"
#include "strutil.h"

#include <stdarg.h>
#include <stddef.h>

/* append list at "tail" to "list" */

#define CACHE_MAX 2048
static list_t *list_cache = 0;
static int cachecount = 0;

/* ----------------------------------------------------------------------------
** Allocate a new list_t.  If there are any allocated but not used, use one of
** those first.  Only call malloc as a last resort.
*/

static inline list_t *new_list_t ()
{
	list_t *p = list_cache;
	if (p) {
		cachecount--;
		list_cache = list_cache->next;
	} else
		p = (list_t *) malloc(sizeof(struct list_s));
	return p;
}

/* ----------------------------------------------------------------------------
** Free a list_t type.  If there are not too many unused list_t allocated,
** save this in the list.  Otherwise, free it.
*/

static inline void free_list_t (list_t *entry)
{
	if (cachecount < CACHE_MAX) {
		entry->next = list_cache;
		list_cache = entry;
		cachecount++;
	} else
		free (entry);
}

/* ----------------------------------------------------------------------------
** Add an entry to the end of the list.
*/

list_t *l_append(list_t *list, list_t *tail) {
	list_t **lp = &list;

	while(*lp) 
		lp = &(*lp)->next;

	*lp = tail;

	return(list);
}

/* ----------------------------------------------------------------------------
** stick the value at "x" onto "list"
*/

list_t *l_push(list_t *list, void *x) {
	list_t *p = new_list_t ();

	p->data = x;
	p->next = list;

	return(p);
}

/* ----------------------------------------------------------------------------
** create a new list with a variable number of pointers
*/

list_t *l_list(void *x, ...) {
	va_list ap;
	list_t *list;
	list_t **lp = &list;

	va_start(ap, x);
	
	for( /* */ ; x; x = va_arg(ap, void *)) {
		*lp = new_list_t();
		(*lp)->data = x;
		lp = &(*lp)->next;
	}

	*lp = NULL;

	va_end(ap);

	return(list);
}

/* ----------------------------------------------------------------------------
** copy "list" into a new list
*/

list_t *l_copy(list_t *list) {
	list_t *head;
	list_t **lp = &head;

	for( /* */ ; list; list = list->next) {
		*lp = new_list_t();
		(*lp)->data = list->data;
		lp = &(*lp)->next;
	}

	*lp = NULL;

	return(head);
}

/* ----------------------------------------------------------------------------
** deallocate and return the value of the first node of "list"
*/

list_t *l_pop(list_t *list, void **x) {
	if(list) {
		list_t *head = list->next;
		if(x) 
			*x = list->data;

		free_list_t(list);
		return(head);
	} else
		return(list);
}

/* ----------------------------------------------------------------------------
** Delete a specefic node in the list.
*/

list_t *l_delete(list_t *list, list_t *victim) {
	list_t *savenode = 0, *node;

	if(!victim || !list)
		return(NULL);
	
	if(list == victim) {
		list = list->next;
		free_list_t(victim);
		return(list);
	}

	for(node = list; node; savenode = node, node = savenode->next) 
		if(node == victim)
			break;

	if(!node)
		return(NULL);

	node = node->next;
	savenode->next = node;
	free_list_t(victim);

	return(list);
}

/* ----------------------------------------------------------------------------
** flip "list" over
*/

list_t *l_reverse(list_t *list) {
	list_t *head = NULL; 
	list_t *next;

	for( /* */ ; list; list = next) {
		next = list->next;
		list->next = head;
		head = list;
	}

	return(head);
}

/* ----------------------------------------------------------------------------
** return the length of "list"
*/

int l_length(list_t *list) {
	int i;

	for(i = 0; list; list = list->next) 
		i++;

	return(i);
}

/* ----------------------------------------------------------------------------
** find a node in a linked list, return a pointer to it
*/

list_t *l_find(list, key, compare)
list_t *list;
void *key;
int compare(list_t *, const void *);
{
	list_t *node;

	assert(compare);

	if(!list)
		return(NULL);

	for(node = list; node; node = node->next) 
		if(compare(node, key))
			return(node);

	return(NULL);
}

/* ----------------------------------------------------------------------------
** free up the entire linked list
*/

void l_free(list_t **list) {
	list_t *next;

	assert(list);

	for( /* */ ; *list; *list = next) {
		next = (*list)->next;
		free_list_t(*list);
	}
}

/* ----------------------------------------------------------------------------
** call "apply" for the value of each node in the list. "cl" 
** is the value to use for the end of the list - leave it
** as "NULL" if you don't care.
*/

void l_map(list_t *list,
	   void apply(void **x, void *cl),
	   void *cl) {
	assert(apply);

	for( /* */ ; list; list = list->next) 
		apply(&list->data, cl);
}

/* ----------------------------------------------------------------------------
** return an array representing the value of each node in "list",
** with "END" being the value for the end of the list, or NULL. 
*/

void **l_2array(list_t *list, void *end) {
	int i;
	int l = l_length(list);

	void **array = (void *) malloc((l + 1) * sizeof(*array));

	for(i = 0; i < l; i++) {
		array[i] = list->data;
		list = list->next;
	}

	array[i] = end;

	return(array);
}
