/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/* 
 * Deal with expressions. Expressions are a subset of statements
 * that can be evaluated to some value; expressions are also 
 * components of control constructs. Most of the interesting stuff
 * in CASL is in expressions (function calls, structures, etc).
 */

#include "casl.h"
#include <setjmp.h>

extern int Debug;
extern asr_t *null;

static asr_t *eval_createlist(asr_t *exn);

static asr_t *compare_list(asr_t *left, asr_t *right, int op);
static asr_t *compare_buffer(asr_t *left, asr_t *right, int op);
static asr_t *complex_expr(asr_t *left, asr_t *right, int op);

/* ----------------------------------------------------------------------------
** Evaluate expressions.
** 
** This is one of the largest functions in the program, because expressions
** are such a core part of the language. An "expression" is anything that can
** be evaluated in the language to a specific value. Almost everything that
** is done in CASL is an expression.
**
** The routine is essentially comprised of two parts:
**
**	- take complex expressions and evaluate them individually. 
**	   "complex" expressions include anything that isn't a simple
** 	   math/logic operation.
**
** 	- do math/logic operations
**
** Math/logic is complicated by the presence of complex types and variable names.
** The breakout of that code is:
**
**	- retrieve values associated with variable names if we're passed
** 	   an identifier rather than a real value
**
**	- special-case comparison of lists, buffers, and strings so that 
** 	   those operations do the expected things.
**
**	- reduce values to integer variables we can work with directly in C
**
** 	- perform the actual math/logic operation
**
*/
	
asr_t *eval_expr(asr_t *exn) {
	asr_t *r = 0;
	asr_t *left, *right, *result;
       	unsigned long ai, bi;
	long sai, sbi;
	int negflags;

#ifdef TRACE_CALLS
	Dprintf("eval_expr(%p)\n");
#endif

	if(!exn)
		return(NULL);

	/* --------------------------------------------------------- 
	 *  PART 1
	 */

	/* there is no node type for "expression", and lots of 
 	  * things (calls, assignments, etc) can be evaluated 
	  * as an expression, so we need to catch the many 
	  * instances where something complex is being interpreted
	  * in the context of an expression.
	  */

	switch(exn->asr_type) {

		/* subroutine call */

	case N_CALL:
		return(eval_call(exn));

		/* assignment */

	case N_ASSIGN:
		return(eval_assign(exn));

		/* struct creation */
	
	case N_NEW:
		return(eval_new(exn));

		/* object copy */

	case N_COPY:
		return(eval_copy(exn));

		/* struct dereference */

	case N_EXTRACT:
		return(eval_extract(exn));

		/* grab a count of bytes from the packet. I didn't think this through very 
		 * well, this should have been in eval_extract, but oh well. 
		 */

	case N_EXTRACT2:
		return(eval_extract2(exn));

	case LIST: 
		return(eval_list(exn));

		/* negated expression */

	case N_NEG:
		error(E_INTERNAL, "negative numbers are not supported by the current CASL");

		/* 1 if false, 0 if true */

	case N_NOT:
		return(eval_not(exn));

		/* get the top element of the list */

	case N_POP:
		return(eval_head(exn));

		/* get the structure name of a buffer */

	case N_SNAME:
		return(eval_sname(exn));

		/* get the last element of a list */

	case N_TAIL:
		return(eval_tail(exn));

		/* insert at  the head of a list */

	case N_PLIST:
		return(eval_plist(exn));

		/* insert at the tail of the list */

	case N_PTAIL:
		return(eval_ptail(exn));

		/* read the function, it's tricky. */

	case N_IDX:
		return(eval_idx(exn));

		/* return the value associated with a symtab entry */

	case N_ID:
		return(eval_id(exn));

		/* passed a buffer as a string, someone's being nutty */
		
	case N_BUFFER:
		return(exn);

	case N_BNOT: {
		asr_t *result = asr_int (~asr_getint_strict (exn->asr_kids[0]), 0);
		alloc_downref (exn->asr_kids[0]);
		return (result);
	}


	case N_CREATELIST:
		return(eval_createlist(exn));

	case N_PREINCR:
		return(eval_incr(exn, 0));

	case N_PREDECR:
		return(eval_decr(exn, 0));

	case N_POSTINCR:
		return(eval_incr(exn, 1));

	case N_POSTDECR:
		return(eval_decr(exn, 1));

		/* just return the number */

	case N_INT:
	case N_NEGINT:
	case N_CHAR:
		return(exn);

	default:
		break;
	}

	/* --------------------------------------------------------- 
	 *  PART 2
	 */

	/* this is some arithmatic/logical operator. Deal with it. */

	left = exn->asr_kids[0];
	right = exn->asr_kids[1];
	
	/* see if we're operating on variable names; if so, retrieve their values */

	left = lookup(left);
	right = lookup(right);
	alloc_upref (left);
	alloc_upref (right);
	alloc_downref (exn->asr_kids[0]);
	alloc_downref (exn->asr_kids[1]);

	/* special-case buffers, strings and lists for comparison */

	if (exn->asr_type == N_NE || exn->asr_type == N_EQ) {
		switch(left->asr_type) {
		case LIST:
		case N_CREATELIST:
		case N_BUFFER:
		case N_COPY:	
			alloc_upref (left);
			alloc_upref (right);
			return(complex_expr(left, right, exn->asr_type));
		}

		switch(right->asr_type) {
		case LIST:
		case N_CREATELIST:
		case N_BUFFER:
		case N_COPY:
			alloc_upref (left);
			alloc_upref (right);
			return(complex_expr(left, right, exn->asr_type));
		}
	}

	/* reduce lhs/rhs to a value (in case rhs/lhs is a complex expression, not a value.
	 * I added this to deal with long chains of && and || operators; this recursively 
	 * evaluates those expressions left-to-right, so:
	 *
	 *	x && y && z && a
	 * 
	 * does
	 *
	 *	(((x && y) && z) && a)
	 */

	if(!asr_basetype(left)) {
		left = eval_expr(left);
	}

	/* deal with short-circuited && and || operators, because 
	 * it's important that we don't evaluate sub-expressions if
	 * a conclusion about the truth of the statement has already
	 * been reached.
	 */

	switch(exn->asr_type) {
	case N_AND:
		ai = asr_getint(left);
		if(!ai)
			result = (asr_int(0, 0));
		else if(asr_getint(right))
			result = (asr_int(1, 0));
		else
			result = (asr_int(0, 0));
		break;

	case N_OR:
		ai = asr_getint(left);
		if(ai)
			result = (asr_int(1, 0));
		else if(asr_getint(right))
			result = (asr_int(1, 0));
		else
			result = (asr_int(0, 0));
		break;

	default:
		result = 0;
		break;
	}     

	if (result) {
		alloc_downref (left);
		alloc_downref (right);
		return (result);
	}

	if (!asr_basetype (left))
		left = eval_expr (left);
	if (!asr_basetype (right))
		right = eval_expr (right);
	
	/* Most things ignore this. */
	negflags = 0;
	if (left->asr_type == N_NEGINT)
		negflags |= 1;
	if (right->asr_type == N_NEGINT)
		negflags |= 2;

	ai = asr_getint_strict(left);
	bi = asr_getint_strict(right);
	sai = (signed long)ai;
	sbi = (signed long)bi;

	alloc_downref (left);
	alloc_downref (right);

	switch(exn->asr_type) {
	case N_PLUS:
		r = asr_int(ai + bi, 0);
		if (negflags) {
			if (r->asr_sinteger < 0) r->asr_type = N_NEGINT;
		}
		return(r);

	case N_MINUS:
		r = asr_int(ai - bi, 0);

		if (negflags) {
			if (r->asr_sinteger < 0) r->asr_type = N_NEGINT;
		} else if (ai < bi) {
			r->asr_type = N_NEGINT;
		}

		return(r);

	case N_MUL:
		r = asr_int(ai * bi, 0);

		if (negflags == 1 || negflags == 2) {
			if (r->asr_integer) r->asr_type = N_NEGINT;
		}

		return(r);

	case N_DIV:
		if(!bi)
			error(E_USER, "Attempt to divide by zero");

		switch (negflags) {
		case 0:
			r = asr_int(ai / bi, 0);
			break;
		case 1:
			r = asr_int ((u_long)(sai / bi), 0);
			break;
		case 2:
			r = asr_int ((u_long)(ai / sbi), 0);
			break;
		case 3:
			r = asr_int ((u_long)(sai / sbi), 0);
			break;
		}
		if (((negflags & 3) == 1 || (negflags & 3) == 2) && r->asr_sinteger < 0)
			r->asr_type = N_NEGINT;
		return(r);

	case N_EQ:
		r = asr_int(ai == bi && (negflags & 2) >> 1 == (negflags & 1), 0);
		return(r);

	case N_NE:
		r = asr_int(ai != bi || (negflags & 2) >> 1 != (negflags & 1), 0);
		return(r);

	case N_GT:
		switch (negflags) {
		case 0:
		case 3:
			r = asr_int(ai > bi, 0);
			break;
		case 1:
			r = asr_int (0, 0);
			break;
		case 2:
			r = asr_int (1, 0);
			break;
		}
		return(r);

	case N_LT:
		switch (negflags) {
		case 0:
		case 3:
			r = asr_int(ai < bi, 0);
			break;
		case 1:
			r = asr_int (1, 0);
			break;
		case 2:
			r = asr_int (0, 0);
			break;
		}
		return(r);

	case N_GE:
		switch (negflags) {
		case 0:
		case 3:
			r = asr_int(ai >= bi, 0);
			break;
		case 1:
			r = asr_int (0, 0);
			break;
		case 2:
			r = asr_int (1, 0);
			break;
		}
		return(r);

	case N_LE:
		switch (negflags) {
		case 0:
		case 3:
			r = asr_int(ai <= bi, 0);
			break;
		case 1:
			r = asr_int (1, 0);
			break;
		case 2:
			r = asr_int (0, 0);
			break;
		}
		return(r);

	case N_MOD:
		if(!bi)
			error(E_USER, "Attempt to divide by zero");

		switch (negflags) {
		case 0:
			r = asr_int(ai % bi, 0);
			break;
		case 1:
			r = asr_int ((u_long)(sai % bi), 0);
			break;
		case 2:
			r = asr_int ((u_long)(ai % sbi), 0);
			break;
		case 3:
			r = asr_int ((u_long)(sai % sbi), 0);
			break;
		}
		if (negflags & 1 && r->asr_sinteger < 0) {
			r->asr_type = N_NEGINT;
		}
		return(r);

	case N_BLSHIFT:
		if (negflags & 1) {
			r = asr_int((u_long)(sai << bi), 0);
			if (r->asr_integer) r->asr_type = N_NEGINT;
		} else
			r = asr_int (ai << bi, 0);
		return(r);

	case N_BRSHIFT:
		if (negflags & 1) {
			r = asr_int((u_long)(sai >> bi), 0);
			if (r->asr_integer) r->asr_type = N_NEGINT;
		} else
			r = asr_int (sai >> bi, 0);
		return(r);

	case N_BOR:
		r = asr_int(ai | bi, 0);
		return(r);

	case N_BAND:
		r = asr_int(ai & bi, 0);
		return(r);

	case N_BXOR:
		r = asr_int(ai ^ bi, 0);
		return(r);

	default:
		/* XXX - simplify things a little bit, if we pass something
		 * wacky to eval_expr, just return zero.
		 */

		return(asr_int(0, 0));
	}

	assert(0);
	return(NULL);
}

/* ----------------------------------------------------------------------------
** return the node associated with an identifier. The return value of  
** this routine is passed directly through eval_expr, which is bad, as
** it adds another case to the interpretation of an eval_expr() call.
*/

asr_t *eval_id(asr_t *node) {
	asr_t *result;

#ifdef TRACE_CALLS
	Dprintf("eval_id(%p)\n", node);
#endif

	result = lookup(node);
	alloc_downref (node);
	alloc_upref (result);
	return (result);
}

/* ----------------------------------------------------------------------------
** evaluate assignment. the LHS of this production can only be either
** an id or an idextract; in either case, all this does is stuff the
** expression on the RHS in the symbol table using the id as a key.
**
** This is one of the spots in the CASL code that is gradually collecting
** code, since assignments are the focal point for allocation in CASL and
** are special-cased in a number of different ways.
*/

extern table_t *Specials;
asr_t *eval_assign(asr_t *node) {
	void (*handler)(asr_t *) = NULL;
	asr_t *sym = node->asr_kids[0];
	asr_t *value = node->asr_kids[1];
	asr_t *result;
	char *id;

#ifdef TRACE_CALLS
	Dprintf("eval_assign(%p)\n", node);
#endif

	value = lookup(value);
	alloc_upref (value);
	alloc_downref (node->asr_kids[1]);

	switch(value->asr_type) {
	case N_BUFFER:
	case LIST:

		/* assign reference */

		break;

	default:
		/* assign copy */

		result = eval_expr (value);
		if(!result)
			result = asr_int(0, 0);
		value = result;
		break;
	}

	if(sym->asr_type == N_IDX) {

		/* no extra reference; this will never assign an ASR */

		result = eval_assign_idx (sym, value);
		return (result);
	}

	/* catch an attempt to set a "special" variable (something with
	 * side effects to the CAPE engine). Do this only if the variable has
	 * already been set (ie, we're changing it after casl.conf set).
	 */

	if(st_global(sym->asr_ident) &&
	     (handler = t_get(Specials, sym->asr_ident))) {
		handler(value);
		alloc_downref (sym);
		return(value);
	}

	id = sym->asr_ident;

	/* global symbol up-reference (survive scope change) */

	if(st_get(id, Level) != value) {
		alloc_upref(value);

		st_replace(id, value, Level);
	}

	return(value);
}

/* ----------------------------------------------------------------------------
** Increment a variable.  The flag post sets if it's a post-increment, meaning
** the value before increment should be returned.
*/

asr_t *eval_incr(asr_t *exn, int post) {
	void (*handler)(asr_t *) = NULL;
	asr_t *sym = exn->asr_kids[0];
	asr_t *value = 0;
	asr_t *rvalue = 0;
	asr_t *result = 0;
	u_long intvalue;
	char *id;

#ifdef TRACE_CALLS
	Dprintf ("eval_incr(%p, %d)\n", exn, post);
#endif

	if (sym->asr_type != N_ID && sym->asr_type != N_IDX) {
		sym = eval_expr (sym);
	}
	if (sym->asr_type != N_ID && sym->asr_type != N_IDX) {
		error (E_USER, "attempt to increment direct value");
	}

	/* Increment a variable.  This is the easy part. */
	if (sym->asr_type == N_ID) {
		id = sym->asr_ident;
		value = lookup (sym);
		switch (value->asr_type) {
		case N_INT:
			intvalue = value->asr_integer + 1;
			result = asr_int (intvalue, 0);
			break;
		case N_NEGINT:
			intvalue = value->asr_integer + 1;
			result = asr_int (intvalue, 0);
			if (intvalue)
				result->asr_type = N_NEGINT;
			break;
		case N_CHAR:
			result = asr_char (value->asr_character + 1, 0);
			break;
		default:
			error (E_USER, "attempt to increment non-integral value");
		}
		if (post) {
			rvalue = value;
		} else {
			rvalue = result;
		}
		alloc_upref (rvalue);
		if (st_global (id) && (handler = t_get (Specials, id))) {
			handler (result);
			alloc_downref (result);
		} else {
			st_replace (id, result, Level);
		}
	} else {
		/* It's not a variable.  It must be an IDX.  Leave this to the
		 * pros and just call eval_idx and eval_assign_idx
		 */
		alloc_upref (sym);
		value = eval_idx (sym);

		result = asr_int (value->asr_integer + 1, 0);
		if (post)
			rvalue = value;
		else {
			alloc_downref (value);
			rvalue = result;
			alloc_upref (rvalue);
		}

		alloc_upref (sym);
		alloc_downref (eval_assign_idx (sym, result));
	}
	return (rvalue);
}

/* ----------------------------------------------------------------------------
** Decrement the value of a variable.  Perhaps this could be combined with
** eval_incr in some way.
*/

asr_t *eval_decr(asr_t *exn, int post) {
	void (*handler)(asr_t *) = NULL;
	asr_t *sym = exn->asr_kids[0];
	asr_t *value = 0;
	asr_t *rvalue = 0;
	asr_t *result = 0;
	u_long intvalue;
	char *id;

#ifdef TRACE_CALLS
	Dprintf ("eval_decr(%p, %d)\n", exn, post);
#endif

	if (sym->asr_type != N_ID && sym->asr_type != N_IDX) {
		sym = eval_expr (sym);
	}
	if (sym->asr_type != N_ID && sym->asr_type != N_IDX) {
		error (E_USER, "attempt to decrement direct value");
	}

	if (sym->asr_type == N_ID) {
		id = sym->asr_ident;
		value = lookup (sym);
		switch (value->asr_type) {
		case N_INT:
			intvalue = value->asr_integer - 1;
			result = asr_int (intvalue, 0);
			if (!value->asr_integer) {
				result->asr_type = N_NEGINT;
			}
			break;
		case N_NEGINT:
			intvalue = value->asr_integer - 1;
			result = asr_int (intvalue, 0);
			if (result->asr_sinteger < 0)
				result->asr_type = N_NEGINT;
			break;
		case N_CHAR:
			result = asr_char (value->asr_character - 1, 0);
			break;
		default:
			error (E_USER, "attempt to decrement non-integral value");
		}
		if (post) {
			rvalue = value;
		} else {
			rvalue = result;
		}
		alloc_upref (rvalue);
		if (st_global (id) && (handler = t_get (Specials, id))) {
			handler (result);
			alloc_downref (result);
		} else {
			st_replace (id, result, Level);
		}
	} else {
		alloc_upref (sym);
		value = eval_idx (sym);

		result = asr_int (value->asr_integer - 1, 0);
		if (post)
			rvalue = value;
		else {
			alloc_downref (value);
			rvalue = result;
			alloc_upref (rvalue);
		}

		alloc_upref (sym);
		alloc_downref (eval_assign_idx (sym, result));
	}
	return (rvalue);
}

/* ----------------------------------------------------------------------------
** Create a copy of a node.  This really only does something for lists and
** buffers.
*/

asr_t *eval_copy(asr_t *exn) {
	asr_t *result;
	exn = exn->asr_kids[0];
      
	switch(exn->asr_type) {
	case N_INT:
	case N_NEGINT:
	case N_CHAR:
	case N_BUFFER:
	case LIST:
		break;

	default:
		alloc_upref (exn);
		result = eval_expr(exn);
		alloc_downref (exn);
		exn = result;
		break;
	}

	result = asr_copy (exn);
	alloc_downref (exn);
	return (result);
}

/* ----------------------------------------------------------------------------
** This may seem like a silly function to have, but it goes through and makes
** sure that the list contains only data values and not variables.
*/

asr_t *eval_list(asr_t *list) {
	asr_t *lp, *t;

	for(lp = list; lp && lp->asr_kids[0]; lp = lp->asr_kids[1]) {
		t = lp->asr_kids[0];

		t = lookup(t);
		alloc_upref (t);
		alloc_downref (lp->asr_kids[0]);

		lp->asr_kids[0] = t;
	}

	return(list);
}

/* ----------------------------------------------------------------------------
** Does a logical not of an expression and returns a integer.
*/

asr_t *eval_not(asr_t *node) {
	asr_t *a = eval_expr(node->asr_kids[0]);
	u_long i;

#ifdef TRACE_CALLS
	Dprintf("eval_not(%p)\n", node);
#endif

	i = asr_getint(a);
	alloc_downref (a);

	return(asr_int(!i, 0));
}

/* ----------------------------------------------------------------------------
** Evaluate a complex expression for non integers.
*/

static asr_t *complex_expr(asr_t *left, asr_t *right, int op) {
	int type = -1;

	left = eval_expr(left);
	right = eval_expr(right);

	switch(left->asr_type) {
	case LIST:
		type = LIST;
		break;

	case N_BUFFER:
		type = N_BUFFER;
		break;

	default:
		switch(right->asr_type) {
		case LIST:
			type = LIST;
			break;
	
		case N_BUFFER:
			type = N_BUFFER;
			break;
		
		default:
			assert(0);
		}

		break;
	}	

	switch(op) {
	case N_NE:
	case N_EQ:
		break;

	default:
		error(E_USER, "Attempt to perform arithmatic on nonscalar types.");
	}

	switch(type) {
		case LIST:
			return(compare_list(left, right, op));
		
		case N_BUFFER:
			return(compare_buffer(left, right, op));
	
		default:
			assert(0);

	}

	/* NOTREACHED */
	return(NULL);
}

/* ----------------------------------------------------------------------------
** Compares a pair of lists.  They are not equal if they are different
** lengths, or contain different data.
*/

static asr_t *compare_list(asr_t *left, asr_t *right, int op) {
	asr_t *a, *b;
	int truth = -1;	

	a = b = NULL;

	if(left->asr_type == LIST)
		a = left;

	if(right->asr_type == LIST)
		b = right;

	if(a && b) 
		truth = list_eq(a, b);

	if(!a && left == null) 
		truth = list_items(b) ? 0 : 1;
		
	if(!b && right == null) 
		truth = list_items(a) ? 0 : 1;

	if (left)
		alloc_downref (left);
	if (right)
		alloc_downref (right);

	if(truth != -1) {
		if(op == N_EQ)
			return(asr_int(truth ? 1 : 0, 0));
		else
			return(asr_int(truth ? 0 : 1, 0));
	} 

	if(op == N_EQ)
		return(asr_int(0, 0));
	else if(op == N_NE)
		return(asr_int(1, 0));
	else
		error(E_USER, "comparison between incompatible types");

	/* NOTREACHED */
	return(NULL);
}

/* ----------------------------------------------------------------------------
** Compare a pair of buffers.  This is a simple memcmp.
*/

static asr_t *compare_buffer(asr_t *left, asr_t *right, int op) {
	asr_t *a, *b;
	int truth = -1;	

	a = b = NULL;

	if(left->asr_type == N_BUFFER)
		a = left;

	if(right->asr_type == N_BUFFER)
		b = right;

	if(a && b) {
		if(a->asr_size != b->asr_size)
			truth = 0;
		else	
			truth = memcmp(a->asr_buf, b->asr_buf, a->asr_size) ? 0 : 1;
	}

	if(!a && left == null) 
		truth = b->asr_size ? 0 : 1;
		
	if(!b && right == null) 
		truth = a->asr_size ? 0 : 1;

	if (left)
		alloc_downref (left);
	if (right)
		alloc_downref (right);

	if(truth != -1) {
		if(op == N_EQ)
			return(asr_int(truth ? 1 : 0, 0));
		else
			return(asr_int(truth ? 0 : 1, 0));
	}

	if(op == N_EQ)
		return(asr_int(0, 0));
	else if(op == N_NE)
		return(asr_int(1, 0));
	else
		error(E_USER, "comparison between incompatible types");

	/* NOTREACHED */
	return(NULL);
}

/* ----------------------------------------------------------------------------
** Do not allow users to assign the null variable.
*/

void special_null_variable(asr_t *val) {

	error(E_USER, "Attempt to assign to the null variable.");
	
	return;
}

/* ----------------------------------------------------------------------------
** Take a list from the parser and turn it into a data list.  This takes care
** of evaluating any expressions in the list.
*/

static asr_t *eval_createlist(asr_t *exn) {
	asr_t *list = exn->asr_kids[0];
	asr_t *head;
	asr_t **lp = &head;

	for(/**/; list; list = list->asr_kids[1]) {
		*lp = asr_node(LIST, NULL, NULL);
		if(list->asr_kids[0]) {
			alloc_upref (list->asr_kids[0]);
			(*lp)->asr_kids[0] = eval_expr(list->asr_kids[0]);
		} else
			(*lp)->asr_kids[0] = NULL;

		lp = &(*lp)->asr_kids[1];
	}

	*lp = NULL;
	return(head);
}
