/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"
#include "casl-conf.h"
#include "fileio.h"

#include <sys/time.h>

/* Deal with "built-in" functions of the CASL language. Builtins are
 * functions callable from within a script that invoke a pre-written
 * C function instead of a CASL subroutine. Builtins are used to 
 * provide all OS/IO functions; builtin functions have complete control
 * over the entire interpreter using all the CASL interpretation
 * routines. 
 */

struct bt_s {
	char *id;
	int (*init)(void *x);
	asr_t *(*bft)(asr_t *);
};

/* ==================================================================== */

/* The Builtin Function Table. Every C function that can be called 
 * directly from a CASL script is entered here, with the name of the
 * function from a casl script and it's corresponding C function name.
 */

asr_t *casl_callstack(asr_t *args);
asr_t *casl_gtod(asr_t *);
asr_t *casl_print(asr_t *);
asr_t *casl_timer_stop(asr_t *args);
asr_t *casl_timer_start(asr_t *args);
asr_t *casl_size(asr_t *args);
asr_t *casl_tobuf(asr_t *args);
asr_t *casl_atoi(asr_t *args);
asr_t *casl_memusage(asr_t *args);
asr_t *casl_wait(asr_t *args);
asr_t *casl_getenv(asr_t *arg);
asr_t *casl_setenv(asr_t *arg);
asr_t *casl_strep(asr_t *arg);
extern asr_t *casl_rand(asr_t *arg);
extern asr_t *casl_sxs(asr_t *arg);

static asr_t *casl_exit(asr_t *a);

static asr_t *casl_putip(asr_t *a);
static asr_t *casl_getip(asr_t *ap);

#ifdef HAVE_LIBGC
asr_t *casl_gcdump(asr_t *args);
asr_t *casl_collect(asr_t *args);
#endif

static struct timeval *tvsub(struct timeval *out, struct timeval *in);
static int ms_delta(struct timeval *start, struct timeval *stop);

/* this table is automatically generated from "gen-builtins" */

static struct bt_s builtin_tab[] = {

	/* INSERT TABLE HERE */

};

/* ==================================================================== */


/* ----------------------------------------------------------------------------
** Convert a string or a list representing a string into an integer.
*/

asr_t *casl_atoi(asr_t *args) {
	asr_t *arg;
	u_long n;
	char *string = 0;

	if(!args || !args->asr_kids[0])
		error(E_USER, "too few arguments for atoi()");
	
	if(args->asr_kids[1])
		error(E_USER, "too many arguments for atoi()");

	alloc_upref (args->asr_kids[0]);
	arg = eval_expr(args->asr_kids[0]);

	if(arg->asr_type == LIST)
		string = list_to_string(arg);
	else if(arg->asr_type == N_BUFFER)
		string = buf2asciiz (arg);
	else
		error(E_USER, "argument to atoi() must represent a string");

	n = string_to_ulong(string);

	free(string);

	alloc_downref (arg);
	return(asr_int(n, 0));
}

/* ----------------------------------------------------------------------------
** Convert a string representation of an IP address into an integer IP address.
*/

static asr_t *casl_getip(asr_t *args) {
	asr_t *arg;
	u_int32_t addr;
	char *str;

	if(!args && !args->asr_kids[0])
		error(E_USER, "too few arguments for getip()");

	if(args->asr_kids[1])
		error(E_USER, "too many arguments for getip()");

	arg = args->asr_kids[0];

	if(arg->asr_type != N_BUFFER)
		error (E_USER, "bad argument for getip()");

	str = buf2asciiz (arg);
	addr = inet_addr(str);
	xfree ((void **)&str);

	if (addr == INADDR_NONE)
		return(asr_int(0, 0));	

	return(asr_int(ntohl(addr), 0));
}

/* ----------------------------------------------------------------------------
** Stop parsing and exit immediately, giving a return code.
*/

static asr_t *casl_exit(asr_t *a) {
	int ev;

	if(a && a->asr_kids[1])
		error(E_USER, "too many arguments for exit()");

	if (a && a->asr_kids[0]->asr_type != N_INT && a->asr_kids[0]->asr_type != N_NEGINT)
		error (E_USER, "bad arguments for exit()");

	if(!a || !a->asr_kids[0])
		ev = 0;
	else {
		ev = asr_getint(a->asr_kids[0]);
	}

	exit(ev);
}
       
/* ----------------------------------------------------------------------------
** Turn an integer into a string representation of an IP addres.
*/

static asr_t *casl_putip(asr_t *a) {
	u_int32_t addr;
	char string[40];

	if(!a && !a->asr_kids[0])
		error(E_USER, "too few arguments for putip()");

	if(a->asr_kids[1])
		error(E_USER, "too many arguments for putip()");

	a = a->asr_kids[0];

	if (a->asr_type != N_INT && a->asr_type != N_NEGINT) {
		error (E_USER, "bad argument for putip()");
	}

	addr = asr_getint(a);

	sprintf(string, "%s", addr_ntoa(htonl(addr)));

	return(asr_string(string, 0));
}

/* ----------------------------------------------------------------------------
** Traverse the asr node list and print a simple string representation.
** List type nodes are handled recursively.
*/

static void print_asr_pretty(asr_t *arg)
{
	switch(arg->asr_type) {
	case N_INT:
		fprintf(stdout, "%lu ", arg->asr_integer);
		break;

	case N_NEGINT:
		fprintf(stdout, "%ld ", arg->asr_sinteger);
		break;

	case N_CHAR:
		fprintf(stdout, "%c ", arg->asr_character);
		break;

	case N_ID:
		fprintf(stdout, "%s ", arg->asr_ident);
		break;

	case N_BUFFER: 
		if(arg == null) {
			fprintf(stdout, "0 ");
			break;
		}
	
		fwrite(arg->asr_buf, 
		       arg->asr_size, 
		       1, stdout);
		fprintf(stdout, " ");

		break;


	case LIST:
		while(arg) {
			if(!arg->asr_kids[0])
				break;

			print_asr_pretty(arg->asr_kids[0]);
			arg = arg->asr_kids[1];
		}				
		break;

	default:
		fprintf(stdout, "(internal node) ");
		break;
	}
}

/* ----------------------------------------------------------------------------
** An example builtin function.  Print out each node in the list "arg".
*/

asr_t *casl_print(asr_t *arg) {
	if (!arg) return (NULL);

	print_asr_pretty (arg);
	fprintf (stdout, "\n");

	return(NULL);
}

/* ----------------------------------------------------------------------------
** Pause execution for a given number of seconds.
*/

asr_t *casl_wait(asr_t *args) {
	u_long s; 

	if(!args || !args->asr_kids[0])
		error(E_USER, "too few arguments for wait()");

	if(args->asr_kids[1])
		error(E_USER, "too many arguments for wait()");

	s = asr_getint(args->asr_kids[0]);
	sleep(s);

	return(NULL);
}     

/* ----------------------------------------------------------------------------
** Print the current memory usage.  This needs to be handled for other OSs
** than BSD...
*/

#ifdef _BSD
#include <sys/resource.h>
#endif

asr_t *casl_memusage(asr_t *args) {
#ifndef _BSD
	printf(">>> Memory usage not available\n");
	return(NULL);
#else
	static int count = 0;

	struct rusage ru;

	if(getrusage(RUSAGE_SELF, &ru) < 0) {
		printf(">>> Unable to obtain memory usage: %s\n", strerror(errno));
		return(NULL);
	}

	printf(">>> [%d] rss %ld text %ld data %ld stack %ld\n",
	        count++,
		ru.ru_maxrss, ru.ru_ixrss, ru.ru_idrss, ru.ru_isrss);

	return(NULL);
#endif
}

/* ----------------------------------------------------------------------------
** This adds all the builtins to the symbol table at startup.
*/

void add_builtins(void) {
	struct bt_s *b = builtin_tab;
	char buf[8192];

	while(b->id) {
		snprintf(buf, 8192, "%s%s", CASL_FUNC_PREFIX, b->id);
		buf[8191] = 0;

		if(b->init) {
			if(b->init(NULL) < 0) {
				error(E_WARN, "unable to initialize function %s", b->id);
				continue;
			}
		}

       		bt_insert(xstrdup(buf), b->bft);
		b++;
	}

	return;
}

/* ----------------------------------------------------------------------------
** This simplifies builtin functions a little by pre-processing their argumet
** list and passing something that only contains direct data.
*/

asr_t *builtin_callargs(asr_t *ap) {
	asr_t *v, *n, *a;
	asr_t *head = NULL;
	asr_t **tail = &head;

	for(n = ap; n; n = n->asr_kids[1]) {
		a = n->asr_kids[0];
		alloc_upref (a);

		switch(a->asr_type) {
		case N_BUFFER:
		case LIST:
			v = a;
	
		default:
			v = eval_expr(a);
			break;
		}

		*tail = asr_node (LIST, v, NULL);
		tail = &((*tail)->asr_kids[1]);
	}

	return(head);
}

/* ----------------------------------------------------------------------------
** Free the arg list.
*/

void builtin_freeargs(asr_t *ap) {
	alloc_downref (ap);
}

/* ----------------------------------------------------------------------------
** Start a timer.  This function is no longer needed.  Use gettimeofday in
** CASL scripts instead.
*/

list_t *Timer_List = NULL;

asr_t *casl_timer_start(asr_t *args) {
	static int i = 0;

	struct ttimer {
		int tag;
		struct timeval tv;
	} *ttv;

	if(args) 
		error(E_USER, "too many arguments for timer_start()");

	i += 1;

	ttv = xalloc(sizeof(*ttv));
	ttv->tag = i;
	gettimeofday(&ttv->tv, NULL);

	Timer_List = l_push(Timer_List, ttv);

	return(asr_int(i, 0));
}

/* ----------------------------------------------------------------------------
** Stop a timer started with timer_start.
*/

asr_t *casl_timer_stop(asr_t *args) {
	list_t *node;
	struct timeval tv;
	int i;

	struct ttimer {
		int tag;
		struct timeval tv;
	} *ttv = 0;

	gettimeofday(&tv, NULL);	      

	if(!args || !args->asr_kids[0]) 
		error(E_USER, "too few arguments for timer_stop()");
	if(args->asr_kids[1]) 
		error(E_USER, "too many arguments for timer_stop()");

	i = asr_getint(args->asr_kids[0]);
	
	for(node = Timer_List; node; node = node->next) {
		ttv = (struct ttimer *) node->data;
		if(ttv->tag == i)
			break;
	}

	if(!node)
		error(E_USER, "no timer found for tag \"%d\"", i);

	Timer_List = l_delete(Timer_List, node);
	i = ms_delta(&ttv->tv, &tv);
	free(ttv);

	return(asr_int(i, 0));
}

/* ----------------------------------------------------------------------------
** Return the size of an asr node.  If it's a list, this returns the number of
** nodes.  That is not terribly consistent, but it is sort of useful.
*/

asr_t *casl_size(asr_t *args) {
	asr_t *arg;

       	if(!args || !args->asr_kids[0])
		error(E_USER, "too few arguments for size");

	if(args->asr_kids[1])
		error(E_USER, "too many arguments for size()");

	arg = args->asr_kids[0];
	alloc_upref (arg);

	switch(arg->asr_type) {
	case N_CREATELIST:
		arg = eval_expr(arg);
	case LIST: {
		asr_t *l;
		int i = 0;		

		for(l = arg; l; l = l->asr_kids[1]) 
			i++;

		alloc_downref (arg);
		return(asr_int(i, 0));	
	}
	
	default:
		alloc_downref (arg);
		return(asr_int(args->asr_kids[0]->asr_size, 0));
	}
}

/* ----------------------------------------------------------------------------
** Turn any asr_node into a buffer representation.
*/

asr_t *casl_tobuf(asr_t *args) {
	asr_t *list;
	asr_t *buf;

	int len;
	u_char *up;

	if(!args || !args->asr_kids[0])
		error(E_USER, "too few arguments for tobuf");

	if(args->asr_kids[1])
		error(E_USER, "too many arguments for tobuf()");

	list = args;

	len = list_coagulate(list, &up);

	buf = asr_buffer(len, "INTERNAL");
	memcpy(buf->asr_buf, up, len);
	xfree ((void **)&up);

	return(buf);
}
       
/* ----------------------------------------------------------------------------
** Get an environment variable.
*/

asr_t *casl_getenv(asr_t *arg) {
	char *string = NULL;
	char *val = NULL;

	if(!arg || !arg->asr_kids[0])
		error(E_USER, "too few arguments for getenv()");
	
	if(arg->asr_kids[1])
		error(E_USER, "too many arguments for getenv()");

	asr_strep(arg->asr_kids[0], &string);

	if(!string || !(val = getenv(string))) {
		if (string) xfree ((void **)&string);
		return(asr_string (NULL, 0));
	}

	xfree((void **)&string);

	return(asr_string(val, 0));
}

/* ----------------------------------------------------------------------------
** Set an environment variable....  ... Er...  This is a very useful function
** since CASL can't call anything else that might try to read such environment.
*/

asr_t *casl_setenv(asr_t *arg) {
	char *string = NULL;
	char *val = NULL;

	if(!arg || !arg->asr_kids[0] || !arg->asr_kids[1]
		|| !arg->asr_kids[1]->asr_kids[0] )
		error(E_USER, "too few arguments for setenv()");
	
	if(arg->asr_kids[1]->asr_kids[1])
		error(E_USER, "too many arguments for setenv()");

	asr_strep(arg->asr_kids[0], &string);
	asr_strep(arg->asr_kids[1]->asr_kids[0], &val);

	if (!string || !val) {
		error (E_USER, "bad arguments for setenv()");
	}
	
	setenv(string, val, 1);

	xfree((void **)&string);
	xfree((void **)&val);

	return(NULL);
}

/* ----------------------------------------------------------------------------
** Return a string representation of an asr node.
*/

asr_t *casl_strep(asr_t *arg) {
	char *string = NULL;
	asr_t *x = NULL;

	if(!arg || !arg->asr_kids[0])
		error(E_USER, "too few arguments for strep()");
	
	if(arg->asr_kids[1])
		error(E_USER, "too many arguments for strep()");
	
	asr_strep(arg->asr_kids[0], &string);
	x = asr_string(string, 0);
	if (string) xfree((void **)&string);

	return(x); 
}

/* ----------------------------------------------------------------------------
** Get the time in ms since midnight.
*/

asr_t *casl_gtod(asr_t *args) {
	struct timeval t;
	u_int32_t ms;

	if(args && args->asr_kids[0])
		error(E_USER, "too many arguments for gettimeofday()");

	memset(&t, 0, sizeof(t));

	gettimeofday(&t, NULL);

	ms = t.tv_sec * 1000;
	if(t.tv_usec > 1000)
		ms += t.tv_usec / 1000;

	return(asr_int(ms, 0));
}

/* ----------------------------------------------------------------------------
** Used by the timer_* functions.
*/

static int ms_delta(struct timeval *start, struct timeval *stop) {
	struct timeval *t = tvsub(start, stop);
        	int ms;
        
	ms = (t->tv_sec * 1000) + (t->tv_usec > 1000) ?
        		t->tv_usec / 1000 : 0;
       
	/*XXX*/
        
	if(!ms) 
		ms++;
        
	return(ms);
}

/* ----------------------------------------------------------------------------
** Used by the timer_* functions.
*/

static struct timeval *tvsub(struct timeval *out, struct timeval *in) {
        	static struct timeval tm;        
	long diff;
        
	diff = in->tv_usec - out->tv_usec;
        
	if(diff < 0) {        
		diff += 1000000;                
		out->tv_sec -= 1;
        	}
       
	tm.tv_usec = diff;
                
	diff = in->tv_sec - out->tv_sec;
              	tm.tv_sec = diff;
       
	return(&tm);
}

/* ----------------------------------------------------------------------------
** Do a callstack trace.
*/

asr_t *casl_callstack(asr_t *args) {
	list_t *n;
	extern list_t *FunctionStack;

	if (args) {
		error (E_USER, "too many arguments for callstack()");
	}

	if(!FunctionStack) {
		printf("<global>\n");
		alloc_upref (null);
		return(null);
	}

	for(n = FunctionStack; n; n = n->next) 
		printf("%s(...)\n", (char *)n->data);

	printf("<global>\n");
	
	alloc_upref (null);
	return(null);
}
	
