/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/*
 * track allocation and handle deallocation of intermediate ASR nodes 
 *
 * Currently the memory allocation is based on reference counts.
 * It does not take very long to increment or decrement an integer.  Each
 * object tracks how much it is referenced.  Anything that allocates something
 * it will pass on is responsible for marking a reference.  There are
 * exceptions, however.  Internal functions usually do not require it if they
 * take an asr and don't return one.  For example. asr_getint does not
 * require any additional reference counts, and should leave the passed asr's
 * count unchanged as of the return.  Also builtins are not required to do
 * anything with the arguments.  The function that calls the builtins takes
 * care of the arguments.  Other functions will be labelled.
 *
 * If unsure, an upref will worst case just leak memory.  An extra downref
 * could cause unpredictible results, asserts, and crashes.  Any extra uprefs
 * can be caught by a memory leak analyzer.  All asrs allocated during run
 * time should be freed.  There should be no memory leaked from any call to
 * eval_statement.
 *
 * A note, all ASR nodes are created with a default reference count of 1.  It
 * is assumed that if they are allocated, they are referenced.
 *
 * Since asr nodes are passed around like candy, there was a lot of allocation
 * and freeing going on.  So asr structures are now kept around after
 * being dereferenced and kept in a pool of up to 2048 entries.  This will
 * make memory allocation profiling harder, so this number should be set
 * to 0 for such.
 */

#include "casl.h"

#define MAX_REFS	       	8192
#define ASR_CACHE		2048

extern int casl_line;
extern char *casl_file;
extern struct arity arity_table[];

struct cache_type {
	struct cache_type *next;
};

/* Max number of sub-notes a node will have */
#define MAX_ARITY 4

/* The pool of unused asr nodes, and the count of entries */
struct cache_type *asr_cache = 0;
int asr_cachec = 0;

/* Also keep the arity structures */
struct cache_type *arity_cache[MAX_ARITY] = {0, 0, 0, 0};
int arity_cachec[MAX_ARITY] = {0, 0, 0, 0};

/* ----------------------------------------------------------------------------
** Allocate a new asr node.  This will get it from the pool if possible.
*/ 

asr_t *asr_new(int arity) {
	asr_t *ap = (asr_t *) asr_cache;
	if (ap) {
		asr_cache = asr_cache->next;
		asr_cachec--;
		memset (ap, 0, sizeof (*ap));
	} else
		ap = (asr_t *) xcalloc(1, sizeof(*ap));
	
	if(arity) {
		if (arity <= MAX_ARITY && arity_cache[arity - 1]) {
			ap->asr_kids = (asr_t **) arity_cache[arity - 1];
			arity_cache[arity - 1] = arity_cache[arity - 1]->next;
			arity_cachec[arity - 1]--;
			memset (ap->asr_kids, 0, sizeof (asr_t *) * arity);
		} else
			ap->asr_kids = (asr_t **) xcalloc(1, sizeof(asr_t *) * arity);
	}

	ap->asr_file = casl_file;
	ap->asr_count = 1;

	return(ap);
}

/* --------------------------------------------------------------------------- 
** Check if the reference count is 0
*/

int alloc_isref(asr_t *ap) {
	if(ap->asr_count)
		return(1);

	return(0);
}

/* --------------------------------------------------------------------------- 
** Add to the reference count of an asr node.  If it is a parser node
** or has a line number, ignore it since it came from the parser.
** Ideally, if nodes are referenced and dereferenced properly, this shouldn't
** be an issue.
*/

void alloc_upref(asr_t *ap) {
	if(ap->asr_line)
		return;

	switch (ap->asr_type) {
	case N_INT:
	case N_NEGINT:
	case N_CHAR:
	case LIST:
	case N_BUFFER:
		break;
	default:
		return;
	}

	if(ap->asr_count > MAX_REFS)
		error(E_USER, "object has too many global references");

	ap->asr_count += 1;

	return;
}

/* --------------------------------------------------------------------------- ** Remove a reference count.  Free the node if there are no references left.
*/

void alloc_downref(asr_t *ap) {
	if(ap->asr_line)
		return;

	switch (ap->asr_type) {
	case N_INT:
	case N_NEGINT:
	case N_CHAR:
	case LIST:
	case N_BUFFER:
		break;
	default:
		return;
	}

	if(!ap->asr_count)
		error(E_INTERNAL, "object at %p has subzero reference count", ap);

	ap->asr_count -= 1;

	asr_safefree (&ap);

	return;
}

/* --------------------------------------------------------------------------- 
** Free a node if it's reference count is at zero.
*/

void asr_safefree(asr_t **ap) {
	asr_t *a;
	asr_t *n;

	assert(ap && *ap);

	a = *ap;

	/* don't free anything created directly by parser */

	if(a->asr_line)
		return;

	if(a->asr_count)
		return;
		
	/* Lists probably don't need a special case..  A downref on all
	   kids *should* take care of it..  But just in case...  Besides,
	   this isn't recursive. */
	if (a->asr_type == LIST) {
		while (a) {
			n = a->asr_kids[1];
			if (a->asr_kids[0]) alloc_downref (a->asr_kids[0]);
			asr_sfree (&a);
			a = n;
		}
		*ap = 0;
		return;
	}

	asr_free(ap);
}

/* --------------------------------------------------------------------------- 
** Free an asr node,  without freeing any children.
*/

void asr_sfree(asr_t **app) {
	asr_t *ap = *app;

	if(arity_table[ap->asr_type].arity) {
		int i;

		for(i = 0; i < arity_table[ap->asr_type].arity; i++)
			ap->asr_kids[i] = NULL;
	}

	asr_free(app);
	return;
}

/* --------------------------------------------------------------------------- 
** Free an asr node and all it's children.  If there is space in the asr pool,
** it will put them there, otherwise it will just free it.  This will
** recursively traverse the list of children.
*/

int StopFree;

void asr_free(asr_t **app) {
	asr_t *ap = *app;
	int type = ap->asr_type;    

	if(arity_table[type].arity) {
		int arity = arity_table[type].arity;
		int i;

		for(i = 0; i < arity; i++)
			if(ap->asr_kids[i])
				asr_free(&ap->asr_kids[i]);

		if (arity < MAX_ARITY && arity_cachec[arity - 1] < ASR_CACHE) {
			arity_cachec[arity - 1]++;
			((struct cache_type *)ap->asr_kids)->next = arity_cache[arity - 1];
			arity_cache[arity - 1] = (struct cache_type *)ap->asr_kids;
			ap->asr_kids = 0;
		} else
			xfree((void **)&ap->asr_kids);
	} 

	switch(ap->asr_type) {
	case N_ID: 
		xfree((void **) &ap->asr_ident);
		break;

	case N_STRING:
		xfree((void **) &ap->asr_str);
		break;

	case N_BUFFER:
		if(ap->asr_base)
			xfree((void **) &ap->asr_base);
		else
			if (ap->asr_size) xfree((void **) &ap->asr_buf);

		break;

	default:
		break;
	}

	ap->asr_type = 0; 		// will most likely cause an 
					// assert if used.

	if(!StopFree) {
		if (asr_cachec < ASR_CACHE) {
			((struct cache_type *)ap)->next = asr_cache;
			asr_cache = (struct cache_type *)ap;
			asr_cachec++;
			ap = 0;
		} else
			xfree((void **) &ap);
	}
	
	*app = NULL;
	
	return;
}

/* --------------------------------------------------------------------------- 
*/
