/* $Id: interface.c,v 1.9 2000/10/31 06:08:27 keiji Exp $ */
/*
** interface.c - The interface part of pakemon.
** Author : Keiji Takeda
**
** Pakemon is an abbreviation of "Packet Monster", a simple packet
** monitoring misuse detector.
** Copyright (C) 1999, 2000 Keiji Takeda <keiji@sfc.keio.ac.jp>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "interface.h"
#include "util.h"

/*****************************************************
  Function: open_pcap
  Purpose:  open and initialize pcap interface.
  Arguments: char *interface : name of the interface
******************************************************/

pcap_t * open_pcap( char **interface )
{

  char          errbuf[PCAP_ERRBUF_SIZE];

  char *        device_name;

  pcap_t *      packet_descriptor;

  if ( *interface == (char *)NULL )
    {

      int device_name_length;

      device_name = pcap_lookupdev( errbuf );

      if( device_name == (char *)NULL)
	  {
	    fprintf(stderr, "Interface Lookup Error : %s\n %s\n", *interface, errbuf);
	    return(pcap_t *)NULL;
	  }

      device_name_length = strlen( device_name );

      *interface = (char *)malloc_check( device_name_length+1, "open_pcap:interface");

      if( *interface == (char *)NULL )
	return(pcap_t *)NULL;

      strncpy( *interface, device_name, device_name_length );

      memset(*interface + device_name_length, 0, 1 );

    }

  /* start listening the interface */
  packet_descriptor = pcap_open_live( *interface, SNAPLEN, PROMISC, TIMEOUT, errbuf );

  if ( packet_descriptor == (pcap_t *)NULL )
    {
      fprintf(stderr, "Open_live pcap Error on : %s\n %s\n", *interface, errbuf);
      return(pcap_t *)NULL;
    }

  return packet_descriptor;

} /* open_pcap */

/*****************************************************
  Function: get_local_info
  Purpose:  lookup local network information
            Defines local network address, local netmask
  Arguments:
            char *: name of interface to look at
	    pcap_t: packet_descriptor
	    struct in_addr *local_netaddr:
	      pointer to a variable to store network address
	    struct in_addr *local_netmask:
	      pointer to a variable to store network mask
  Returns:
******************************************************/
void get_local_info( char * interface,
		     struct in_addr *local_netaddr,
		     struct in_addr *local_netmask)
{
  char errbuf[PCAP_ERRBUF_SIZE]; /* buffer to store error message */

  printf("interface = %s\n",interface);
  if ( pcap_lookupnet( interface, &(local_netaddr->s_addr), 
		       &(local_netmask->s_addr), errbuf )    < 0)
    {
      fprintf( stderr, "Network Lookup Error on : %s\n %s\n", interface, errbuf);
    }
} /* get local info */

/************************************************
  Function: reset_promisc
  Purpose: to reset NIC device to promiscus mode
  Argument:
     char *nic_name : NIC device name(eg. eth0)
  Returns: int
     -1 : device satus flag failure
     -2 : NIC was alredy set promiscus
     Socket Number : success
 ***********************************************/
int reset_promisc( char *nic_name )
{
    int sock;

    struct ifreq interface;

    /***** open socket *****/
    if ( ( sock = socket( AF_INET, SOCK_RAW, IPPROTO_RAW )) < 0)
      {
	puts("Unable open SOCK_RAW.\n"); 
	return( -1 );
      }

    strcpy( interface.ifr_name, nic_name );

    if ( ioctl( sock, SIOCGIFFLAGS, &interface ) < 0)
      {
	puts("Unable to get device status flag.\n");
	return( -1 );
      }

    /***** Reset promiscus flag status *****/
    if ( ( interface.ifr_flags & IFF_PROMISC ) != 0)
      {
	/***** set Promiscus flag *****/
	interface.ifr_flags &= ~(IFF_PROMISC);

	/***** set NIC status flag *****/ 
	if ( ioctl( sock, SIOCSIFFLAGS, &interface) < 0)
	  {
	    puts("Unable to set device status flag.");
	    return( -2 );
	  }
      }

    return( sock );

} /*** reset_promisc ***/













