/* File   : makeudp.c
 * Author : Karyl F. Stein <xenon@xenos.net>
 * Purpose: Create a UDP packet
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "makeudp_sysconf.h"
#include "makeudp_conf.h"
#include <stdio.h>

#ifdef SunOS
#include <sys/types.h>        /* u_char      */
#include <netinet/in.h>       /* sockaddr_in */
#include <arpa/nameser.h>     /* MAXDNAME    */
#include <netinet/in_systm.h> /* n_long      */
#include <resolv.h>           /* in_addr     */
#endif /* SunOS */

#include <unistd.h>           /* STDIN_FILENO         */
#include <netinet/udp.h>      /* UDP header struct    */
#include <netinet/in.h>       /* in_addr struct       */
#include "spak_func.h"        /* Function prototypes  */
#include "makeudp_func.h"     /* Function prototypes  */
#include "datadisp.h"         /* Function prototypes  */

/* Values used by the flag variable */
#define GOT_CKSUM 0x01
#define GOT_LEN   0x02
#define GOT_PDST  0x04
#define GOT_PLEN  0x08
#define GOT_PSRC  0x10
#define VERBOSE   0x20

/* Set defaults if not defined previously or not valid.  Change in the */
/* makeudp_conf.h file and not here.                                   */
#ifndef UDP_DEFAULT_SOURCE_PORT
# define UDP_DEFAULT_SOURCE_PORT 0
#endif  /* UDP_DEFAULT_SOURCE_PORT */
#ifndef UDP_DEFAULT_PROTOCOL
# define UDP_DEFAULT_PROTOCOL 17
#endif  /* UDP_DEFAULT_PROTOCOL */


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s <d_port>\n", name);
  fprintf(stderr, "       [-h] [-s <src_port>] [-c <hdr_cksum>] [-l <length>] [-o <output_file>]\n");
  fprintf(stderr, "       [-i <input_file>] [-ps <source>] [-pd <destination>] [-pp <protocol>]\n");
  fprintf(stderr, "       [-pl <length>] [-pz <value>] [-v]\n");
  fprintf(stderr, " <d_port>  Destination port for the packet to have, (in numerical or word\n");
  fprintf(stderr, "           form).\n");
  fprintf(stderr, " -c        Set the header checksum to <hdr_cksum>.  Default: Correct value if\n");
  fprintf(stderr, "           -ps and -pd are specified, otherwise 0.\n");
  fprintf(stderr, " -h        Print this help message.\n");
  fprintf(stderr, " -i        Read the packet data from the file <input_file>.  If <input_file>\n");
  fprintf(stderr, "           is -, stdin will be used.\n");
  fprintf(stderr, " -l        Set the header length to <length>.  Default: Correct value\n");
  fprintf(stderr, " -o        Print the packet data to <output_file>.  Default: stdout\n");
  fprintf(stderr, " -pd       Set the pseudo-header destination address to <destination>, (in\n");
  fprintf(stderr, "           numerical or word form).\n");
  fprintf(stderr, " -pl       Set the length of the header as stored in the pseudo-header to\n");
  fprintf(stderr, "           <length>.  Default: Correct value\n");
  fprintf(stderr, " -pp       Set the protocol as stored in the pseudo-header to <protocol>.\n");
  fprintf(stderr, "           Default: %d\n", UDP_DEFAULT_PROTOCOL);
  fprintf(stderr, " -ps       Set the source address of the packet as stored in the pseudo-\n");
  fprintf(stderr, "           header to <source>.\n");
  fprintf(stderr, " -pz       Set the value of the 8-bit 0 padding in the pseudo-header to\n");
  fprintf(stderr, "           <value>.  Defalt: 0\n");
  fprintf(stderr, " -s        Set the source port to <src_port>.  Default: %d\n", UDP_DEFAULT_SOURCE_PORT);
  fprintf(stderr, " -v        Turn on verbose mode, (print packet data).\n");
  exit(0);
}


int main (int argc, char *argv[]) {
  char tmpchar;
  char *data = NULL, *input_file = NULL, *options = NULL, *output_file = NULL;
  FILE *ifp, *ofp;
  int data_len = 0, flag = 0, place = 0, tmpint;
  struct in_addr source, destination;
  struct udphdr *packet;
  void *datagram, *ptr;

  /* Make sure the correct number of arguments are given */
  if ((argc < 2) || (strcmp(argv[1], "-h") == 0))
    usage(argv[0]);

  /* Create a new packet */
  datagram = (void *) new_udp_packet();
  packet = &((struct full_udphdr *) datagram)->packet;

  /* Set defaults */
  destination.s_addr = source.s_addr = 0;

  /* Set destination port */
  packet->UDP_FIELD_DEST = my_atoi(argv[1], 16, "destination port");

  /* Parse the arguments */
  tmpint = 1;
  while (++tmpint < argc) {
    if (*((char *) ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Header Checksum */
    if (strcmp(ptr, "c") == 0) {
      packet->UDP_FIELD_CSUM = my_atoi(argv[++tmpint], 16, "-c");
      flag |= GOT_CKSUM;
    }

    /* Print help screen */
    else if (strcmp(ptr, "h") == 0)
      usage(argv[0]);

    /* Input File */
    else if (strcmp(ptr, "i") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -i argument expects a file name\n");
      else data = append_data(data, &data_len, argv[tmpint]);
    }

    /* Header Length */
    else if (strcmp(ptr, "l") == 0) {
      packet->UDP_FIELD_LEN = htons(my_atoi(argv[++tmpint], 16, "-l"));
      flag |= GOT_LEN;
    }

    /* Output File */
    else if (strcmp(ptr, "o") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -o argument expects a file name\n");
      else {
        if (output_file != NULL)
          free(output_file);
	output_file = argv[tmpint];
      }
    }

    /* Destination Address (for pseudo-header) */
    else if (strcmp(ptr, "pd") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -pd argument expects an IP number\n");
      else {
#ifdef INET_ATON
	if (inet_aton(argv[tmpint], &destination) == 0)
#else
	if ((destination.s_addr = inet_addr(argv[tmpint])) == -1)
#endif  /* INET_ATON */
	  destination.s_addr = getaddrbyname(argv[tmpint]);
	((struct full_udphdr *) datagram)->destination = destination.s_addr;
	flag |= GOT_PDST;
      }
    }

    /* Header Length (for pseudo-header) */
    else if (strcmp(ptr, "pl") == 0) {
      ((struct full_udphdr *) datagram)->length =
	my_atoi(argv[++tmpint], 16, "-pl");
      flag |= GOT_PLEN;
    }

    /* Protocol (for pseudo-header) */
    else if (strcmp(ptr, "pp") == 0)
      ((struct full_udphdr *) datagram)->protocol =
	my_atoi(argv[++tmpint], 8, "-pp");

    /* Source Address (for pseudo-header) */
    else if (strcmp(ptr, "ps") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "Warning: The -ps argument expects an IP number\n");
      else {
#ifdef INET_ATON
	if (inet_aton(argv[tmpint], &source) == 0)
#else
	if ((source.s_addr = inet_addr(argv[tmpint])) == -1)
#endif  /* INET_ATON */
	  source.s_addr = getaddrbyname(argv[tmpint]);
	((struct full_udphdr *) datagram)->source = source.s_addr;
	flag |= GOT_PSRC;
      }
    }

    /* Zero pad (illegal if not 0) */
    else if (strcmp(ptr, "pz") == 0)
      ((struct full_udphdr *) datagram)->zero =
        my_atoi(argv[++tmpint], 8, "-pz");

    /* Source Port */
    else if (strcmp(ptr, "s") == 0)
      packet->UDP_FIELD_SRC = htons(my_atoi(argv[++tmpint], 16, "-s"));

    /* Set verbose mode */
    else if (strcmp(ptr, "v") == 0)
      flag |= VERBOSE;

    else usage(argv[0]);
  }

  /* Append data */
  if (data != NULL) {
    if ((datagram = (void *) realloc(datagram, UDP_DEFAULT_HEADER_SIZE +
                                     UDP_PSEUDO_HEADER_SIZE +
				     data_len)) == NULL) {
      fprintf(stderr, "Out of Memory\n");
      exit(1);
    }
    memcpy(datagram + UDP_DEFAULT_HEADER_SIZE + UDP_PSEUDO_HEADER_SIZE, data,
	   data_len);
    packet = & ((struct full_udphdr *) datagram)->packet;
    ((struct full_udphdr *) datagram)->length += data_len;
  }

  /* Set the packet length fields correctly */
  if (!(flag & GOT_PLEN))
    ((struct full_udphdr *) datagram)->length =
      htons(UDP_DEFAULT_HEADER_SIZE + data_len);
  if (!(flag & GOT_LEN))
    packet->UDP_FIELD_LEN = htons(UDP_DEFAULT_HEADER_SIZE + data_len);

  /* Generate Checksum */
  if ((!(flag & GOT_CKSUM)) && ((flag & GOT_PSRC) && (flag & GOT_PDST)))
    packet->UDP_FIELD_CSUM = checksum(datagram, UDP_DEFAULT_HEADER_SIZE +
				      data_len + UDP_PSEUDO_HEADER_SIZE);

  /* Print packet information */
  if (flag & VERBOSE)
    print_full_udp_data((struct full_udphdr *) datagram);

  /* Write the packet data to a file or stdout */
  write_packet(output_file, packet, UDP_DEFAULT_HEADER_SIZE + data_len);
}
