/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include <stdio.h>
#include <sys/types.h>

#ifdef SET_TEST
extern int LLL;
static void dumpbuf(u_char *buf, int len);
static void dumpbits(u_long val, int len);
#endif

/* set an arbitrary range of bits in a buffer to an integer value */

static u_char highmask[] = { 0, 128, 192, 224, 240, 248, 252, 254, 255 };
static u_char lowmask[] = { 0, 1, 3, 7, 15, 31, 63, 127, 255 };

/* ----------------------------------------------------------------------------
** Get a range of bits and return them as an integer.
*/

u_long getbits(u_char *data, int off, int len) {
	int pos, olen = len;
	int usedFirst, usedLast;
	
	u_long val;

	pos = (off >> 3);
	pos = pos + (len >> 3);

	if(len < 8 && (off & 7) + len < 8) {
		usedFirst = off & 7;
		usedLast  = 8 - (len + usedFirst);
		val = data[pos] & ~(highmask[usedFirst] | lowmask[usedLast]);
		val >>= usedLast;
		return(val);
	} else if(len < 8) 
		pos++;

	usedFirst = 8 - (off & 7);
	usedLast = (len - usedFirst) % 8;

	val = (data[pos--] & highmask[usedLast]) >> (8 - usedLast);
	len -= usedLast;

	while(len >= 8) {
		val |= data[pos--] << usedLast;
		usedLast += 8;
		len -= 8;
	}
	
	if(len)
		val |= (data[pos] & lowmask[usedFirst]) << (olen - usedFirst);
	return(val);
}

static int setbitslt8(u_char *data, int off, int len, u_long val);

/* ----------------------------------------------------------------------------
** Set a range of bits to an integer value.
*/

int 
setbits(u_char *data, int off, int len, u_long val)
{
	int pos;
	int usedFirst, usedLast;
	u_char c;

	if (len < sizeof (u_long) * 8 - len) {
		val = val & (((u_long)-1) >> (sizeof (u_long) * 8 - len));
	}

#ifdef SET_TEST	
	printf("--------------------------------------------------\n");
	printf("%p off %d, %d bits in %d buffer\n", data, off, len, LLL);
	printf("val %lu in %d bits = ", val, len);
	dumpbits(val, len);
	printf("current buffer:\n");
	dumpbuf(data, LLL);
#endif

       	/* cope with calls for less than 8 bits at a time */

	if(len < 8) 
#ifndef SET_TEST
		return(setbitslt8(data, off, len, val));
#else
	{


		setbitslt8(data, off, len, val);
		printf("new buffer:\n");
		dumpbuf(data, LLL);
		return(0);

	}
#endif
       
	/* find out what byte we start in, and from that,  
	 * what byte we finish in...
  	 */

	pos = ((off + len) >> 3);

	/* find out how many bits we use in the first and
 	 * last bytes 
	 */

	usedFirst = 8 - (off & 7);
	usedLast = (len - usedFirst) % 8;

	/* set last byte */

	c = (val & lowmask[usedLast]) << (8 - usedLast);

	data[pos] = data[pos] & lowmask[(8 - usedLast)];
	data[pos] = data[pos] | c;
        
	val >>= usedLast;
	pos--;
	
	len -= usedLast;

	/* middle bytes */

	while(len >= 8) {
		data[pos--] = val & 0xFF;
		val >>= 8;
		len -= 8;
	}

	/* first byte */

	if(len) {
		data[pos] = data[pos] & highmask[(8 - usedFirst)];
		data[pos] = data[pos] | val;
	}

#ifdef SET_TEST
       	printf("new buffer:\n");
       	dumpbuf(data, LLL);
#endif

	return(0);
}

/* ----------------------------------------------------------------------------
** set 1..7 bits to an integer value
*/

static int 
setbitslt8(u_char *data, int off, int len, u_long val)
{
	u_char c;

	int boff, tail;
	int pos;

	pos = off >> 3;

	boff = off & 7;
	tail = (8 - boff) - len;
	tail = tail < 0 ? -tail : tail;

	if((boff + len) > 8) {
		c = (val & lowmask[tail]) << (8 - tail);
		val >>= tail;

		data[pos + 1] = data[pos + 1] & lowmask[(8 - tail)];
		data[pos + 1] = data[pos + 1] | c;

		data[pos] = data[pos] & highmask[boff];
		data[pos] = data[pos] | val;
	} else {
		c = val << tail;

		data[pos] = data[pos] & (highmask[boff] | lowmask[tail]);
		data[pos] = data[pos] | c;
	}

	return(0);
}

#ifdef TEST

/* ----------------------------------------------------------------------------
** For testing purposes.
*/

int main(int argc, char **argv) {
	u_char b[] = { 170, 170, 170, 170, 170, 
		       170, 170, 170, 170, 170 };
	u_long x;

	printf("%lu\n", strtoul(argv[3], NULL, 10));

	setbits(b, atoi(argv[1]), atoi(argv[2]), strtoul(argv[3], NULL, 10));
	x = getbits(b, atoi(argv[1]), atoi(argv[2]));

	printf("%lu\n", x);
	
	exit(0);
}

#endif
