/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

/* 
 * deal with the intricacies of function calls, argument passing,
 * and scoping.
 */

#include "casl.h"

static asr_t *eval_func(asr_t *, asr_t *, char *);
void chew_varargs(asr_t *formal, asr_t *clist);
void assign_argument(asr_t *formal, asr_t *calling);
void scope_enter(void);
void scope_exit(void);

#define NO_LOOP		0
#define IN_LOOP 	1
#define BREAK_LOOP 	2
#define CONTINUE_LOOP	3

extern int Debug;

list_t *FunctionStack = NULL;

char *CurrentFunction = NULL;

typedef asr_t *(*builtin_t)(asr_t *);

/* ----------------------------------------------------------------------------
** call a subroutine. Look up the function identifier in the symbol 
** table and verify that it's a function; doing this gives us the 
** entry point for the function. Pass this to eval_func(), which handles
** the gory details.
*/

asr_t *eval_call(asr_t *node) {
	asr_t *func = NULL;
	asr_t *ret = NULL;
	char *cp;

	char buf[8192];

#ifdef TRACE_CALLS
	Dprintf("eval_call(%p)\n", node);
#endif

	snprintf(buf, 8192, "%s%s", CASL_FUNC_PREFIX, 
			node->asr_kids[0]->asr_ident);
	buf[8191] = 0;

	/* catch attempts to call builtin (C-language) functions, call them directly */

	func = st_get(buf, 0);

	if(!func || func->asr_type != N_FUNC) {
		asr_t *n;
		builtin_t bfp;
		
		/* XXX - come up with a way to make sure st_get returns a 
		 * builtin_t here, otherwise we're going to wind up calling a 
		 * char * or something at some point.
		 */

		bfp = (builtin_t)
			st_get(buf, BUILTIN);
		if(!bfp)
			error(E_USER, "attempt to call non-existant function \"%s\".",
				node->asr_kids[0]->asr_ident);

		n = builtin_callargs(node->asr_kids[1]);

#ifdef TRACE_CALLS
		Dprintf("builtin at %p (%p)\n", bfp, n);
#endif

		cp = CurrentFunction;
		CurrentFunction = node->asr_kids[0]->asr_ident;

		ret = bfp(n);

		CurrentFunction = cp;

		if (n) builtin_freeargs(n);

	} else {	     
		char *cp = xstrdup(node->asr_kids[0]->asr_ident);
		FunctionStack = l_push(FunctionStack, cp);

#ifdef TRACE_CALLS
		Dprintf("--------------------------------------------\n");
		Dprintf("calling %s()\n", node->asr_kids[0]->asr_ident);
#endif			
		/* handle the actual function call */

		cp = CurrentFunction;
		CurrentFunction = node->asr_kids[0]->asr_ident;
	       	
		ret = eval_func(func, node->asr_kids[1], node->asr_kids[0]->asr_ident);
		
		CurrentFunction = cp;

		FunctionStack = l_pop(FunctionStack, (void *) &cp);
		xfree((void **) &cp);
		
#ifdef TRACE_CALLS
		Dprintf("--------------------------------------------\n");
#endif

	}

	return(ret ? ret : asr_int(0, 0));
}

/* ----------------------------------------------------------------------------
** deal with the actual execution of a called subroutine. This is hairy; it
** implements varargs. This function deals only with CASL-defined subroutines;
** eval_call explicitly deals with builtin functions.
*/

asr_t *eval_func(asr_t *func, asr_t *cargs, char *fname) {
	extern int Looping;
	int s = 0;

	int i = 0;
	asr_t *fargs = func->asr_kids[1];
	asr_t *rv;

#ifdef TRACE_CALLS
	Dprintf("eval_func(%p, %p)\n", func, cargs);
#endif

	if (cargs) {
		cargs = builtin_callargs (cargs);
	}

	scope_enter();

	if(fargs) {
		asr_t *cn, *fn;

		cn = cargs;

		/* walk the list of formal arguments */
		
		for(fn = fargs; fn; fn = fn->asr_kids[1]) {
			i += 1;

			/* this shouldn't ever happen */

			if(!fn->asr_kids[0]) 
				error(E_USER, "Null formal argument at position %d"
				      " in function \"%s\"", i, fname);
							
			/* see if we have a calling argument matching
			 * the position of this formal argument.
			 */

			if(!cn || !cn->asr_kids[0]) 
				error(E_USER, "too few arguments for function \"%s\"",
				      fname);
			
			/* deal with varargs (more calling args than formals, 
			 * last formal becomes list of surplus calling args)
			 */

			if(cn->asr_kids[1] && !fn->asr_kids[1]) {
				chew_varargs(fn->asr_kids[0], cn);
				break;
			} else {
				/* do normal argument assignment */
				assign_argument(fn->asr_kids[0], cn->asr_kids[0]);

				cn = cn->asr_kids[1];
			}
		}
	} else if (cargs) {
		error (E_USER, "too many arguments to %s()", fname);
	}

	/* execute the actual function body. We get "rv" back when we
	 * hit a "return" statement; we get NULL if we hit the end of
	 * the statement list without seeing a return.
	 */

	s = Looping;
	Looping = 0;

	rv = eval_statement(func->asr_kids[2]);

	Looping = s;

	scope_exit();

	if (cargs) {
		builtin_freeargs (cargs);
	}

	/* eval_call catches the "no return value" case. */
	return(rv);
}

/* ----------------------------------------------------------------------------
** assign to formal arguments; this handles argument passing
** for functions.
** 
** The way I do this is really simple; a function declaration
** contains the "formal arguments" for the function in parens,
** and a call to a function contains the "calling arguments". 
** We use the formal arguments as a list of names of local 
** variables in the called function, and we step through the 
** list of calling arguments, evaluating each and inserting
** the value in the symbol table with the id of the formal
** argument, as if they were local variables assigned.
*/ 

void assign_argument(asr_t *formal, asr_t *calling) {
	asr_t *x = NULL;

	int save = Level;
	Level = UPLEVEL;

	if(calling->asr_type == N_ID) {
		asr_t *x = calling;

		if(!(calling = st_get(calling->asr_ident, Level)))
			error(E_USER, "undefined identifier \"%s\" in arg pass "
			      "to function \"%s\"", x->asr_ident, 
			      FunctionStack ? (char *) FunctionStack->data : "<unknown>");
		alloc_upref (calling);
		alloc_downref (x);
	}

	if (calling->asr_type == N_CALL) {
		save = Level;
		Level = UPLEVEL;
		calling = eval_expr (calling);
		Level = save;
	}

	Level = save;

	switch(calling->asr_type) {
	case N_BUFFER:
	case N_INT:
	case N_NEGINT:
	case N_CHAR:
		break;

	default:
		/* retrieve this id from the caller's symbol table */

		save = Level;
		Level = UPLEVEL;
		calling = eval_expr(calling);
		Level = save;

		break;

	}

	eval_assign((x = asr_node(N_ASSIGN, formal, calling)));
	asr_sfree(&x);
}

/* ----------------------------------------------------------------------------
** handle the varargs arg passing case (fewer formals than calling) by making
** a list out of the "trailing" calling args and assigning that list to the
** last formal argument. 
*/

void chew_varargs(asr_t *formal, asr_t *clist) {
	asr_t *head;
	asr_t *assign;
	asr_t **lp = &head;
       
	for(; clist; clist = clist->asr_kids[1]) {
		int save;

		asr_t *val = clist->asr_kids[0];

		save = Level;
		Level = UPLEVEL;

		if(val->asr_type == N_ID) {
			asr_t *x = val;

			if(!(val = st_get(val->asr_ident, Level)))
	       			error(E_USER, "undefined identifier \"%s\" in arg pass"
				      " to function \"%s\"", x->asr_ident, 
				      FunctionStack ? (char *) FunctionStack->data :
				      "<unknown>");
			alloc_upref (val);
			alloc_downref (x);
		}		

		Level = save;

		if (val->asr_type == N_CALL) {
			save = Level;
			Level = UPLEVEL;
			val = eval_expr (val);
			Level = save;
		}

		switch(val->asr_type) {
		case N_BUFFER:
		case N_INT:
		case N_NEGINT:
		case N_CHAR:
			break;

		default:
			save = Level;
			Level = UPLEVEL;
			val = eval_expr(val);
			Level = save;
			break;
		}

		/* make sure called function doesn't deallocate this */

		alloc_upref(val);

		*lp = asr_node(LIST, NULL, NULL);
		(*lp)->asr_kids[0] = val;
		lp = &(*lp)->asr_kids[1];
	}

	assign = asr_node(N_ASSIGN, formal, head);
	eval_assign (assign);
	asr_sfree (&assign);
}

/* ----------------------------------------------------------------------------
** perform any operations we may need to do before entering a new scope
*/

void scope_enter() {
	
	/* new local symbol table */

	st_push();
	Level++;
}

/* ----------------------------------------------------------------------------
** perform any cleanup we may need to do after we leave a scope forever
*/

void scope_exit() {

	/* Clean up references to any local variables that are unreferenced */
	st_collect ();

	/* return to previous symbol table */
	st_pop();
	Level--;
}
