/* 
 * This is source code to CASL (Custom Audit Scripting Language)
 *
 * Copyright 1998 Secure Networks, Inc.
 * Copyright 1999 Network Associates, Inc.
 * All Rights Reserved
 *
 * BEFORE YOU INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT,
 * CAREFULLY READ THE TERMS AND CONDITIONS IN THE FILE
 * "LICENSE.TXT" ACCOMPANYING THIS DOCUMENT. IF THE FILE
 * "LICENSE.TXT" IS MISSING, IT MAY BE OBTAINED FROM
 * NETWORK ASSOCIATES. NETWORK ASSOCIATES IS PERMITTING
 * THE USE, DISTRIBUTION, AND LIMITED MODIFICATION OF THIS
 * SOFTWARE PRODUCT ON A NON-COMMERCIAL BASIS SUBJECT TO
 * ALL OF THE CONDITIONS IN THE FILE "LICENSE.TXT." BY INSTALLING,
 * USING, OR MODIFYING THE SOFTWARE PRODUCT, YOU AND ANY
 * SUBSEQUENT USER ARE AGREEING TO BE BOUND BY ALL OF THE
 * TERMS AND CONDITIONS IN THE FILE "LICENSE.TXT." IF YOU DO
 * NOT AGREE TO ALL OF THOSE TERMS AND CONDITIONS, DO NOT
 * INSTALL, USE, OR MODIFY THIS SOFTWARE PRODUCT.
 */

#include "casl.h"

/* Manage the built-in "list" type of the CASL language. 
 *
 * This code is a result of a decision to make the language "list" type be
 * the same as the internal representation of a list, thus forcing data item
 * lists to be integrated into the tree, thus preventing modifying pointers
 * to these lists.  Any volunteers to make this better?
 *
 * The quickest way around this is to keep the asr pointer the same, but
 * modify the asr_kids to insert a new node.
 */

static asr_t *list_get_tail(asr_t *list);
static asr_t *list_get_head(asr_t *list);
static void list_item_append(asr_t *list, asr_t *item);
static void list_item_prepend(asr_t *list, asr_t *item);

/* ----------------------------------------------------------------------------
** Evaluate insert item at tail - "append" in CASL.
*/

asr_t *eval_ptail(asr_t *ap) {
	asr_t *c1, *c2, *i1, *i2;
	asr_t *list;

#ifdef TRACE_CALLS
	Dprintf("eval_ptail(%p)\n", ap);
#endif

	i1 = c1 = ap->asr_kids[0];
	i2 = c2 = ap->asr_kids[1];

	/* retrieve from symbol table */

	if(c1->asr_type == N_ID) {
		asr_t *newa = st_get(c1->asr_ident, Level);
		if (newa)
			alloc_upref (newa);
		alloc_downref (c1);
		c1 = newa;
	}
	else
		c1 = eval_expr(c1);

	if(c2->asr_type == N_ID) {
		asr_t *newa = st_get(c2->asr_ident, Level);
		if (!newa)
			error(E_USER, "attempt to refer to undefined id %s\n",
				i2->asr_ident);
		alloc_upref (newa);
		alloc_downref (c2);
		c2 = newa;
	}

	c2 = eval_expr(c2);

	if(!c1) {
		c2 = asr_node(LIST, c2, NULL);	
		if (i1->asr_type == N_ID) {
			alloc_upref (c2);
			st_replace (i1->asr_ident, c2, Level);
		}
		return (c2);
	}

	/* deal with insertion cases (which of source/target is a list) */
		
	if(c1->asr_type != LIST) {
		list = asr_node(LIST, c1, asr_node(LIST, c2, NULL));
		return(list);
	} else if(c1->asr_type == LIST) {
		list_item_append(c1, c2);
		return(c1);
	}

	assert(0);
	return(NULL);
}

/* ----------------------------------------------------------------------------
** Evaluate insert item at head - "prepend" in CASL.
*/

asr_t *eval_plist(asr_t *ap) {
	asr_t *c1, *c2, *i1, *i2;
	asr_t *list;

#ifdef TRACE_CALLS
	Dprintf("eval_plist(%p)\n", ap);
#endif

	i1 = c1 = ap->asr_kids[0];
	i2 = c2 = ap->asr_kids[1];

	/* retrieve from symbol table */

	if(c1->asr_type == N_ID) {
		asr_t *newa = st_get (c1->asr_ident, Level);
		if (newa) {
			alloc_upref (newa);
		}
		alloc_downref (c1);
		c1 = newa;
	}
	else
		c1 = eval_expr(c1);

	if(c2->asr_type == N_ID) {
		asr_t *newa = st_get (c2->asr_ident, Level);
		if (!newa)
			error(E_USER, "attempt to refer to undefined id %s\n",
				i2->asr_ident);

		alloc_upref (newa);
		alloc_downref (c2);
		c2 = newa;
	}

	c2 = eval_expr(c2);

	if(!c1) {
		c2 = asr_node(LIST, c2, NULL);	
		if (i1->asr_type == N_ID) {
			alloc_upref (c2);
			st_replace (i1->asr_ident, c2, Level);
		}
		return (c2);
	}

	/* deal with insertion cases (which of source/target is a list) */

	if(c1->asr_type != LIST) {
		list = asr_node(LIST, c2, asr_node(LIST, c1, NULL));
		return(list);
	} else if(c1->asr_type == LIST) {
		list_item_prepend(c1, c2);
		return(c1);
	}

	assert(0);
	return(NULL);
}
	
/* ----------------------------------------------------------------------------
** Evaluate remove item from tail - "tail" in CASL.
*/

asr_t *eval_tail(asr_t *ap) {
	asr_t *result;
	ap = ap->asr_kids[0];

#ifdef TRACE_CALLS
	Dprintf("eval_tail(%p)\n", ap);
#endif

	ap = eval_expr(ap);

	if(ap->asr_type == N_ID) {
		asr_t *v = st_get(ap->asr_ident, Level);
		if(!v) 
			error(E_USER, "attempt to refer to undefined id %s\n",
				ap->asr_ident);
		
		alloc_upref (v);
		alloc_downref (ap);
		ap = v;
	}

	if(ap->asr_type != LIST) {
		return(ap);
	}

	result = list_get_tail(ap);
	alloc_downref (ap);

	/* brutal hack, get around parser creation of list literals with identifiers */

	if(result && result->asr_type == N_ID) {
		ap = st_get(result->asr_ident, Level);
		alloc_downref (result);
		alloc_upref (ap);
		result = ap;
	}

	return(result);
}

/* ----------------------------------------------------------------------------
** Evaluate remove item from head - "head" in CASL.
*/

asr_t *eval_head(asr_t *ap) {
	asr_t *result;
	ap = ap->asr_kids[0];

#ifdef TRACE_CALLS
	Dprintf("eval_head(%p)\n", ap);
#endif
	ap = eval_expr(ap);

	if(ap->asr_type == N_ID) {
		asr_t *v = st_get(ap->asr_ident, Level);
		if(!v) {
			error(E_USER, "attempt to refer to undefined id %s\n",
				ap->asr_ident);
		}
		
		alloc_downref (ap);
		alloc_upref (v);
		ap = v;
	}

	if(ap->asr_type != LIST) {
		return(ap);
	}

	result = list_get_head(ap);
	alloc_downref (ap);

	/* brutal hack, get around parser creation of list literals with identifiers */

	if(result && result->asr_type == N_ID) {
		ap = st_get(result->asr_ident, Level);
		alloc_downref (result);
		alloc_upref (ap);
		result = ap;
	}

	return(result);
}

/* ----------------------------------------------------------------------------
** Make a value copy of a list - "copy" list in CASL.
*/

asr_t *list_copy(asr_t *list) {
	asr_t *head;
	asr_t **lp = &head;

	for(/**/ ; list; list = list->asr_kids[1]) {
		*lp = asr_node(LIST, NULL, NULL);
		if (list->asr_kids[0]) alloc_upref(list->asr_kids[0]);
		(*lp)->asr_kids[0] = list->asr_kids[0];
		lp = &(*lp)->asr_kids[1];
	}

	*lp = NULL;

	return(head);
}
		
/* ----------------------------------------------------------------------------
** Return the count of items in a list.
** This borrows a reference from the caller.
*/

int list_items(asr_t *a) {
	int c = 0;

	if(a->asr_kids[0] == NULL)
		return(0);

	for(/**/; a; a = a->asr_kids[1])
		c++;

	return(c);	
}

/* ----------------------------------------------------------------------------
** Returns true (1) if the list a is equivelent to the list b, false (0)
** otherwise.  This function borrows a reference from the caller.
*/

int list_eq(asr_t *a, asr_t *b) {
	asr_t *na, *nb;

	for(na = a, nb = b; na; na = na->asr_kids[1], nb = nb->asr_kids[1]) {
		asr_t *eq;

		if(!nb)
			return(0);

		alloc_upref (na->asr_kids[0]);
		alloc_upref (nb->asr_kids[0]);
		eq = asr_node(N_EQ, na->asr_kids[0], nb->asr_kids[0]);
		eq = eval_expr(eq);

		if(eq->asr_integer != 1) {
			asr_sfree (&eq);
			return(0);
		}
		asr_sfree (&eq);
	}
		
	if(na != nb) 
		return(0);

	return(1);
}

/* ----------------------------------------------------------------------------
** Recursively expand the contents of a list into a single list containing all
** the items.
*/

static list_t *expand(asr_t *list, list_t *l);

asr_t *list_expand(asr_t *list) {
	asr_t *head = NULL;
	list_t *l = NULL;
	list_t *l2 = NULL;

	if(list->asr_type == N_CREATELIST)
		list = eval_expr(list);

	if(list->asr_type != LIST) 
		return(asr_node(LIST, list, NULL));

	l = expand(list, l);

	for (l2 = l; l2; l2 = l2->next) {
		head = asr_node (LIST, (asr_t *)l2->data, head);
		if (head->asr_kids[0]) alloc_upref (head->asr_kids[0]);
	}

	l_free(&l);

	return(head);
}

/* ----------------------------------------------------------------------------
** Do the actual expanding work.
*/

list_t *expand(asr_t *list, list_t *l) {
	for(; list; list = list->asr_kids[1]) {
		asr_t *t = list->asr_kids[0];

		if(!t)
			continue;

		if(t->asr_type == N_ID) {
			if(!(t = st_get(t->asr_ident, Level)))
				error(E_USER, 
				      "undefined identifier \"%s\" in list expansion",
				      list->asr_kids[0]->asr_ident);
		} 		       			

		if(t->asr_type == LIST) 
			l = expand(t, l);
		else
			l = l_push(l, t);
	}
		
	return(l);
}

/* ----------------------------------------------------------------------------
** Insert an item at the beginning of the list.  Because the head pointer has
** to stay the same, the simplest thing to do is insert the new node second,
** and swap the asr_kids pointers, so the head is the same, but points to
** a new node.
*/

static void list_item_prepend(asr_t *list, asr_t *item) {
	asr_t *node = asr_node(LIST, NULL, NULL);

	asr_t **kids = node->asr_kids;
	node->asr_kids = list->asr_kids;
	list->asr_kids = kids;
	list->asr_kids[0] = item;
	list->asr_kids[1] = node;
	return;
}

/* ----------------------------------------------------------------------------
** Append an asr node to the end of a list.  Right now this traverses the
** entire list to find the end.  Perhaps data lists can be changed to a
** special item that points to the beginning and the end of the list in the
** first node.
*/

static void list_item_append(asr_t *list, asr_t *item) {
	asr_t *n, *np = 0, *l;

	if(!list->asr_kids[0]) {
		list->asr_kids[0] = item;
		return;
	}

	for(n = list; n; n = n->asr_kids[1]) 
		np = n;

	l = asr_node(LIST, NULL, NULL);

	np->asr_kids[1] = l;
	l->asr_kids[0] = item;
	l->asr_kids[1] = NULL;

	return;
}

/* ----------------------------------------------------------------------------
** Get the head item from a list.  Since the head pointer needs to be
** unchanged, free the second node and move the data from the second to the
** first, and return the origonal data from the first.
*/

static asr_t *list_get_head(asr_t *list) {
	asr_t *deathrow, *next, *result, *headitem;

	result = list->asr_kids[0];
	deathrow = list->asr_kids[1];
	next = 0;
	headitem = 0;
	if (deathrow) {
		headitem = deathrow->asr_kids[0];
		next = deathrow->asr_kids[1];
		deathrow->asr_kids[0] = 0;
		deathrow->asr_kids[1] = 0;
		asr_free (&deathrow);
	}
	list->asr_kids[0] = headitem;
	list->asr_kids[1] = next;
	return (result);
}

/* ----------------------------------------------------------------------------
** Return the last item from the list.  This is where the idea of pointing to
** the last item in the node list wouldn't work as well.
*/

static asr_t *list_get_tail(asr_t *list) {
	asr_t **deathrow = 0, **inmate, *result, *gaschamber;

	if(!list || !list->asr_kids[0]) {
		alloc_upref (null);
		return(null);
	}

	if (!list->asr_kids[1]) {
		result = list->asr_kids[0];
		list->asr_kids[0] = 0;
		return (result);
	}

	inmate =&list;

	while (*inmate) {
		if ((*inmate)->asr_kids[0]) {
			deathrow = inmate;
		}
		inmate = &(*inmate)->asr_kids[1];
	}

	if (*deathrow == list) {
		result = (*inmate)->asr_kids[0];
		asr_free (&list->asr_kids[1]);
		list->asr_kids[0] = 0;
		return (0);
	}

	gaschamber = *deathrow;
	*deathrow = gaschamber->asr_kids[1];
	result = gaschamber->asr_kids[0];
	gaschamber->asr_kids[0] = 0;
	asr_free (&gaschamber);
	return (result);
}

/* -------------------------------------------------------------------- */
