/* sendip.c
	Sends specified ip packets
*/

#define USAGE "Command line options:
	./sendip <hostname> -p <type> -d <data> <options>
	
	<hostname> is the name of the host to send the packet to, defaults to
	localhost
	<type> is the packet type: currently IP, UDP, TCP or ICMP, defaults to IP
	<data> is the data payload of the packet as a string of hexadecimal
	numbers, defaults to 10 zero bytes

	For all options, specify the option, followed a space, then its value.
   Any option can be left out, in which case the default value is used.  For
   all options, if you specify \"r\", sendip will generate a random value.

	OPTION     DESCRIPTION      DEFAULT
	Socket options (values are 0 or 1)
	-sd        Debug            0
	-sr        Don't route      0 (ie do route)
	-sb        Allow broadcast  0
	
	IP options:
	-is        Source IP        127.0.0.1
	-id        Destination IP   Correct
	-ih        Header length    Correct
	-iy        Type of service  0 (see RFC791 et al)
	-il        Length           Correct
	-ii        Identification   Random
	-ifr       Reserved flag    0
	-ifd       Don't fragment   0
	-ifm       More fragments   0
	-if        Frag offset      0
	-it        TTL              255
	-ip        Protocol         Correct for type
	-ic        Checksum         Correct
	-io        Options - not yet implemented
	
	ICMP options:
	-ct        Type             8 (See RFC792)
	-cd        Code             0 (See RFC792)
	-cc        Checksum         Correct
	
	UDP options:
	-us        Source port      0
	-ud        Destination port 0
	-ul        Length           Correct
	-uc        Checksum         Correct
	
	TCP options:
	-ts        Source port      0
	-td		  Destination port 0
	-tn        Sequence number  Random
	-ta        Acknowledgement  0
	-tt        Data offset      Correct
	-tr        Reserved         0 (rfc793, reserved includes ecn and cwr)
   -tfe       Set ecn bit      0 (rfc2481)
   -tfc       Set cwr bit      0 (rfc2481)
	-tfu       Set urg bit      0 unless -tu is specified
	-tfa       Set ack bit      0 unless -ta is specified
	-tfp       Set psh bit      0
	-tfr       Set rst bit      0
	-tfs       Set syn bit      1
	-tff       Set fin bit      0
	-tw        Window           65535
	-tc        Checksum         Correct
	-tu        Urgent pointer   0
	-to..      Options  (no defaults; \"x\" means that you must put
                             an argument there, but it's ignored)
                  0 End of option list
                  -toeol       x
                  1 No op
                  -tonop       x
                  2 Maximum segment size
                  -tomss       512
                  3 Window scale
                  -towscale    0
                  4 Selective Ack permitted
                  -tosackok    x
                  5 Selective Ack (l_edge1:r_edge1,l_edge2:r_edge2...)
                  -tosack      322:338,340:362,364:380
                  8 Timestamp (tsval:tsecr)
                  -tots        123456:443221

   RIP options: (RIP is transmitted using UDP so UDP options also apply)
   You can have up to 25 RIP entries or 1 authenticate (must be first) and up
   to 24 RIP entries in a valid RIP packet. The well-known UDP port for RIP-1
   and RIP-2 is 520. Messages intended for another router's process should be
   sent to UDP 520. Routing update messages should be sent from UDP 520.
   Unsolicited routing update messages (which is what you will be doing, I
   would guess ;-) should have both their origin and destination UDP ports 520.

Command:     Name:        Subcategory:   Default:      Valid range:
  -rv        RIP version                 2               1 or 2
Note that commands 3 and 4 are obsolete while 5 and 6 are undocumented in RFCs.
  -rc        RIP command                 1               1-request, 2-response
                                                         3-traceon, 4-traceoff,
                                                         5-poll, 6-poll entry
For each of the RIP entry fields, type - if you want the default to be used
  -re        RIP entry    Address family AF_INET (2)     AF_INET6 (10) not
                                                         supported (generally)
                                                         for RIP-1
                          Route tag      0               not valid for RIP-1
                          IPv4 address   0.0.0.0         any IPv4 address
                          Subnet mask    255.255.255.0   any netmask - not
                                                         valid for RIP-1
                          Next hop       0.0.0.0         not valid for RIP-1
                          Metric         16              1-16
  -ra        authenticate                Brave new world any string - not valid
                                                         for RIP-1 (quote the
                                                         string!!)
             The authenticate option uses THE FIRST of the 25 availble routing
             entries. To distinguish this type of entry from a routing entry,
             the address family is 0xFFFF (and the authentication tag is 2) 

  RIP actions:
  -rd        RIP default request. Ask for the routers' entire routing tables.
             Do not use any other rip option with -rd.

  Sendip options:
  -v         be verbose, print packet sent in hex/ascii to stderr
  -h         print out this help message
"

#define __USE_BSD   /* GLIBC */
#define _BSD_SOURCE /* LIBC5 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/ip_icmp.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <rpc/types.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include "protocols.h"
#include <stdlib.h>


/* IP_MAXPACKET is not defined on pre-glibc boxes so define it in case */
#ifndef IP_MAXPACKET
#define IP_MAXPACKET 65535
#endif

/* MAXOOSTNAMELEN is not defined on BSD boxes */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* sockaddr_storage is not defined on linux boxes */
#if(!defined _SS_SIZE && !defined _SS_MAXSIZE)
/* Structure large enough to hold any socket address (with the historical
   exception of AF_UNIX).  We reserve 128 bytes.  */
#if ULONG_MAX > 0xffffffff
# define __ss_aligntype __uint64_t
#else
# define __ss_aligntype __uint32_t
#endif
#define _SS_SIZE        128
#define _SS_PADSIZE     (_SS_SIZE - (2 * sizeof (__ss_aligntype)))

struct sockaddr_storage
{
	__SOCKADDR_COMMON (__ss_);  /* Address family, etc.  */
	__ss_aligntype __ss_align;  /* Force desired alignment.  */
    char __ss_padding[_SS_PADSIZE];
};
#endif

void do_sockopt(int s, char *opt, char *val);
void usage_exit(const char *err);

const char *progname;

/* print usage message and exit */
void usage_exit(const char *err) {
	fprintf(stderr,"Usage: %s <hostname> <options>\n",progname);
	fprintf(stderr,err);
	fprintf(stderr,"See sendip.c, sendip -h or the man page for more details\n");
	exit(1);
}

/* handle socket options */
void do_sockopt(int s, char *opt, char *val) {
	const int zero = 0;
	const int one = 1;
	int r;

	if(*val != '0' && *val != '1' && *val != 'r') {
		close(s);
		usage_exit("Value for socket option must be 0, 1 or r\n");
	}
	if(*val=='r') {
		*val=rand()&1;
	} else {
		*val-=0x30;
	}

	switch(opt[2]) {
	case 'd':
	case 'D':
		if(*val)
			r = setsockopt(s, SOL_SOCKET, SO_DEBUG, &one, sizeof(one));
		else
			r = setsockopt(s, SOL_SOCKET, SO_DEBUG, &zero, sizeof(zero));
		if(r)
			perror("Warning: setsockopt(SO_DEBUG)");
		break;
	case 'r':
	case 'R':
		if(*val)
			r = setsockopt(s, SOL_SOCKET, SO_DONTROUTE, &one, sizeof(one));
		else
			r = setsockopt(s, SOL_SOCKET, SO_DONTROUTE, &zero, sizeof(zero));
		if(r)
			perror("Warning: setsockopt(SO_DONTROUTE)");
		break;
	case 'b':
	case 'B':
		if(*val)
			r = setsockopt(s, SOL_SOCKET, SO_BROADCAST, &one, sizeof(one));
		else
			r = setsockopt(s, SOL_SOCKET, SO_BROADCAST, &zero, sizeof(zero));
		if(r)
			perror("Warning: setsockopt(SO_DONTROUTE)");
		break;
	default:
		close(s);
		usage_exit("Unknown socket option\n");
		break;
	}
}	


/* Main program code */
int main(int argc, char **argv)  {
	/* socket stuff */
	int s;                            /* socket for sending   */

	/* hostname stuff */
	char hnamebuf[MAXHOSTNAMELEN];    /* buffer for hostname  */
	char *hostname = NULL;            /* target hostname      */
	struct hostent *host = malloc(sizeof(struct hostent));

	/* protocol independent storage for addresses */
	struct sockaddr_storage *to = malloc(sizeof(struct sockaddr_storage));

	/* casts for specific protocols */
	struct sockaddr_in *to4 = (struct sockaddr_in *)to; /* IPv4 */
	struct sockaddr_in6 *to6 = (struct sockaddr_in6 *)to; /* IPv6 */

	int af_type = AF_INET; /* Address family, default: AF_INET */
	int tolen = sizeof(struct sockaddr_in); /* sockaddr_storage length */

	/* general packet stuff */
	unsigned char *outpack = NULL;    /* actual output packet */
	void *header = NULL;              /* any header           */

	/* packet stuff for IPv4 */
	struct ip *iph = NULL;            /* IP packet header     */
	unsigned char *outp;
	struct rip_options rippack;       /* RIP data packet      */

	struct icmp *icp=NULL;            /* ICMP packet header   */
	struct tcphdr *tcp=NULL;          /* TCP packet header    */
	unsigned char *tcpopts=NULL;      /* TCP options          */
	struct udphdr *udp=NULL;          /* UDP packet header    */
	int rip = 0;                      /* Are we doing RIP     */
	int ripDefault = 0;               /* Are we making default RIP request */

	/* packet stuff for IPv6 */
	struct ip6_hdr *ip6h = NULL;
	struct icmp6_hdr *icmp6 = NULL;

	char *data = NULL;                /* packet data argument */
	int sent;                         /* number of bytes sent */
	int verbose=0;                    /* verbosity            */

	srand(time(NULL));

	/* Variable initializations */
	progname=argv[0];
	if ((outpack = malloc(IP_MAXPACKET)) == NULL) {
		perror("malloc()");
		return 2;
	}
	header = (void *)outpack;
	memset(outpack, 0, IP_MAXPACKET);
	memset(hnamebuf, 0, MAXHOSTNAMELEN);
	memset(to, 0, sizeof(struct sockaddr_storage));

	/* Process command line options - first make sure we have a host name */
	if (argc < 2 ) usage_exit("No hostname specified\n");
	if (argv[1][0]=='-') {
		if(strcmp(argv[1],"-h"))
			usage_exit("Hostname must be first option\n");
		else {
			printf(USAGE);
			return 0;
		}
	}

	hostname=argv[1];

	/* Check address family */
	if (argc > 2 && !strcmp(argv[2], "-A")) {
		if (!strcasecmp(argv[3], "inet6")) {
			af_type = AF_INET6;
		} else if (!strcasecmp(argv[3], "inet")) {
			af_type = AF_INET;
		} else {
			usage_exit("Only inet and inet6 supported.\n");
		}
		argc -= 2;
		argv += 2;
	} else {
		af_type = AF_INET;
	}

	
	if ((host = gethostbyname2(hostname, af_type)) == NULL) {
		perror("Couldn't get destination host: gethostbyname2()");
		return 2;
	};
	switch (af_type) {
	case AF_INET:
		iph = (struct ip *)header;
		to4->sin_family = host->h_addrtype;
		memcpy(&to4->sin_addr, host->h_addr, host->h_length);
		outp = outpack + sizeof(struct ip);
		ip_initialize(iph);
		break;
	case AF_INET6:
		ip6h = (struct ip6_hdr *)header;
		to6->sin6_family = host->h_addrtype;
		memcpy(&to6->sin6_addr, host->h_addr, host->h_length);
		outp = outpack + sizeof(struct ip6_hdr);
		ip6_initialize(ip6h);
		break;
	default:
		exit(-1);
	}

	/* Now we can open the socket before handling the rest of the command line
		options */
	if ((s = socket(af_type, SOCK_RAW, IPPROTO_RAW)) < 0) {
		perror("Couldn't open RAW socket");
		return 2;
	}
        /* @@@@ at 001104 OpenBSD needs this */
        else
        { int on=1;
          if (setsockopt(s, IPPROTO_IP,IP_HDRINCL,&on,sizeof(on)) <0)
          {  perror ("Couldn't setsockopt IP_HDRINCL");
             return 2;
          }
        }


	/* Now process the rest of the command line options */
	argc-=2;
	argv+=2;

	while(argc > 1 || ( argc>0 && strcmp( argv[0], "-rd")==0)) {
		if(argv[0][0]!='-') {
			close(s);
			usage_exit("Option not starting with a -\n");
		}
		switch(argv[0][1]) {
		case 'c':  /* icmp options */
			if(icp)
				do_icmpopt(icp,argv[0],argv[1]);
			else if (icmp6)
				do_icmp6opt(icmp6, argv[0], argv[1]);
			else {
				close(s);
				usage_exit("Can not specify -c* without -p ICMP\n");
			}
			break;
		case 'd':  /* packet data */
			data=argv[1];
			break;
		case 'h':  /* help */
			close(s);
			printf(USAGE);
			exit(0);
			break;
		case 'i':  /* ip options */
			switch (af_type) {
			case AF_INET:
				do_ipopt(iph,argv[0],argv[1]);
				break;
			case AF_INET6:
				do_ip6opt(ip6h, argv[0], argv[1]);
				break;
			default:
				close(s);
				usage_exit("Only AF_INET and AF_INET6 supported\n");
			}
			break;
		case 'p':  /* packet type */
			if(icp || udp || tcp) {
				close(s);
				usage_exit("Can't specify multiple protocols\n");
			}
			switch(argv[1][0]) {
			case 'I':
			case 'i':
				if(strcasecmp(argv[1],"icmp")) {
					close(s);
					usage_exit("Protocol unknown\n");
				}
				switch (af_type) {
				case AF_INET:
					icp=(struct icmp *)outp;
					icmp_initialize(icp);
					outp+=4; /*sizoef(struct icmphdr) includes too much gubbins */;
					break;
				case AF_INET6:
					icmp6 = (struct icmp6_hdr *)outp;
					icmp6_initialize(icmp6);
					outp += 4;
					break;
				}
				break;
			case 'U':
			case 'u':
				if(strcasecmp(argv[1],"UDP")) {
					close(s);
					usage_exit("Protocol unknown\n");
				}
				udp=(struct udphdr *)outp;
				udp_initialize(udp);
				outp+=sizeof(struct udphdr);
				break;
			case 'T':
			case 't':
				if(strcasecmp(argv[1],"TCP")) {
					close(s);
					usage_exit("Protocol unknown\n");
				}
				tcp=(struct tcphdr *)outp;
				tcp_initialize(tcp);
				outp+=sizeof(struct tcphdr);
                                tcpopts=malloc(256);
                                if (!tcpopts)
                                { close(s);
                                  perror("Cannot alloc memory for tcp opts");
                                  exit(2); }
                                *tcpopts=0; /* no tcp option yet */
				break;
			default:
				close(s);
				usage_exit("Protocol unknown\n");
				break;
			}
			break;  /* end of packet type */
		case 'r':  /* RIP options -- this is a UDP protocol */
			if( !udp)
			{
				close(s);
				usage_exit("Can not specify -r* without -p UDP\n");
			}
			else /* specify the rip options */
			{
				rip = 1;
				if( strcmp( argv[0], "-rd") == 0)
				{
				  ripDefault = 1; /* make up for -rd not taking a parameter */
                                  /* use &argv[0] as a placeholder */
				  do_ripopt( &rippack, argv[0], &argv[0]);
				}
				else
				do_ripopt( &rippack, argv[0], &argv[1]);

				if( (strcmp( argv[0], "-re") == 0) ||
				    (strcmp( argv[0], "-ra") == 0) ||
				    (strcmp( argv[0], "-rd") == 0))
				{
					rip_finalize( &rippack, outp);
					outp += 24;
				}
				if( strcmp( argv[0], "-re") == 0)
				{
					argc -= 5;
					argv += 5;
				}
			}
			break;
		case 's':  /* socket options */
			do_sockopt(s,argv[0],argv[1]);
			break;
		case 't':  /* tcp options */
			if(tcp)
				do_tcpopt(tcp, tcpopts,argv[0],argv[1]);
			else {
				close(s);
				usage_exit("Can not specify -t* without -p TCP\n");
			}
			break;
		case 'u':  /* udp options */
			if(udp)
				do_udpopt(udp,argv[0],argv[1]);
			else {
				close(s);
				usage_exit("Can not specify -u* without -p UDP\n");
			}
			break;
		case 'v':  /* be verbose - print packet sent */
			verbose++;
			argc++; argv--;
			break;
		default:  /* something else */
			close(s);
			fprintf( stderr, "%s\n", argv[0]);
			usage_exit("Unknown option\n");
			break;
		}
		if( ripDefault==1) /* -rd takes no parameters */
		{
		  --argc; ++argv;
		}
		else
		{
		argc-=2; argv+=2;
	}
	}

	if(ripDefault==0 && argc==1) {
		close(s);
		if(!strcmp(argv[1],"-h")) {
			printf(USAGE);
			return 0;
		} else {
			usage_exit("Argument needs a parameter\n");
		}
	}


	if(tcp && *tcpopts) {
		/* Put tcp options in ouptack */
		/*    pad with 0, then copy to outpack, adjust outp */
		while (*tcpopts%4) {
			tcpopts[++*tcpopts]=0;
		}
		memcpy(outp, tcpopts+1, *tcpopts);
		outp+=*tcpopts;
		/*    adjust tcp data offset */
		tcp->th_off+=*tcpopts/4;
		/*    free mem */
		free(tcpopts);
		tcpopts=NULL;
	}


	if(!rip) {
		/* finalize packet for sending - first, copy data */
		/* Add data from outp - this code is foul */
		if(data) {
			for(;outp-outpack<IP_MAXPACKET && *data;data++, outp++) {
				if(isdigit(*data)) {
					*outp=(*data)-'0';
				} else if('A' <= *data && *data <= 'F') {
					*outp=(*data)-'A'+0x0a;
				} else if('a' <= *data && *data <= 'f') {
					*outp=(*data)-'a'+0x0a;
				}
				(*outp)*=0x10;
				data++;
				if(*data) {
					if(isdigit(*data)) {
						*outp+=(*data)-'0';
					} else if('A' <= *data && *data <= 'F') {
						*outp+=(*data)-'A'+0x0a;
					} else if('a' <= *data && *data <= 'f') {
						*outp+=(*data)-'a'+0x0a;
					}
				} else data--; /* force loop exit */
			}
		} else {  /* no data - just add 10 bytes to size */
			outp+=10;
			if(outp-outpack > IP_MAXPACKET) 
				outp = outpack+IP_MAXPACKET;
		}
	} /* !rip */

	/* then do any further finalization (eg checksums) */
	switch (af_type) {
	case AF_INET:
		ip_set_addr(header, (struct sockaddr_in *)to);
		if(icp) icmp_finalize(iph, icp, outp-outpack);
		if(udp) udp_finalize(iph, udp, outp-outpack);
		if(tcp) tcp_finalize(iph, tcp, outp-outpack);
		ip_finalize(header, outp-outpack, (struct sockaddr_in *)to);
		tolen = sizeof(struct sockaddr_in);
		break;
	case AF_INET6:
		ip6_set_addr(header, (struct sockaddr_in6 *)to);
		if (icmp6) icmp6_finalize(header, icmp6, outp-outpack);
		if (udp) udp_finalize(header, udp, outp-outpack);
		if (tcp) tcp_finalize(header, tcp, outp-outpack);
		ip6_finalize(header, outp-outpack, (struct sockaddr_in6 *)to);
		tolen = sizeof(struct sockaddr_in6);
		break;
	}
	
	if(verbose) { 
		int i, j;  
		for(i=0; i<outp-outpack; ) {
			for(j=0; j<4 && i+j<outp-outpack; j++)
				fprintf(stderr, "%02x ", outpack[i+j]); 
			fprintf(stderr, "     ");
			for(j=0; j<4 && i+j<outp-outpack; j++)
				fprintf(stderr, "%c", isprint(outpack[i+j])?outpack[i+j]:'.'); 
			fprintf(stderr, "\n");
			i+=j;
		}
	}

	/* Send the packet */
	sent = sendto(s, (char *)outpack, outp-outpack, 0, (void *)to, tolen);
	if (sent == outp-outpack) {
		fprintf(stderr,"Sent %d bytes to %s\n",sent,hostname);
	} else {
		if (sent < 0)
			perror("sendto");
		else
			fprintf(stderr, "Only sent %d of %d bytes to %s\n", 
				sent, outp-outpack, hostname);
	}
	close(s);
	return 0;
}
