/* $Id: pakemon.c,v 1.19 2001/01/05 01:10:51 keiji Exp $ */
/*
** pakemon.c - Pakemon(Packet Monster) Main Source Code
** Author: Keiji Takeda (keiji@sfc.keio.ac.jp)
**
** pakemon is an abbreviation of "Packet Monster", a simple packet
** monitoring misuse detector.
** Copyright (C) 1999, 2000 Keiji Takeda <keiji@sfc.keio.ac.jp>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pakemon.h"
#include "util.h"
#include "interface.h"
#include "signature.h"

#define int_ntoa(x) inet_ntoa(*((struct in_addr *)&x))


/************************************************
  Function: pakemon_syslog
  Purpose: log alerts and additional information
  Argument: int type,
            int errnum,
            struct ip* iph,
	    void *data

  This function is derived from  nids_syslog in libnids.c
  (Original code was written by Rafal Wojtczuk,
   modification for pakemon is done by Keiji Takeda)
 ***********************************************/
void pakemon_syslog(int type, int errnum, struct ip * iph, void *data)
{
  char saddr[20], daddr[20];
  char buf[1024];
  char buf2[1024];
  time_t log_time;

  struct host *this_host;
  unsigned char flagsand = 255, flagsor = 0;
  int i;
  u_int scaned_host = 0;

  FILE *alert_log, *dump_log;
  
  switch (type) {

  case NIDS_WARN_IP:
    if (errnum != NIDS_WARN_IP_HDR){
      strcpy(saddr, int_ntoa(iph->ip_src.s_addr));
      strcpy(daddr, int_ntoa(iph->ip_dst.s_addr));

      if( pkmprm.std_alert )
	{
	  time( &log_time );

	  printf( "%s %s, packet (apparently) from %s to %s\n", 
		 ctime( &log_time ), nids_warnings[errnum], saddr, daddr);

	  /* make beep sound */
	  putchar('\007');
	}

      if( pkmprm.use_syslog )
	{
	  syslog( nids_params.syslog_level,
		 "%s, packet (apparently) from %s to %s\n", 
		 nids_warnings[errnum], saddr, daddr);
	}
      else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
	{
	  time( &log_time );
	  
	  fprintf( alert_log,"%s %s, packet (apparently) from %s to %s\n", 
		   ctime( &log_time ), nids_warnings[errnum], saddr, daddr);
	  fclose( alert_log );
	}
      else
	{
	  time( &log_time );
	  fprintf(stderr, "%s Failed to open alert_log_file.\n", 
		  ctime( &log_time ) );
	}
    }
    else
      {

	/* The header is broken */

	if( pkmprm.use_syslog )
	  {
	    syslog(nids_params.syslog_level, "%s\n", nids_warnings[errnum]);
	  }
	else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
	  {
	    time( &log_time );
	  
	    fprintf( alert_log,"%s %s\n", ctime( &log_time ), nids_warnings[errnum] );
	    
	    fclose( alert_log );
	  }
	else
	  {
	    time( &log_time );
	    fprintf(stderr, "%s Failed to open alert_log_file.\n", ctime( &log_time ));
	  }
      }
    break;

  case NIDS_WARN_TCP:
    strcpy(saddr, int_ntoa(iph->ip_src.s_addr));
    strcpy(daddr, int_ntoa(iph->ip_dst.s_addr));

    if ( errnum != NIDS_WARN_TCP_HDR )
      {

	if( pkmprm.std_alert )
	  {
	    time( &log_time );

	    printf( "%s %s, packet (apparently) from %s to %s\n", 
		 ctime( &log_time ), nids_warnings[errnum], saddr, daddr);

	    /* make beep sound */
	    putchar('\007');
	  }

      if( pkmprm.use_syslog )
	{
	  syslog( nids_params.syslog_level,
		  "%s,from %s:%hi to  %s:%hi\n", nids_warnings[errnum], saddr,
		  ntohs((( struct tcphdr *) data)->th_sport ), daddr,
		  ntohs((( struct tcphdr *) data)->th_dport ) );
	}
      else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
	{
	  time( &log_time );

	  fprintf( alert_log,
		   "%s %s,from %s:%hi to  %s:%hi\n", ctime( &log_time ),
		   nids_warnings[errnum], saddr,
		   ntohs((( struct tcphdr *) data)->th_sport ), daddr,
		   ntohs((( struct tcphdr *) data)->th_dport ) );
	  fclose( alert_log );
	}
      }      
    else
      {
	if( pkmprm.std_alert )
	  {
	    time( &log_time );
	    printf(  "%s %s,from %s to %s\n", ctime( &log_time ), 
		     nids_warnings[errnum], saddr, daddr);
	    /* make beep sound */
	    putchar('\007');
	  }

	if( pkmprm.use_syslog )
	  {
	    syslog( nids_params.syslog_level, "%s,from %s to %s\n",
	       nids_warnings[errnum], saddr, daddr);
	  }
	else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
	  {
	    time( &log_time );
	    fprintf( alert_log, "%s %s,from %s to %s\n",
		     ctime( &log_time ),
		     nids_warnings[errnum], saddr, daddr);
	    fclose( alert_log );
	  }
      }
    break;
  
  case NIDS_WARN_SCAN:
    this_host = (struct host *) data;
    sprintf(buf, "Scan from %s. ", int_ntoa(this_host->addr));
    sprintf(buf2, "Scanned ");
    for (i = 0; i < this_host->n_packets; i++) {
      if( scaned_host != this_host->packets[i].addr )
	{
	  scaned_host = this_host->packets[i].addr;
	  strcat( buf2, int_ntoa( scaned_host ));
	  sprintf( buf2 + strlen(buf2), ":");
	};
      sprintf( buf2 + strlen(buf2), "%hi,", this_host->packets[i].port );
      flagsand &= this_host->packets[i].flags;
      flagsor |= this_host->packets[i].flags;
    }
    if (flagsand == flagsor) {

      i = flagsand;

      switch (flagsand) {

      case 2:
	strcat(buf, "scan type: SYN");
	break;
      case 0:
	strcat(buf, "scan type: NULL");
	break;
      case 1:
	strcat(buf, "scan type: FIN");
	break;
      default:
	sprintf(buf + strlen(buf), "flags=0x%x", i);
      }
    }
    else
      strcat(buf, "various flags");

    scaned_host = 0;

    if( pkmprm.std_alert )
      {
	time( &log_time );
	printf( "%s %s\n", ctime( &log_time), buf );
	/* putchar('\007'); */
      }

    if( pkmprm.use_syslog )
      {
	syslog( nids_params.syslog_level, "%s\n", buf);
      }
    else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
      {
	time( &log_time );
	fprintf( alert_log,  "%s %s\n", ctime( &log_time ), buf );
	fclose( alert_log );
      }

    if( pkmprm.dump_log_mode != LOG_NONE )
      {
	if( ( dump_log = fopen( pkmprm.dump_log_name, "a" )) != NULL )
	  {
	    fprintf( dump_log, "%s\n%s\n\n", buf, buf2 );
	  }
      }

    break;
  
  default:
    if( pkmprm.std_alert )
      {
	time( &log_time );
	printf( "%s Unknown warning number ?\n", ctime( &log_time ) );
	/* putchar('\007'); */
      }

    if( pkmprm.use_syslog )
      {
	syslog(nids_params.syslog_level, "Unknown warning number ?\n");
      }
    else if( ( alert_log = fopen( pkmprm.alert_log_name, "a" )) != NULL )
      {
	time( &log_time );
	fprintf( alert_log,  "%s Unknown warning number ?\n",
		 ctime( &log_time ) );
	fclose( alert_log );
      }
  }
}


/************************************************
  Function: final
  Purpose: Finalize resources
  Argument: int signal
 ***********************************************/
void final( int signal )
{
  struct report *i_report, *j_report;

  if( pkmprm.interface_name != (char *)NULL )
    free_check( pkmprm.interface_name, "final:pkm.interface");

  /*** free report link ***/
  i_report = head_report;

  while( i_report != (struct report *)NULL )
    {
      j_report = i_report;
      i_report = i_report->next_report;
      free_check(j_report, "final:j_report");
    }

  free_s_tree( tcp_tree );
  free_s_tree( udp_tree );
  free_s_tree( icmp_tree );

  if( nids_params.pcap_filter != (char *)NULL )
    free( nids_params.pcap_filter );

  reset_promisc( pkmprm.interface_name );

  puts("\n********** pakemon terminate ***************\n");

  exit( signal );

} /*** final ***/

/************************************************
  Function: Usage
  Purpose: Print how to use this program.
 ***********************************************/
void usage(void)
{
  puts("");
  puts("USAGE:pakemon [-options]");
  puts("Options:");
  puts("        -c <signature>:Use signature(configuration) file <signature>");
  puts("        -i <if>       :set network interface <if>");
  puts("        -u <user id>  :Use <user id> for the process");
  puts("        -v            :Dump all trafic in text format on standard output");
  puts("        -q            :Quiet Mode (Doesn't display alert message on screen.)");
  puts("        -s            :Use syslog to record alerts");
  puts("        -l <alert_log>:Specify <alert_log>");
  puts("        -n            :Log none of payload");
  puts("        -t            :Use only text format for packet_log");
  puts("        -d <dump_log> :Specify <dump_log>");
  puts("        -h            :Show this message.");
  puts("");
  fflush(stdout);
  return;
} /* usage */

/*****************************************************
  Function: read_ip
  Purpose: reads ip packet
  Arguments:
      struct ip_info ip_packet : ip packet to read
******************************************************/

void read_ip(struct ip_info *ip_packet )
{

  return ;

} /* read_ip */

/************************************************
  Function: dump_packet_info
  Purpose: dump packet information on stdout
  Arguments:
     char struct ip_info ip_packet : packet information 
                                     package
 ************************************************/

void dump_packet_info(struct ip_info ip_packet, 
		      struct in_addr local_netaddr,
		      struct in_addr local_netmask )
{

  char header[DEFAULT_BUFFER_SIZE];

  /*** source and destination ip addresses for compalison ***/
  u_long src_ip, dst_ip;

  /*** pointer to get contents of payload in the packet ***/
  u_char *ptr_i;
  
  char *protocol_name;

  switch( (ip_packet.ip_header)->ip_p )
    {
    case IPPROTO_TCP:
      {
	protocol_name = "TCP";
	break;
      }
    case IPPROTO_UDP:
      {
	protocol_name = "UDP";
	break;
      }
    case IPPROTO_ICMP:
      {
	protocol_name = "ICMP";
	break;
      }
    default:
      {
	protocol_name = "UNK";
      }
    }
      
  memset( header, 0, DEFAULT_BUFFER_SIZE );

  dst_ip = (u_long)((ip_packet.ip_header)->ip_dst).s_addr;

  src_ip = (u_long)((ip_packet.ip_header)->ip_src).s_addr;

  if( local_netmask.s_addr == 0 ){

    /*** print the info package ***/
    sprintf( header, "%-5s%-15s(%5lu) > %-15s(%5lu): ",
	     protocol_name,
	     ip_packet.src_IP, ip_packet.src_number,
	     ip_packet.dest_IP, ip_packet.dest_number );

  }else{

    /*** when netmask is set, make order of addresses ***/

    /*** copy destination ip to dst_ip ***/

    /*** destination is on the local network ****/
    if( ( dst_ip & (u_long)local_netmask.s_addr ) == local_netaddr.s_addr )
      {

	/*** source IP also on the local net ***/
	if( ( src_ip & (u_long)local_netmask.s_addr)
	   == local_netaddr.s_addr )
	  {
	    /*** src IP <= dest IP ***/

	    if( src_ip <=  dst_ip )
	      {

		/*** src -> dest order ***/
		sprintf(header, "%-5s%-15s(%5lu) > %-15s(%5lu): ",
			 protocol_name,
			 ip_packet.src_IP, ip_packet.src_number,
			 ip_packet.dest_IP, ip_packet.dest_number );

	      }else {

		/*** src IP > dest IP  ***/

		/*** dest <- src order ***/
		sprintf(header, "%-5s%-15s(%5lu) < %-15s(%5lu): ",
			 protocol_name,
			 ip_packet.dest_IP, ip_packet.dest_number,
			 ip_packet.src_IP, ip_packet.src_number );

	      } /* if srcIP < destIP */

	  }else{

	    /*** src IP is not on local net ***/

	    /*** dest < src ***/

	    sprintf(header, "%-5s%-15s(%5lu) < %-15s(%5lu): ",
		     protocol_name,
		     ip_packet.dest_IP, ip_packet.dest_number,
		     ip_packet.src_IP, ip_packet.src_number);

	  } /* if else (src is on local net ) */

      }else{

	/*** dest IP is not on local net ***/

	sprintf( header, "%-5s%-15s(%5lu) > %-15s(%5lu): ",
		 protocol_name,
		 ip_packet.src_IP, ip_packet.src_number,
		 ip_packet.dest_IP, ip_packet.dest_number);

      } /* if else (dest IP is on local?) */

  } /* else netmask is set */

  printf("%s\n", header);

  /*** print printable contents of the payload ***/
  for ( ptr_i = ip_packet.payload; ptr_i < ip_packet.endof_payload; ptr_i++ )
      {

	/* if the char is printable, '\n' or '\r' then print it. */

	if ( isprint(*ptr_i) )
	  printf("%c", *ptr_i);
	else
	  printf(".");
      } /* for */

  printf("\n");

  return;

} /* dump_packet_info() */

/*****************************************************
  Function: read_payload
  Purpose: read_payload
  Arguments:
      struct ip_info ip_packet : ip packet to process
******************************************************/

void read_payload( struct ip_info *ip_packet )
{

  int i;

  time_t detection_time;

  char *protocol_name;

  u_char *capital; /* buffer to store capitalized payload */

  c_node **c_tree_index;

  FILE *alert_log, *dump_log;

  s_node **s_tree_index = (s_node **)NULL;

  c_node * detect_sig = (c_node *)NULL;

  alert_log = dump_log = (FILE *)NULL;

  switch( (ip_packet->ip_header)->ip_p )
    {

    case IPPROTO_TCP:
      {
	return; /* handled by tcp_callback */
      }
    case IPPROTO_UDP:
      {
	s_tree_index = ( s_node **)malloc_check( sizeof (s_node **),"read_payload:s_tree_index" );

	if( s_tree_index == ( s_node **)NULL )
	  final( 0 );

	protocol_name = "UDP";
	*s_tree_index = udp_tree;
	break;
      }
    case IPPROTO_ICMP:
      {
	s_tree_index = ( s_node **)malloc_check( sizeof (s_node **),"read_payload:s_tree_index" );
	if( s_tree_index == ( s_node **)NULL )
	  final( 0 );

	protocol_name = "ICMP";
	*s_tree_index = icmp_tree;
	break;
      }
    default:
      {
	s_tree_index = ( s_node **)malloc_check( sizeof (s_node **),"read_payload:s_tree_index" );
	protocol_name = "UNK";
	*s_tree_index = (s_node *)NULL;
      }
    }

  c_tree_index = ( c_node **)malloc_check( sizeof (c_node **),"read_payload:c_tree_index" );

  if( c_tree_index == (c_node **)NULL )
    {
      free_check( s_tree_index,"read_payload:s_tree_index" );
      final( 0 );
    }

  capital = ( u_char * )malloc_check( ( ip_packet->payload_len + 1 ) * sizeof( u_char ), "read_payload:capital" );

  if( capital == (u_char *)NULL )
    {
      free_check( s_tree_index,"read_payload:s_tree_index" );
      free_check( c_tree_index,"read_payload:c_tree_index" );
      final( 0 );
    }

  *c_tree_index = (c_node *)NULL;

  /* making capital string */
  for( i = 0; i < ip_packet->payload_len; i++ )
    {
      *(capital + i) = toupper( *( ip_packet->payload + i ) );
    }

  /* termination of data array, just for emergency */
  *( capital + ip_packet->payload_len ) = (u_char)NULL; 

  while( (detect_sig = detect_signature( ip_packet, capital, s_tree_index, c_tree_index )) != (c_node *)NULL )
    {

      *c_tree_index = (*c_tree_index)->next;
      
      if( *c_tree_index == (c_node *)NULL )
	{
	  *s_tree_index = (*s_tree_index)->next;
	}
      
      /* print aleart message */
      time( &detection_time );

      if( pkmprm.std_alert )
	{
	  printf("ALERT:%s %s %s(%lu)->%s(%lu) %s\n", 
		 ctime( &detection_time ),
		 protocol_name,
		 ip_packet->src_IP, ip_packet->src_number,
		 ip_packet->dest_IP, ip_packet->dest_number,
		 detect_sig->name
		 );
	
	  /* make beep sound */
	  putchar('\007');

	}

      if( pkmprm.use_syslog )					    
	{
	  syslog(LOG_ALERT,
		 "ALERT %s %s(%lu)->%s(%lu) %s\n", 
		 protocol_name,
		 ip_packet->src_IP, ip_packet->src_number,
		 ip_packet->dest_IP, ip_packet->dest_number,
		 detect_sig->name
		 );
	
	}
      else if( pkmprm.alert_log_name != (char *)NULL )
	{
	  if( ( alert_log = fopen( pkmprm.alert_log_name, "a"  )) != NULL)
	    {

	      fprintf( alert_log, "\n%s %s(%lu)->%s(%lu) %s",
		       protocol_name,
		       ip_packet->src_IP, ip_packet->src_number,
		       ip_packet->dest_IP, ip_packet->dest_number,
		       ctime( &detection_time ) 
		       );

	      fprintf( alert_log, "%s\n", detect_sig->name );

	      fclose( alert_log );

	    }
	}

      if( pkmprm.dump_log_mode == LOG_NONE )
	continue;

      if ( pkmprm.dump_log_name == (char *)NULL )
	{
	  fprintf(stderr, "No dump_log_name.");
	  continue;
	}
	  
      if( ( dump_log = fopen( pkmprm.dump_log_name, "a" ) ) == (FILE *)NULL )
	{
	  fprintf(stderr, "Failed to open dump log file.");
	  continue;
	}

      fprintf( dump_log, "\n%s %s(%lu)->%s(%lu) %s",
	       protocol_name,
	       ip_packet->src_IP, ip_packet->src_number,
	       ip_packet->dest_IP, ip_packet->dest_number,
	       ctime( &detection_time ) 
	       );

      fprintf( dump_log, "%s (%s %lu %lu)\n", 
	       detect_sig->name,
	       protocol_name,
	       ((s_node *)detect_sig->s_node)->src_number,
	       ((s_node *)detect_sig->s_node)->dest_number
	       );

      if ( pkmprm.dump_log_mode == LOG_TEXT) 
	{
	  text_dump_payload( dump_log, ip_packet, TERM_WIDTH );
	}
      else if( pkmprm.dump_log_mode == LOG_BIN )
	{
	  bin_dump_payload( dump_log, ip_packet );
	}

      fclose( dump_log );
    }

  free_check(s_tree_index, "read_payload:s_tree_index" );
  free_check(c_tree_index, "read_payload:c_tree_index" );
  free_check(capital, "read_payload:capital" );
  return;

} /* read_payload  */

/************************************************
  Function: text_dump_payload
  Purpose: dump payload data on to a given log file
  Arguments:
      FILE *dump_log : file to log 
      struct ip_info * ip_packet : packet to be logged
      int width : width of text in the log file
 ************************************************/
void text_dump_payload(FILE *dump_log, struct ip_info * ip_packet, int width )
{

  int i;

  char *letter;

  /*** print printable contents of the payload ***/
  for ( i = 0; i < ip_packet->payload_len; i++ )
      {

	/* if the char is printable then print it. */

	if ( ( i != 0 ) && ( (i % width) == 0) )
	  fprintf(dump_log, "\n");

	letter = ip_packet->payload + i;

	if ( isprint( *letter ) )
	  fwrite(letter, 1, 1, dump_log);
	else
	  fwrite(".", 1, 1, dump_log);

      } /* for */

  return;
} /* text_dump_payload */

/************************************************
  Function: bin_dump_payload
  Purpose: dump payload data on to a given log file in hex and txt
  Arguments:
      FILE *dump_log : file to log 
      struct ip_info * ip_packet : packet to be logged
      int width : width of text in the log file
 ************************************************/
void bin_dump_payload(FILE *dump_log, struct ip_info * ip_packet )
{

  int i;

  u_char *letter;

  u_char bin_buff[STD_BUF_SIZE], text_buff[STD_BUF_SIZE], tmp_buff[4];

  memset( bin_buff, 0, STD_BUF_SIZE);

  memset( text_buff, 0, STD_BUF_SIZE);

  for ( i = 0; i < ip_packet->payload_len; i++ )
      {

	/* if the char is printable then print it. */

	if ( ( i != 0 ) && ( (i % 16) == 0) )
	  {
	    fprintf( dump_log, "%-48s   %-16s\n", bin_buff, text_buff );
	    memset( bin_buff, 0, STD_BUF_SIZE);
	    memset( text_buff, 0, STD_BUF_SIZE);
	  }

	letter = ip_packet->payload + i;

	sprintf(tmp_buff, "%2x ", *letter);
	strcat(bin_buff, tmp_buff);

	if ( isprint( *letter ) )
	  strncat( text_buff, letter, 1);
	else
	  strncat( text_buff, ".", 1);
      } /* for */
  
  fprintf( dump_log, "%-48s   %-16s\n", bin_buff, text_buff);

  return;

} /* bin_dump_payload */

/************************************************
  Function: ip_callback
  Purpose: handle captured (and reconstructed)
           IP packets
 ***********************************************/
void ip_callback(struct ip * a_packet ) {

  struct ip_info ip_packet; /* pakemon.h */

  /*** initialize ip_packet with 0 ***/
  memset(&ip_packet, 0, sizeof(ip_packet));

  ip_packet.ip_header = a_packet;

  switch( ip_packet.ip_header->ip_p )
    {

    case IPPROTO_TCP:
      return;
    case IPPROTO_UDP:
      return;
    case IPPROTO_ICMP:
      icmp_callback( &ip_packet );
      break;
    default:
      strcpy( ip_packet.src_IP, (char *)inet_ntoa( (ip_packet.ip_header)->ip_src) );

      strcpy( ip_packet.dest_IP, (char *)inet_ntoa( (ip_packet.ip_header)->ip_dst ) );

      ip_packet.payload = (u_char *)(ip_packet.ip_header) + 
	( (ip_packet.ip_header)->ip_hl ) * 4;

      ip_packet.payload_len = ntohs((ip_packet.ip_header)->ip_len);

      ip_packet.endof_payload = (u_char *)a_packet + ip_packet.payload_len;

      read_ip( &ip_packet );
    }

  /* dump the result */
  if(pkmprm.verbose_mode)
    dump_packet_info( ip_packet, local_netaddr, local_netmask);

} /* ip_callback */

/************************************************
  Function: make_filter
  Purpose: create BPF syle filter description from 
           given string as the last option of the command line
 ***********************************************/
void make_filter(int optind, int args, char **argv )
{

  char * filter_ind, * reader;
  int i, filter_len = 0;

  for( i = optind; i < args; i++)
    filter_len += strlen( argv[i]) + 1;

  nids_params.pcap_filter =(char *)malloc_check( filter_len, "make_filter:nids_params." );

  if( nids_params.pcap_filter == NULL )
    {
      fprintf( stderr, "memory error\n");
      exit(0);
    }

  for( i = optind, filter_ind = nids_params.pcap_filter; i < args; i++ )
    {

      reader = argv[i];

      while( *reader != '\0' )
	{
	  *filter_ind++ = *reader++;
	}
      *filter_ind++ = ' ';
    }

  *filter_ind = '\0';

  return;

}  

/************************************************
  Function: init_params()
  Purpose: initialize parameters
 ***********************************************/
void init_params( void )
{

  /* set default parameter */

  pkmprm.verbose_mode   = FALSE;
  pkmprm.std_alert      = TRUE;
  pkmprm.use_syslog     = FALSE;
  pkmprm.interface_name = (char *) NULL;
  pkmprm.sigfile        = DEFAULT_SIG_FILE;
  pkmprm.uid            = 0;
  pkmprm.dump_log_mode  = LOG_BIN;
  pkmprm.alert_log_name = DEFAULT_ALERT_LOG;
  pkmprm.dump_log_name  = DEFAULT_DUMP_LOG;
  nids_params.scan_num_ports = SCAN_PORT_NUM;
  nids_params.syslog    = pakemon_syslog;

  /* initialize global valuables */

  head_report           = (struct report *)NULL;

  /* initialize local network address */
  local_netaddr.s_addr = 0;

  /* initialize local netmask */
  local_netmask.s_addr = 0;

  return;

}  /* init_params */

/************************************************
  Function: init_pcap()
  Purpose: initialize pcap interface
           confirm pkmprm.interface_name
	   get local network address/mask
  Returns: 0 on failure
           1 on success	   
 ***********************************************/
int init_pcap( void )
{

  /* packet descriptor to get local information */
  pcap_t * packet_descriptor;

  packet_descriptor = open_pcap( &( pkmprm.interface_name ) );

  if( packet_descriptor == (pcap_t *)NULL )
    {
      fprintf( stderr, "Failed opening packets caputure interface\n");
      fprintf( stderr, "You need root privilege to make the interface promiscuous.\n");
      return ( 0 ); /* failure */
    }

  /* get network address and net mask --interface.c */
  get_local_info( pkmprm.interface_name, &local_netaddr, &local_netmask );

  pcap_close( packet_descriptor );

  printf( "local_netaddr= %s\n", (char *)inet_ntoa( local_netaddr ) );

  printf( "local_netmask= %s\n", (char *)inet_ntoa( local_netmask ) );
  
  return ( 1 ); /* success */

} /* init_pcap */

/************************************************
  Function: init_log()
  Purpose: initialize log files 
 ***********************************************/
int init_log( void )
{
  FILE *alert_log = (FILE *)NULL;

  FILE *dump_log = (FILE *)NULL;

  if( pkmprm.alert_log_name != (char *)NULL )
    {

      if ( ( alert_log = fopen( pkmprm.alert_log_name, "a" ) ) == NULL )
	{
	  fprintf( stderr, "Cannot Open Alert Log File :%s \n", pkmprm.alert_log_name );
	  return( 0 );
	}

      printf( "Alert Log File= %s\n", pkmprm.alert_log_name );

      fclose( alert_log );

    }
  else if( pkmprm.use_syslog )
    {
      printf( "Using Syslog\n" );
    }

  if( ( pkmprm.dump_log_mode != LOG_NONE ) &&
      ( pkmprm.dump_log_name != (char *)NULL )      )
    {
      if ( ( dump_log = fopen( pkmprm.dump_log_name, "a" ) ) == NULL )
	{

	  fprintf( stderr, "Cannot Open Dump Log File :%s\n", 
		   pkmprm.dump_log_name );

	  return ( 0 );
	}

      printf( "Dump Log File= %s\n", pkmprm.dump_log_name );

      fclose( dump_log );
    }
  else
    {
      puts( "Dump Log Disabled. \n" );
    }
  
  return ( 1 );

} /* init_log */

/************************************************
  Function: Main Function
  Purpose: This is invoked when the pakemon command is 
           excuted. Initiates functions of the
	   program and waits incoming packets.
  Options:
   -c <signature> :Use signature(configuration) file <signature>
   -i <if>        :set network interface <if>
   -u <user id>   :Use <user id> for the process insted of root
   -v             :Dump all trafic in text format on sandard output
   -q             :Quiet Mode (Doesn't display alert message on screen.)
   -s             :Use syslog to record alerts
   -l <alert_log> :Specify <alert_log>
   -n             :Log none of payload
   -t             :Use only text format for packet_log
   -d <dump_log>  :Specify <dump_log>
   -h             :Show this message.

 ***********************************************/

int main(int args, char *argv[])
{

  int option;       /* command line options  */

  init_params();

  /***** read options *****/
  while(( option = getopt(args, argv, "c:i:u:vsl:ntd:h")) != -1)
    {

      switch( option )
	{

	case 'c':        /* specify signature file */
	  pkmprm.sigfile = optarg;
	  break;

	case 'i':        /* to specify NIC, added by Y.Yamano */
	  pkmprm.interface_name = optarg;
	  break;

	case 'u':
	  pkmprm.uid = atoi (optarg);
	  break;

	case 'v':        /* verbose mode display every packet data on stdio */
	  pkmprm.verbose_mode   = TRUE;
	  break;
	
	case 'q':        /* quiet mode simply doesn't show alert */
	  pkmprm.std_alert   = FALSE;
	  break;
	
	case 's':
	  pkmprm.use_syslog = TRUE;
	  pkmprm.alert_log_name = (char *)NULL;
	  openlog("pakemon", 0, LOG_LOCAL0 );
	  break;

	case 'l':        /* specify alert log  */
	  pkmprm.alert_log_name =  optarg;
	  break;

	case 'n':
	  pkmprm.dump_log_mode = LOG_NONE;
	  pkmprm.dump_log_name = (char *)NULL;
	  break;

	case 't':
	  if( pkmprm.dump_log_mode == LOG_NONE )
	    {
	      fputs("-t option confict", stderr );
	      break; 
	    }
	  pkmprm.dump_log_mode = LOG_TEXT;
	  break;

	case 'd':       /* specify dump log  */
	  if( pkmprm.dump_log_mode == LOG_NONE )
	    {
	      fputs("-d option confilict", stderr );
	      break;
	    }

	  pkmprm.dump_log_name = optarg;
	  break;

	case 'h':
	  usage();
	  exit(0);

	default:
	  usage();
	  exit(0);

	} /* switch */
    }/* while */

  if( optind < args )
    make_filter( optind, args, argv);

  /****** signal handlings for finalization ***********/
  /* final() handles finalization when signal is sent */
  /****************************************************/

  signal( SIGHUP, final);
  signal( SIGINT, final);
  signal( SIGTERM, final);
  signal( SIGKILL, final); /* cannot be handled on linux */
  signal( SIGQUIT, final);

  /* initalize and check if the interface works */
  if( !init_pcap() ){
    puts( "pcap Initialization Failure" );
    final( 0 );
  }

  if( !init_signature( pkmprm.sigfile ) ){
    puts( "Signature Initialization Failure" );
    final( 0 );
  }

  printf("Signature File= %s\n", pkmprm.sigfile );

  if( !init_log() ){
    fprintf( stderr, "Signature Initialization Failure.\n");
    final( 0 );
  }

  nids_params.device = pkmprm.interface_name;

  if ( !nids_init () )
    {
      fprintf( stderr,"%s\n", nids_errbuf );
      final( 0 );
    }

  if( pkmprm.uid != 0 )setuid( pkmprm.uid );

  /*** files can be seen only by this user ***/
  umask( 077 );

  nids_register_ip( ip_callback );

  nids_register_tcp ( tcp_callback );

  nids_register_udp ( udp_callback );

  puts("");

  nids_run ();

  return 0;

}
