source decode.tcl
source table_edit.tcl

wm withdraw .
after idle {
	capture::new .analyzer
}

namespace eval capture {
	packet pkt_view 4096
	packet pkt_capture 4096
	packet pkt_tmp
	packet pkt_default
}


proc capture::new {w} {
variable $w
upvar 0 $w data


	destroy $w
	toplevel $w
	wm geometry $w +200+10

	#-----------
	# first pane
	set p [frame $w.pane1 -height 200 -width 550]

	# row of buttons
	set f [frame $p.buttons]
	  button $f.clear -font fixed -bd 1 -text "Clear" -width 10 \
				-command "capture::packets:clear $w"
	  set data(w:start) [button $f.start -font fixed -bd 1 -text "Start" \
				-width 10 -command "capture::packets:start $w"]
	  entry $f.filter -font fixed -bd 1 -textvariable capture::${w}(v:filter) \
			-width 30 -selectbackground lightskyblue -selectborderwidth 0 -bg gray92

	  pack $f.clear $f.start $f.filter -side left

	# the summary
	set data(w:sum:header) [canvas $p.ch -bd 0 -width 400 -height 14]
	  $p.ch create text 3 2 -anchor nw -font fixed -text "Time"
	  $p.ch create text 100 2 -anchor nw -font fixed -text "Len"
	  $p.ch create text 130 2 -anchor nw -font fixed -text "Summary"

	set sb_sum [scrollbar $p.sbsum -bd 1 -orient vertical -highlightthickness 0 \
			-takefocus 0 -width 8]
      $sb_sum configure -command "$p.c yview"

	set data(w:sum) [canvas $p.c -bd 1 -relief sunken -width 400 -height 140 -takefocus 1 \
			-highlightthickness 0 -yscrollcommand "$sb_sum set" \
			-yscrollincrement 14]
	  set data(v:selection) [$p.c create rect -1 -1 -1 -1 -outline lightblue -fill lightblue]
	  $p.c lower $data(v:selection) all
	  bind $p.c <1> "capture::packets:select $w %y"
	  bind $p.c <ButtonPress-3> "capture::packets:actionMenu $w %x %y %X %Y"
	  bind $p.c <Enter> "$p.c delete selectedframe"

	# the layout
	grid x			$f					-sticky news
	grid x			$data(w:sum:header)	-sticky news
	grid $sb_sum	$data(w:sum)		-sticky news
	grid rowconfigure $p {0 1} -weight 0
	grid rowconfigure $p {2} -weight 1
	grid columnconfigure $p {0} -weight 0
	grid columnconfigure $p {1} -weight 1
	grid propagate $p 0


	#------------
	# second pane 
	set p [frame $w.pane2 -height 150]

	# the raw data
	set sb_raw [scrollbar $p.sbraw -bd 1 -orient vertical -highlightthickness 0 \
			-takefocus 0 -width 8]
      $sb_raw configure -command "$p.raw yview"

	set data(w:raw) [text $p.raw -bd 1 -relief sunken -yscrollcommand "$sb_raw set" \
			-height 5 -font fixed -cursor {}]
	  bindtags $data(w:raw) "$p.raw $w all"
	  $p.raw tag configure highlight -background salmon

	# the layout
	grid $sb_raw	$data(w:raw)	-sticky news
	grid configure $sb_raw -pady 1
	grid rowconfigure $p {0} -weight 1
	grid columnconfigure $p {0} -weight 0
	grid columnconfigure $p {1} -weight 1
	grid propagate $p 0


	#-----------
	# third pane 
	set p [frame $w.pane3 -height 300]

	# the mappers
	  # the list of decoded mappers
	  set data(w:map) [text $p.map -bd 1 -relief sunken	-height 10 -width 10 \
			-font fixed -cursor {}]
		bindtags $data(w:map) "$p.map $w all"
		bind $data(w:map) <1> "capture::mapper:select $w %x %y"
		$p.map tag configure highlight -background salmon

	  # the display of the mapper
	  set data(w:disp) [table::new $p -size 20 -highlight 1 -highlightbg salmon \
									-highlightcmd "capture::mapper:highlight $w" \
									-tip "capture::mapper:tip $w"]

	# the layout
	grid [frame $p.dum -width 10]	$data(w:map)	$data(w:disp)	-sticky news
	grid configure $data(w:disp) -pady 1
	grid rowconfigure $p {0} -weight 1
	grid columnconfigure $p {0 1} -weight 0
	grid columnconfigure $p {2} -weight 1
	grid propagate $p 0


	#--------------------------------------------------------------
	# the pane
	frame $w.rsz1 -bg gray -height 2 -cursor sb_v_double_arrow
	frame $w.rsz2 -bg gray -height 2 -cursor sb_v_double_arrow

	pane::new:vertical $w.pane1 $w.rsz1 $w.pane2
	pane::new:vertical $w.pane2 $w.rsz2 $w.pane3

	pack $w.pane1 -fill x
	pack $w.rsz1 -fill x
	pack $w.pane2 -fill x
	pack $w.rsz2 -fill x
	pack $w.pane3 -fill both -expand true


	# general bindings
	bind Scroll <Up> "capture::packets:scroll $w %K"
	bind Scroll <Down> "capture::packets:scroll $w %K"
	bind Scroll <Shift-Up> "capture::mapper:scroll $w %K"
	bind Scroll <Shift-Down> "capture::mapper:scroll $w %K"
	bind Scroll <Mod1-Up> "capture::mapper:scroll $w %K"
	bind Scroll <Mod1-Down> "capture::mapper:scroll $w %K"
	bindtags $w "$w Scroll all"

	# the combo box for edition
	set combo [toplevel $w.combo]
	  wm overrideredirect $combo 1
	  wm withdraw $combo

	  set data(w:combo) [text $combo.t -bd 1 -width 42 -height 10 -font fixed \
			-takefocus 0 -wrap none]
	  bindtags $combo.t "$combo.t . all"
	  $combo.t tag configure highlight -background lightskyblue
	  bind $combo.t <Motion> "capture::edit:combo:scroll $w Motion %x %y"
	  bind $combo.t <1> {
			set tmp [%W get "@%x,%y linestart" "@%x,%y lineend"]
			set __edit__ [lindex $tmp 0]
	  }
	  pack $combo.t


	# action menu
	menu $w.menu -tearoff 0 -bd 1 -activeborderwidth 1
	  $w.menu add command -label "Dump packet" \
			-command "puts \"\\n\[capture::dump:packet $w\]\""
	  $w.menu add command -label "Dump mappers" \
			-command "puts \"\\n\[capture::dump:mappers $w\]\""
	  $w.menu add separator
	  $w.menu add command -label "Generate packet" \
			-command "puts \"\\n\[capture::dump:generate $w\]\""


	# some init
	set data(v:nextline) 0
	set data(v:index) 0
}

#---------------------------------------------------------------------------------------

proc capture::set:filter {w filter} {
	set capture::${w}(v:filter) $filter
}

#---------------------------------------------------------------------------------------

proc capture::packets:clear {w} {
variable $w
upvar 0 $w data


	packets:stop $w
	set data(v:nextline) 0
	set data(v:index) 0

	$data(w:sum) delete all
	  set data(v:selection) [$data(w:sum) create rect -1 -1 -1 -1 \
			-outline lightblue -fill lightblue]
	  $data(w:sum) lower $data(v:selection) all
	$data(w:raw) delete 1.0 end
	$data(w:map) delete 1.0 end
	$data(w:disp).text delete 1.0 end

	foreach n [array names data p:*] {
		unset data($n)
	}
	foreach n [array names data u:*] {
		unset data($n)
	}
}

proc capture::packets:add {w time len summary} {
variable $w
upvar 0 $w data

	set y [expr $data(v:nextline)*14]
	incr data(v:nextline)
	$data(w:sum) create text 3 $y -font fixed -anchor nw -text $time
	$data(w:sum) create text 100 $y -font fixed -anchor nw -text $len
	$data(w:sum) create text 130 $y -font fixed -anchor nw -text $summary

	$data(w:sum) configure -scrollregion [list 0 0 10000 $y]
}

proc capture::packets:start {w} {
variable $w
upvar 0 $w data

	$data(w:start) configure -bg LightPink2 -activebackground pink1 \
		-text "Stop" -command "capture::packets:stop $w"

	set data(v:capturing) 1
	capture::packets:receive $w
}

proc capture::packets:receive {w} {
variable $w
upvar 0 $w data

	while {$data(v:capturing)} {
		set nb [pkt_capture capture -timeout 500 -filter $data(v:filter)]
		if {$nb > 0} {
			set data(p:$data(v:nextline)) [pkt_capture getdata]
			set time [pkt_capture timestamp]
			pkt_capture decodefrom ethernet
			packets:add $w [clock format [lindex $time 0] -format "%H:%M:%S"].[format %06d [lindex $time 1]] \
					$nb [decode::summary pkt_capture 1]
		}
		$data(w:sum) yview moveto 1.0
		update
	}
}

proc capture::packets:insert {w pkt} {
variable $w
upvar 0 $w data


	pkt_capture setfrom [$pkt getdata]
	set data(p:$data(v:nextline)) [pkt_capture getdata]
	set time [pkt_capture timestamp]
	if {[lindex $time 0] == 0} {
		set time [list [clock seconds] 0]
	}
	pkt_capture decodefrom ethernet
	packets:add $w [clock format [lindex $time 0] -format "%H:%M:%S"].[format %06d [lindex $time 1]] \
			[$pkt size] [decode::summary pkt_capture 1]

	$data(w:sum) yview moveto 1.0
	update
}

proc capture::packets:stop {w} {
variable $w
upvar 0 $w data

	$data(w:start) configure -bg lightgray -activebackground #ececec \
			-text "Start" -command "capture::packets:start $w"

	set data(v:capturing) 0
}

#---------------------------------------------------------------------------------------

proc capture::packets:scroll {w key} {
variable $w
upvar 0 $w data


	if {$key == "Up" && $data(v:current)>0} {
		incr data(v:current) -1
	} elseif {$key == "Down" && $data(v:current)<$data(v:nextline)-1} {
		incr data(v:current)
	}

	packets:select $w $data(v:current) 1
}


#---------------------------------------------------------------------------------------

proc capture::packets:select {w y {direct 0}} {
variable $w
upvar 0 $w data


	if {$data(v:nextline) < 1} {
		return
	}

	if {$direct} {
		$data(w:sum) coord $data(v:selection) 0 [expr $y*14] 10000 [expr $y*14+13]
		# TODO : sync view to see item
	} else {
		set y [expr int([$data(w:sum) canvasy $y]/14)*14]
		set data(v:current) [expr $y/14]
		if {$data(v:current) >= $data(v:nextline)} {
			return
		}
		$data(w:sum) coord $data(v:selection) 0 $y 10000 [expr $y+13]
	}

	# get data
	pkt_view setfrom $data(p:$data(v:current))
	# raw dump
	$data(w:raw) delete 0.0 end
	$data(w:raw) insert 0.0 [dumpraw $data(p:$data(v:current))]
	# decode the packet
	$data(w:map) delete 0.0 end
	set data(v:mappers) [pkt_view decodefrom ethernet]
	foreach m $data(v:mappers) {
		$data(w:map) insert end "$m\n"
	}
	# add a data pseudo-mapper
	$data(w:map) insert end "data\n"
	# init display	
	$data(w:map) tag remove highlight 1.0 end
	set pos "[expr $data(v:index)+1].0"
	$data(w:map) tag add highlight $pos "$pos lineend"
	mapper:display $w $data(v:index)
}

#=======================================================================================

proc capture::packets:actionMenu {w x y X Y} {
variable $w
upvar 0 $w data

	# get the packet if any
	if {$data(v:nextline) < 1} {
		return
	}
	set y [expr int([$data(w:sum) canvasy $y]/14)*14]
	set data(t:actionline) [expr $y/14]
	if {$data(t:actionline) > $data(v:nextline)} {
		return
	}
	# draw frame around selected packet
	$data(w:sum) create rectangle 1 $y [expr [winfo width $data(w:sum)]-2] \
			[expr $y+13] -tag selectedframe
	# post popup
	tk_popup $w.menu $X $Y
}


#=======================================================================================


proc capture::raw:highlight {w limits} {
variable $w
upvar 0 $w data


	# remove old highlighting
	$data(w:raw) tag remove highlight 0.0 end
	# if same limits return
	if {[lindex $limits 0] > [lindex $limits 1]} {
		return
	}
	# compute the ranges
	set sline [expr int([lindex $limits 0]/16)+1]
	set schar [expr [lindex $limits 0] % 16]
	set eline [expr int([lindex $limits 1]/16)+1]
	set echar [expr [lindex $limits 1] % 16]
	# do it : first line, then whole line, then remaining char on last line
	set shexa "$sline.[expr 10+3*$schar]"
	set ehexa "$sline.57"
	set sascii "$sline.[expr 59+$schar]"
	set eascii "$sline.75"
	while {$sline < $eline} {
		$data(w:raw) tag add highlight $shexa $ehexa
		$data(w:raw) tag add highlight $sascii $eascii
		# next line
		incr sline
		set shexa "$sline.10"
		set ehexa "$sline.57"
		set sascii "$sline.59"
		set eascii "$sline.75"
	}
	# last part
	$data(w:raw) tag add highlight $shexa "$sline.[expr 10+3*($echar+1)-1]"
	$data(w:raw) tag add highlight $sascii "$sline.[expr 59+($echar+1)]"
}

#---------------------------------------------------------------------------------------

proc capture::mapper:scroll {w key} {
variable $w
upvar 0 $w data


	if {$key == "Up" && $data(v:index)>0} {
		incr data(v:index) -1
	} elseif {$key == "Down" && $data(v:index)<[llength $data(v:mappers)]} {
		incr data(v:index)
	}

	mapper:select $w $data(v:index) dummy 1
}

#---------------------------------------------------------------------------------------

proc capture::mapper:select {w x y {direct 0}} {
variable $w
upvar 0 $w data


	if {$direct} {
		set index [expr $x+1].0
	} else {
		set index [$data(w:map) index "@$x,$y linestart"]
	}
	set mapper [$data(w:map) get $index "$index lineend"]

	if {$mapper == ""} {
		return
	}

	$data(w:map) tag remove highlight 1.0 end
	$data(w:map) tag add highlight $index "$index lineend"
	mapper:display $w [expr int($index)-1]
}

#---------------------------------------------------------------------------------------

proc capture::mapper:display {w index} {
variable $w
upvar 0 $w data

	set data(v:index) $index

	table::clear $data(w:disp)

	if {$index < 0 || $index > [llength $data(v:mappers)]} {
		return
	}
	set mapper [lindex $data(v:mappers) $index]
	set text $data(w:disp).text
	if {$mapper == ""} {
		# data pseudo-mapper
		set lastmapper [lindex $data(v:mappers) [expr $index-1]]
		set start [expr [lindex [pkt_view $lastmapper limits] 1]+1]
		set end [expr [pkt_view size]-1]
		set limits [list $start $end]

		# special case of data
		$text insert insert "[pkt_view gettext -offset $start]"
		bindtags $text "$text Text $w all"
		$text configure -insertofftime 300 -insertontime 600 -background gray92

	} else {
		# simple mapper
		bindtags $text "$text Table $w all"
		$text configure -insertofftime 1000000000 -insertontime 0 -background lightgray
		# display fields of the mapper
		capture::mapper:display:item $w $mapper "" ""
		# get the limits
		set limits [pkt_view $mapper limits]
	}

	# highlight whole mapper
	raw:highlight $w $limits
}


proc capture::mapper:display:item {w mapper item level} {
variable $w
upvar 0 $w data


#puts "mapper:display:item $w $mapper '$item' '$level'"
#if {$level == "        "} {exit}

	if {$item == ""} {
		set list [pkt_view $mapper configure]
	} else {
		set list [pkt_view $mapper configure $item]
	}

	foreach {name value} $list {
		set type [pkt_view $mapper type $item$name]
#puts "    ($level) item = $item$name  --  type = $type"

		if {[lindex $type 0] == "array" && [lindex $type 1] == "mapper"} {
			table::add:noeditable $data(w:disp) "$level$name" $item$name $mapper
			set i 0
			foreach dummy $value {
				table::add:noeditable $data(w:disp) "  $level${name}($i)" $item${name}($i) $mapper
				mapper:display:item $w $mapper "$item${name}($i)" "    $level"
				incr i
			}
		} else {
			if {[lindex $type 1] == "mapper"} {
				table::add:noeditable $data(w:disp) "$level${name}" $item${name} $mapper
				mapper:display:item $w $mapper "$item$name" "  $level"
			} else {
				table::add:editable $data(w:disp) "$level$name" $item$name $mapper $value
			}
		}
	}
}

#---------------------------------------------------------------------------------------

proc capture::mapper:highlight {w option tag} {
variable $w
upvar 0 $w data


	set data(t:mapper) $tag
	if {$data(t:mapper) == "" || $option == "\n"} {
		return
	}

	raw:highlight $w [pkt_view $data(t:mapper) limits $option]
}

#---------------------------------------------------------------------------------------

proc capture::mapper:tip {w option tag mode} {
variable $w
upvar 0 $w data


	if {$mode == "help"} {
		return [pkt_view $tag help $option]
	} else {
		set type [lindex [pkt_view $tag type $option] 1]
		if {$type == "list" || $type == "flags"} {
			set raw [pkt_view $tag getraw $option]
			return "$raw ([format 0x%x $raw]) - [pkt_view $tag getverbose $option]"
		} elseif {$type == "address_ip"} {
			return "[pkt_view $tag getverbose $option] ([pkt_view $tag getraw $option])"
		} elseif {[lindex $type 0] == "record"} {
			return "[pkt_view $tag getverbose $option]"
		}
	}

	return ""
}


#=======================================================================================


proc capture::dump:generate {w} {
variable $w
upvar 0 $w data

	# get packet and decode it
	pkt_tmp setfrom $data(p:$data(t:actionline))
	# generate code for it
	set mappers {}
	set last ""
	set result "<packet> resize [expr [pkt_tmp size]+1]\n"
	append result "<packet> fill 0\n"
	append result "<packet> deletemapper\n\n"

	foreach m [pkt_tmp decodefrom ethernet] {
		append result "#---------------------------------------------------------------------------------------\n"
		append result "#    $m\n"
		append result "#---------------------------------------------------------------------------------------\n"
		append result "<packet> newmapper $m -setdefaults\n"

		# set default on reference packet
		pkt_default fill 0
		pkt_default deletemapper
		pkt_default newmapper $m -setdefaults
		append result "<packet> $m configure"
		# output diff against default
		set empty 1
		foreach {option actual} [pkt_tmp $m configure] {
			if {[string compare $actual [pkt_default $m configure $option]]} {
				set empty 0
				append result " \\\n"
				if {[info exists data(u:$m:$option)]} {
					append result "\t\t$option $data(u:$m:$option)"
				} else {
					append result "\t\t$option [list $actual]"
				}
			}
		}
		if {$empty} {
			set c [pkt_tmp $m configure]
			set option [lindex $c 0]
			set actual [lindex $c 1]
			append result " \\\n"
			append result "\t\t$option [list $actual]"
		}
		append result "\n\n"
		set last $m
	}

	# data of the packet
	append result "#---------------------------------------------------------------------------------------\n"
	append result "#    data not decoded\n"
	append result "#---------------------------------------------------------------------------------------\n"
	set offset [expr [lindex [pkt_tmp $m limits] 1]+1]
	append result "<packet> data \"[dumptext [pkt_tmp getdata -offset $offset]]\" \n"
	return $result
}

#---------------------------------------------------------------------------------------

proc capture::dump:mappers {w} {
variable $w
upvar 0 $w data

	# get packet and decode it
	pkt_tmp setfrom $data(p:$data(t:actionline))
	# dump the mappers
	set result ""
	foreach m [pkt_tmp decodefrom ethernet] {
		append result "[dumpmapper pkt_tmp $m]\n"
	}
	return $result
}

#---------------------------------------------------------------------------------------

proc capture::dump:packet {w} {
variable $w
upvar 0 $w data

	# dump
	return "<packet> setfrom \"[dumptext $data(p:$data(t:actionline))]\""
}




#=================================================================================================
#	Manage a basic pane windows
#

namespace eval pane {} {
}

proc pane::new:horizontal {left rsz right} {
variable pane

	set pane($rsz:horizontal) 1
	set pane($rsz:left) $left
	set pane($rsz:rsz) $rsz
	set pane($rsz:right) $right

	bind $rsz <ButtonPress-1> "+pane::setup $rsz %x %y"
	bind $rsz <Button1-Motion> "+pane::move $rsz %x %y"
	bind $rsz <ButtonRelease-1> "+pane::doIt $rsz"
}

proc pane::new:vertical {left rsz right} {
variable pane

	set pane($rsz:horizontal) 0
	set pane($rsz:left) $left
	set pane($rsz:rsz) $rsz
	set pane($rsz:right) $right

	bind $rsz <ButtonPress-1> "+pane::setup $rsz %x %y"
	bind $rsz <Button1-Motion> "+pane::move $rsz %x %y"
	bind $rsz <ButtonRelease-1> "+pane::doIt $rsz"
}

proc pane::setup {w x y} {
variable pane

	set pane($w:x) [winfo rootx $pane($w:rsz)]
	set pane($w:y) [winfo rooty $pane($w:rsz)]
	set pane($w:w) [winfo width $pane($w:rsz)]
	set pane($w:h) [winfo height $pane($w:rsz)]
	set pane($w:oldx) $x
	set pane($w:oldy) $y

	if {$pane($w:horizontal)} {
		tmpLine [expr $pane($w:x)+$x] $pane($w:y) \
				[expr $pane($w:x)+$x] [expr $pane($w:y)+$pane($w:h)] $w
	} else {
		tmpLine $pane($w:x) [expr $pane($w:y)+$y] \
				[expr $pane($w:x)+$pane($w:w)] [expr $pane($w:y)+$y] $w
	}
}

proc pane::move {w x y} {
variable pane

	set min 20

	if {$pane($w:horizontal)} {
		tmpLine [expr $pane($w:x)+$pane($w:oldx)] $pane($w:y) \
				[expr $pane($w:x)+$pane($w:oldx)] [expr $pane($w:y)+$pane($w:h)] $w

		set width [expr [winfo width $pane($w:left)]+$x]
		if {$width<$min} {
			set x [expr $min-[winfo width $pane($w:left)]]
		}
		set width [expr [winfo width $pane($w:right)]-4-$x]
		if {$width<$min} {
			set x [expr [winfo width $pane($w:right)]-4-$min]
		}

		tmpLine [expr $pane($w:x)+$x] $pane($w:y) \
				[expr $pane($w:x)+$x] [expr $pane($w:y)+$pane($w:h)] $w
		set pane($w:oldx) $x
	} else {
		tmpLine $pane($w:x) [expr $pane($w:y)+$pane($w:oldy)] \
				[expr $pane($w:x)+$pane($w:w)] [expr $pane($w:y)+$pane($w:oldy)] $w

		set height [expr [winfo height $pane($w:left)]+$y]
		if {$height<$min} {
			set y [expr $min-[winfo height $pane($w:left)]]
		}
		set height [expr [winfo height $pane($w:right)]-4-$y]
		if {$height<$min} {
			set y [expr [winfo height $pane($w:right)]-4-$min]
		}
		tmpLine $pane($w:x) [expr $pane($w:y)+$y] \
				[expr $pane($w:x)+$pane($w:w)] [expr $pane($w:y)+$y] $w
		set pane($w:oldy) $y
	}

}

proc pane::doIt {w} {
variable pane

	if {$pane($w:horizontal)} {
		tmpLine [expr $pane($w:x)+$pane($w:oldx)] $pane($w:y) \
				[expr $pane($w:x)+$pane($w:oldx)] [expr $pane($w:y)+$pane($w:h)] $w

		$pane($w:left) configure -width [expr [winfo width $pane($w:left)]+$pane($w:oldx)]
		$pane($w:right) configure -width [expr [winfo width $pane($w:right)]-4-$pane($w:oldx)]
	} else {
		tmpLine $pane($w:x) [expr $pane($w:y)+$pane($w:oldy)] \
				[expr $pane($w:x)+$pane($w:w)] [expr $pane($w:y)+$pane($w:oldy)] $w

		$pane($w:left) configure -height [expr [winfo height $pane($w:left)]+$pane($w:oldy)]
		$pane($w:right) configure -height [expr [winfo height $pane($w:right)]-$pane($w:oldy)]
	}
}
