/* File   : makearp.c
 * Author : J Hadi Salim <ad828@freenet.carleton.ca>
 * Purpose: Create an ARP packet.
 *
 * Notes  : This file has been modified by Karyl F. Stein <xenon@xenos.net>.
 *          Any bugs are my fault.
 *
 * Spak is Copyright (C)1996 Karyl F. Stein <xenon@xenos.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "makearp_func.h"
#include "makearp_conf.h"
#include <stdio.h>

#ifdef SunOS
# include <sys/types.h>   /* u_short         */
# include <sys/socket.h>  /* struct sockaddr */
#endif  /* SunOS */

#include <net/if.h>      /* ifreq structure       */
#include <net/if_arp.h>  /* ARP header structure  */
#include "datadisp.h"    /* Function prototypes   */

/* Values used by the flag variable.  Do not change. */
#define GOT_SMAC 0x01
#define GOT_DMAC 0x02
#define VERBOSE  0x04

/* Set defaults if not defined previously or not valid.  Change these */
/* values in makearp_conf.h instead of here.                          */
#ifndef DEF_HW_TYPE
#define DEF_HW_TYPE 1
#endif  /* DEF_HW_TYPE */
#ifndef DEF_PROTOCOL_TYPE
#define DEF_PROTOCOL_TYPE 0x0800
#endif  /* DEF_PROTOCOL_TYPE */
#ifndef DEF_OPERATION
#define DEF_OPERATION 1
#endif  /* DEF_OPERATION */
#ifndef DEF_DST_MAC
#define DEF_DST_MAC "00:00:00:00:00:00"
#endif  /* DEF_DST_MAC */


/* Function: usage
 * Input   : Command name.
 * Output  : Print a usage message and exit
 */
void usage (char *name) {
  fprintf(stderr, "usage: %s\n", name);
  fprintf(stderr, "       [-ht <hw_type>] [-pt <prot_type] [-hl <len>] [-pl <len>]\n");
  fprintf(stderr, "       [-op <operation>] [-o <output_file>] [-v]\n");
  fprintf(stderr, " -hl  Set the length of hardware address.  Default: Correct value\n");
  fprintf(stderr, " -ht  Set the hardware type to <hardware_type>.  Default: %d\n", DEF_HW_TYPE);
  fprintf(stderr, " -pl  Set the length of protocol address.  Default: Correct value\n");
  fprintf(stderr, " -pt  Set the protocol type to <protocol_type>.  Default: 0x%x\n", DEF_PROTOCOL_TYPE);
  fprintf(stderr, " -o   Print the packet data to <output_file>.  Default: stdout\n");
  fprintf(stderr, " -op  Set the operation to <operation>.  Default: %d\n", DEF_OPERATION);
  fprintf(stderr, " -v   Turn on verbose mode, (print packet data).\n");
  fprintf(stderr, "\nFor a harware type of 1 (Ethernet), the following flags are valid:\n");
  fprintf(stderr, "       [-sm <src_mac>] [-dm <dst_mac>]\n");
  fprintf(stderr, " -dm  Set the destination MAC address of the frame to <dest_mac>.  The\n");
  fprintf(stderr, "      <dest_mac> argument takes the form a:b:c:d:e:f where a through f are\n");
  fprintf(stderr, "       two digit hexadecimal numbers.  Default: %s\n", DEF_SRC_MAC);
  fprintf(stderr, " -sm  Set the source MAC address of the frame to <src_mac>.  The <src_mac>\n");
  fprintf(stderr, "      argument takes the form a:b:c:d:e:f where a through f are two digit\n");
  fprintf(stderr, "      hexadecimal numbers.  Default: ");
#ifdef DEF_SRC_MAC
  fprintf(stderr, "%s\n", DEF_SRC_MAC);
#else
  fprintf(stderr, "Guessed by the program.\n");
#endif  /* DEF_SRC_MAC */
  fprintf(stderr, "\nFor a protocol type of 0x0800, (IP), the following flags are valid:\n");
  fprintf(stderr, "       [-si <source>] [-di <destination>]\n");
  fprintf(stderr, " -di  Set the destination IP address to <destination>.  The <destination>\n");
  fprintf(stderr, "      argument should be a valid IP number or host name.\n");
  fprintf(stderr, " -si  Set the source IP address to <source>.  The <source> argument should\n");
  fprintf(stderr, "      be a valid IP number or host name.\n");
  exit(0);
}


int main (int argc, char *argv[]) {
  char *output_file = NULL, *ptr;
  int datagram_len = sizeof(struct arphdr), flag = 0, tmpint = 0;
  struct arphdr *packet;
  struct in_addr dest_ip, src_ip;
  unsigned char dest_mac[6], src_mac[6];
  void *datagram;

  /* Set defaults */
  dest_ip.s_addr = src_ip.s_addr = 0;

  /* Create a new packet */
  datagram = (void *) new_arp_packet();
  packet = (struct arphdr *) datagram;

  /* Parse the arguments */
  while (++tmpint < argc) {
    if (*(ptr = argv[tmpint]) != '-')
      usage(argv[0]);
    ptr++;

    /* Destination IP Address (IP Protocol) */
    if (strcmp(ptr, "di") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "%s: (warning) The -di argument expects an IP number or host name\n", argv[0]);
      else {
#ifdef INET_ATON
        if (inet_aton(argv[tmpint], &dest_ip) == 0)
#else
	if ((dest_ip.s_addr = inet_addr(argv[tmpint])) == -1)
#endif  /* INET_ATON */
          dest_ip.s_addr = getaddrbyname(argv[tmpint]);
      }
    }

    /* Destination MAC Address (Ethernet 10Mbps) */
    else if (strcmp(ptr, "dm") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "%s: (warning) The -dm argument expects a hardware address\n", argv[0]);
      else {
	if (hwaddread(dest_mac, argv[tmpint]) == -1) {
	  fprintf(stderr,"%s: Invalid destination MAC address\n", argv[0]);
	  exit(1);
	}
	flag |= GOT_DMAC;
      }
    }

    /* Print help screen */
    else if (strcmp(ptr, "h") == 0)
      usage(argv[0]);

    /* Hardware Address Length */
    else if (strcmp(ptr, "hl") == 0)
      packet->ar_hln = my_atoi(argv[++tmpint], 8, "-hl");

    /* Hardware Type */
    else if (strcmp(ptr, "ht") == 0)
      packet->ar_hrd = my_atoi(argv[++tmpint], 16, "-ht");

    /* Output File */
    else if (strcmp(ptr, "o") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "%s: (warning) The -o argument expects a file name\n",
		argv[0]);
      else {
        if (output_file != NULL)
          free(output_file);
        output_file = argv[++tmpint];
      }
    }

    /* Operation */
    else if (strcmp(ptr, "op") == 0)
      packet->ar_op = my_atoi(argv[++tmpint], 16, "-op");

    /* Protocol Address Length */
    else if (strcmp(ptr, "pl") == 0)
      packet->ar_pln = my_atoi(argv[++tmpint], 8, "-pl");

    /* Protocol Type */
    else if (strcmp(ptr, "pt") == 0)
      packet->ar_pro = my_atoi(argv[++tmpint], 16, "-pt");

    /* Source Address (IP Protocol) */
    if (strcmp(ptr, "si") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "%s: (warning) The -si argument expects an IP number or host name\n", argv[0]);
      else {
#ifdef INET_ATON
        if (inet_aton(argv[tmpint], &src_ip) == 0)
#else
	if ((src_ip.s_addr = inet_addr(argv[tmpint])) == -1)
#endif  /* INET_ATON */
          src_ip.s_addr = getaddrbyname(argv[tmpint]);
      }
    }

    /* Source MAC Address (Ethernet 10Mbps) */
    else if (strcmp(ptr, "sm") == 0) {
      if (argv[++tmpint] == NULL)
        fprintf(stderr, "%s: (warning) The -sm argument expects a hardware address\n", argv[0]);
      else {
	if (hwaddread(src_mac, argv[tmpint]) < 0) {
	  fprintf(stderr,"%s: Invalid source MAC address\n", argv[0]);
	  exit(1);
	}
	flag |= GOT_SMAC;
      }
    }

    /* Set verbose mode */
    else if (strcmp(ptr, "v") == 0)
      flag |= VERBOSE;
  }

  /* Set the destination MAC if needed and not already defined.  We do this */
  /* now instead of first so that we do not waste the time to do this if    */
  /* not required.                                                          */
  /*                                                                        */
  if (!flag & GOT_DMAC)
    dest_mac[0] = dest_mac[1] = dest_mac[2] = dest_mac[3] = dest_mac[4] = dest_mac[5] = 0;

  /*
#ifdef DEF_DST_MAC
    if (hwaddread(dest_mac, DEF_DST_MAC) == -1)
      fprintf(stderr, "%s: (warning) DEF_DST_MAC invalid.  Using 00:00:00:00:00:00\n", argv[0]);
    else
#endif  /* DEF_DST_MAC */
  /*
    if (hwaddread(dest_mac, "00:00:00:00:00:00") == -1) {
      fprintf(stderr, "%s: Fatal error in hwaddread().\n", argv[0]);
      exit(1);
    }
  }
  */

  /* Set the source MAC if needed and not already defined.  We do this now  */
  /* instead of first so that we do not waste the time to do this if not    */
  /* required.                                                              */
  if (!flag & GOT_SMAC) {
#ifdef DEF_SRC_MAC
    if (hwaddread(src_mac, DEF_SRC_MAC) == -1)
      fprintf(stderr, "%s: (warning) DEF_SRC_MAC invalid.  Using 00:00:00:00:00:00\n", argv[0]);
    else if (hwaddread(src_mac, "00:00:00:00:00:00") == -1) {
      fprintf(stderr, "%s: Fatal error in hwaddread().\n", argv[0]);
      exit(1);
    }
#else
    if (get_default(&ifr, MAC) == -1) {
      fprintf(stderr, "%s: Unable to load MAC address.  Using 00:00:00:00:00:00\n", argv[0]);
      if (hwaddread(src_mac, "00:00:00:00:00:00") == -1) {
	fprintf(stderr, "%s: Fatal error in hwaddread().\n", argv[0]);
	exit(1);
      }
    } else if ((memcpy(src_mac, ifr.ifr_hwaddr.sa_data, 6)) == NULL) {
      fprintf(stderr, "%s: Fatal error in memcpy()\n");
      exit(1);
    }
#endif  /* DEF_SRC_MAC */
  }

  /* Create the rest of the packet */
  if ((datagram = (void *) realloc(datagram, sizeof(struct arphdr) + 
				   get_resize_len(datagram))) == NULL) {
    fprintf(stderr, "Out Of Memory\n");
    exit(1);
  }
  packet = (struct arphdr *) datagram;
  ptr = ((char *) datagram) + 8;
  switch (packet->ar_hrd) {
  case 1:  /* Ethernet 10Mbps */
    if (memcpy(ptr, src_mac, 6) == NULL) {
      fprintf(stderr, "Fatal error in memcpy()\n");
      exit(1);
    }
    (char *) ptr += 6;
    datagram_len += 6;
    packet->ar_hln = 6;
  }
  switch (packet->ar_pro) {
  case 0x800:  /* IP */
    if (memcpy(ptr, &src_ip.s_addr, 4) == NULL) {
      fprintf(stderr, "Fatal error in memcpy()\n");
      exit(1);
    }
    (char *) ptr += 4;
    datagram_len += 4;
    packet->ar_pln = 4;
  }

  switch (packet->ar_hrd) {
  case 1:  /* Ethernet 10Mbps */
    if (memcpy(ptr, dest_mac, 6) == NULL) {
      fprintf(stderr, "Fatal error in memcpy()\n");
      exit(1);
    }
    (char *) ptr += 6;
    datagram_len += 6;
  }
  switch (packet->ar_pro) {
  case 0x800:  /* IP */
    if (memcpy(ptr, &dest_ip.s_addr, 4) == NULL) {
      fprintf(stderr, "Fatal error in memcpy()\n");
      exit(1);
    }
    datagram_len += 4;
  }

  packet->ar_pro = htons(packet->ar_pro);
  packet->ar_hrd = htons(packet->ar_hrd);
  packet->ar_op = htons(packet->ar_op);

  if (flag & VERBOSE)
    print_arp_data(packet);

  /* Write the frame data to a file or stdout */
  write_packet(output_file, datagram, datagram_len);
}
