/*
 *  linux/fs/tcfs/file.c
 *
 *  Copyright (C) 1992  Rick Sladkey
 *
 *  Changes Copyright (C) 1994 by Florian La Roche
 *   - Do not copy data too often around in the kernel.
 *   - In tcfs_file_read the return value of kmalloc wasn't checked.
 *   - Put in a better version of read look-ahead buffering. Original idea
 *     and implementation by Wai S Kok elekokws@ee.nus.sg.
 *
 *  Expire cache on write to a file by Wai S Kok (Oct 1994).
 *
 *  Total rewrite of read side for new TCFS buffer cache.. Linus.
 *
 *  tcfs regular file handling functions
 */

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/pagemap.h>
#include <linux/lockd/bind.h>

#include <linux/tcfs_fs.h>
#include <linux/tcfs/tcfs_library.h>

#include <asm/segment.h>
#include <asm/system.h>

extern int tcfs_debug;
#define TCFSDBG_FACILITY		TCFSDBG_FILE

#if 0
#undef dprintk
#undef dfprintk
#define dprintk		printk
#define dfprintk(fac, args...)	printk(## args)
#endif

extern int tcfs_ioctl (struct inode *, struct file *, unsigned int,
			unsigned long);
static int  tcfs_file_mmap(struct file *, struct vm_area_struct *);
static ssize_t tcfs_file_read(struct file *, char *, size_t, loff_t *);
static ssize_t tcfs_file_write(struct file *, const char *, size_t, loff_t *);
static int  tcfs_file_flush(struct file *);
static int  tcfs_fsync(struct file *, struct dentry *dentry);

static struct file_operations tcfs_file_operations = {
	NULL,			/* lseek - default */
	tcfs_file_read,		/* read */
	tcfs_file_write,		/* write */
	NULL,			/* readdir - bad */
	NULL,			/* select - default */
	tcfs_ioctl,		/* ioctl - default */
	tcfs_file_mmap,		/* mmap */
	tcfs_open,		/* open */
	tcfs_file_flush,		/* flush */
	tcfs_release,		/* release */
	tcfs_fsync,		/* fsync */
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL,			/* revalidate */
	tcfs_lock,		/* lock */
};

struct inode_operations tcfs_file_inode_operations = {
	&tcfs_file_operations,	/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	tcfs_readpage,		/* readpage */
	tcfs_writepage,		/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL,			/* smap */
	tcfs_updatepage,		/* updatepage */
	tcfs_revalidate,		/* revalidate */
};

/* Hack for future TCFS swap support */
#ifndef IS_SWAPFILE
# define IS_SWAPFILE(inode)	(0)
#endif

/*
 * Flush all dirty pages, and check for write errors.
 *
 */
static int
tcfs_file_flush(struct file *file)
{
	struct inode	*inode = file->f_dentry->d_inode;
	int		status;

	dfprintk(VFS, "tcfs: flush(%x/%ld)\n", inode->i_dev, inode->i_ino);

	status = tcfs_wb_file(inode, file);
	if (!status) {
		status = file->f_error;
		file->f_error = 0;
	}
	return status;
}

static ssize_t
tcfs_file_read(struct file * file, char * buf, size_t count, loff_t *ppos)
{
	struct dentry * dentry = file->f_dentry;
	ssize_t result;
	int res;
	void *ks=NULL;
	struct tcfs_header tcfsheader;
	struct inode *inode = dentry->d_inode; /* NULL? naaa */

	dfprintk(VFS, "tcfs: read(%s/%s, %lu@%lu)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		(unsigned long) count, (unsigned long) *ppos);

	result = tcfs_revalidate_inode(TCFS_DSERVER(dentry), dentry);

	if (!result) {
		if (inode->u.tcfs_i.fhandle) {
			*inode->u.tcfs_i.fhandle=*(struct tcfs_fh *)dentry->d_fsdata;
		}

		res = tcfs_get_header (inode, &tcfsheader, NULL);

		if (res>=0 && !(ks=tcfs_get_key (inode))) {
			return -EACCES;
		}

		result = generic_file_read(file, buf, count, ppos);
	}

	return result;
}

static int
tcfs_file_mmap(struct file * file, struct vm_area_struct * vma)
{
	struct dentry *dentry = file->f_dentry;
	int	status;

	dfprintk(VFS, "tcfs: mmap(%s/%s)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);

	status = tcfs_revalidate_inode(TCFS_DSERVER(dentry), dentry);
	if (!status)
		status = generic_file_mmap(file, vma);
	return status;
}

/*
 * Flush any dirty pages for this process, and check for write errors.
 * The return status from this call provides a reliable indication of
 * whether any write errors occurred for this process.
 */
static int
tcfs_fsync(struct file *file, struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	int status;

	dfprintk(VFS, "tcfs: fsync(%x/%ld)\n", inode->i_dev, inode->i_ino);

	status = tcfs_wb_file(inode, file);
	if (!status) {
		status = file->f_error;
		file->f_error = 0;
	}
	return status;
}


/* 
 * Write to a file (through the page cache).
 */
static ssize_t
tcfs_file_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	struct dentry * dentry = file->f_dentry;
	struct inode * inode = dentry->d_inode;
	ssize_t result;
	int res=0;
	void *ks=NULL;
	struct tcfs_header tcfsheader;

	dfprintk(VFS, "tcfs: write(%s/%s(%ld), %lu@%lu)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name,
		inode->i_ino, (unsigned long) count, (unsigned long) *ppos);

	result = -EBUSY;
	if (IS_SWAPFILE(inode))
		goto out_swapfile;
	result = tcfs_revalidate_inode(TCFS_DSERVER(dentry), dentry);
	if (result)
		goto out;

	result = count;
	if (!count)
		goto out;

	if (inode->u.tcfs_i.fhandle) {
		*inode->u.tcfs_i.fhandle=*(struct tcfs_fh *)dentry->d_fsdata;
	}

	res = tcfs_get_header (inode, &tcfsheader, NULL);

	if (res>=0 && !(ks=tcfs_get_key (inode))) {
		return -EACCES;
	}

	result = generic_file_write(file, buf, count, ppos);
out:
	return result;

out_swapfile:
	printk(KERN_INFO "TCFS: attempt to write to active swap file!\n");
	goto out;
}

/*
 * Lock a (portion of) a file
 */
int
tcfs_lock(struct file *filp, int cmd, struct file_lock *fl)
{
	struct inode * inode = filp->f_dentry->d_inode;
	int	status = 0;
	void 	*ks;

	dprintk("TCFS: tcfs_lock(f=%4x/%ld, t=%x, fl=%x, r=%ld:%ld)\n",
			inode->i_dev, inode->i_ino,
			fl->fl_type, fl->fl_flags,
			fl->fl_start, fl->fl_end);

	if (!inode)
		return -EINVAL;

	ks=tcfs_get_key (inode);
	if (!ks) {
		return -EACCES;
	}

	/* No mandatory locks over TCFS */
	if ((inode->i_mode & (S_ISGID | S_IXGRP)) == S_ISGID)
		return -ENOLCK;

	/* Fake OK code if mounted without NLM support */
	if (TCFS_SERVER(inode)->flags & TCFS_MOUNT_NONLM) {
		if (cmd == F_GETLK)
			status = LOCK_USE_CLNT;
		goto out_ok;
	}

	/*
	 * No BSD flocks over TCFS allowed.
	 * Note: we could try to fake a POSIX lock request here by
	 * using ((u32) filp | 0x80000000) or some such as the pid.
	 * Not sure whether that would be unique, though, or whether
	 * that would break in other places.
	 */
	if (!fl->fl_owner || (fl->fl_flags & (FL_POSIX|FL_BROKEN)) != FL_POSIX)
		return -ENOLCK;

	/*
	 * Flush all pending writes before doing anything
	 * with locks..
	 */
	status = tcfs_wb_all(inode);
	if (status < 0)
		return status;

	if ((status = nlmclnt_proc(inode, cmd, fl)) < 0)
		return status;
	else
		status = 0;

	/*
	 * Make sure we re-validate anything we've got cached.
	 * This makes locking act as a cache coherency point.
	 */
 out_ok:
	TCFS_CACHEINV(inode);
	return status;
}
