/*
 *  linux/include/linux/tcfs_fs.h
 *
 *  OS-specific tcfs filesystem definitions and declarations
 */

#ifndef _LINUX_TCFS_FS_H
#define _LINUX_TCFS_FS_H

#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/in.h>

#include <linux/sunrpc/sched.h>
#include <linux/tcfs.h>
#include <linux/tcfs_mount.h>
#include <linux/tcfs_ioctl.h>
/*
 * Enable debugging support for tcfs client.
 * Requires RPC_DEBUG.
 */
#if 0
#define RPC_DEBUG

#ifdef RPC_DEBUG
# define TCFS_DEBUG
#endif
#endif

/*
 * TCFS_MAX_DIRCACHE controls the number of simultaneously cached
 * directory chunks. Each chunk holds the list of tcfs_entry's returned
 * in a single readdir call in a memory region of size PAGE_SIZE.
 *
 * Note that at most server->rsize bytes of the cache memory are used.
 */
#define TCFS_MAX_DIRCACHE		16

#define TCFS_MAX_FILE_IO_BUFFER_SIZE	16384
#define TCFS_DEF_FILE_IO_BUFFER_SIZE	4096

/*
 * The upper limit on timeouts for the exponential backoff algorithm.
 */
#define TCFS_MAX_RPC_TIMEOUT		(6*HZ)

/*
 * Size of the lookup cache in units of number of entries cached.
 * It is better not to make this too large although the optimum
 * depends on a usage and environment.
 */
#define TCFS_LOOKUP_CACHE_SIZE		64

/*
 * superblock magic number for TCFS
 */
#define TCFS_SUPER_MAGIC			0x6969

#define TCFS_FH(dentry)			((struct tcfs_fh *)((dentry)->d_fsdata))
#define TCFS_INODE_FH(inode)		(inode->u.tcfs_i.fhandle)
#define TCFS_DSERVER(dentry)		(&(dentry)->d_sb->u.tcfs_sb.s_server)
#define TCFS_SERVER(inode)		(&(inode)->i_sb->u.tcfs_sb.s_server)
#define TCFS_CLIENT(inode)		(TCFS_SERVER(inode)->client)
#define TCFS_ADDR(inode)			(RPC_PEERADDR(TCFS_CLIENT(inode)))
#define TCFS_CONGESTED(inode)		(RPC_CONGESTED(TCFS_CLIENT(inode)))

#define TCFS_READTIME(inode)		((inode)->u.tcfs_i.read_cache_jiffies)
#define TCFS_OLDMTIME(inode)		((inode)->u.tcfs_i.read_cache_mtime)
#define TCFS_CACHEINV(inode) \
do { \
	TCFS_READTIME(inode) = jiffies - 1000000; \
	TCFS_OLDMTIME(inode) = 0; \
} while (0)
#define TCFS_ATTRTIMEO(inode)		((inode)->u.tcfs_i.attrtimeo)
#define TCFS_MINATTRTIMEO(inode) \
	(S_ISDIR(inode->i_mode)? TCFS_SERVER(inode)->acdirmin \
			       : TCFS_SERVER(inode)->acregmin)
#define TCFS_MAXATTRTIMEO(inode) \
	(S_ISDIR(inode->i_mode)? TCFS_SERVER(inode)->acdirmax \
			       : TCFS_SERVER(inode)->acregmax)

#define TCFS_FLAGS(inode)		((inode)->u.tcfs_i.flags)
#define TCFS_REVALIDATING(inode)		(TCFS_FLAGS(inode) & TCFS_INO_REVALIDATE)
#define TCFS_WRITEBACK(inode)		((inode)->u.tcfs_i.writeback)

/*
 * These are the default flags for swap requests
 */
#define TCFS_RPC_SWAPFLAGS		(RPC_TASK_SWAPPER|RPC_TASK_ROOTCREDS)

/* Flags in the RPC client structure */
#define TCFS_CLNTF_BUFSIZE	0x0001	/* readdir buffer in longwords */

struct hash_entry {
	unsigned int uid;
	unsigned char deskey[KEYSIZE];
	void *ks;
	unsigned int count;
	unsigned char permanent;
	struct hash_entry *next;
};

struct gid_hash_entry {
	unsigned int gid;
	unsigned char gidkey[KEYSIZE+KEYSIZE/8];
	void *ks;
	unsigned int ins;
	struct hash_entry *key;
	struct gid_hash_entry *next;
};

#define HASH_SIZE	10

#ifdef __KERNEL__

/*
 * This struct describes a file region to be written.
 * It's kind of a pity we have to keep all these lists ourselves, rather
 * than sticking an extra pointer into struct page.
 */
struct tcfs_wreq {
	struct rpc_listitem	wb_list;	/* linked list of req's */
	struct rpc_task		wb_task;	/* RPC task */
	struct file *		wb_file;	/* dentry referenced */
	struct page *		wb_page;	/* page to be written */
	struct wait_queue *	wb_wait;	/* wait for completion */
	unsigned int		wb_offset;	/* offset within page */
	unsigned int		wb_bytes;	/* dirty range */
	unsigned int		wb_count;	/* user count */
	int			wb_status;
	pid_t			wb_pid;		/* owner process */
	unsigned short		wb_flags;	/* status flags */

	struct tcfs_writeargs	wb_args;	/* TCFS RPC stuff */
	struct tcfs_fattr	wb_fattr;	/* file attributes */
};

#define WB_NEXT(req)		((struct tcfs_wreq *) ((req)->wb_list.next))

/*
 * Various flags for wb_flags
 */
#define TCFS_WRITE_CANCELLED	0x0004	/* has been cancelled */
#define TCFS_WRITE_UNCOMMITTED	0x0008	/* written but uncommitted (TCFSv3) */
#define TCFS_WRITE_INVALIDATE	0x0010	/* invalidate after write */
#define TCFS_WRITE_INPROGRESS	0x0100	/* RPC call in progress */
#define TCFS_WRITE_COMPLETE	0x0200	/* RPC call completed */

#define WB_CANCELLED(req)	((req)->wb_flags & TCFS_WRITE_CANCELLED)
#define WB_UNCOMMITTED(req)	((req)->wb_flags & TCFS_WRITE_UNCOMMITTED)
#define WB_INVALIDATE(req)	((req)->wb_flags & TCFS_WRITE_INVALIDATE)
#define WB_INPROGRESS(req)	((req)->wb_flags & TCFS_WRITE_INPROGRESS)
#define WB_COMPLETE(req)	((req)->wb_flags & TCFS_WRITE_COMPLETE)

/*
 * linux/fs/tcfs/proc.c
 */
extern int tcfs_proc_getattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fattr *fattr);
extern int tcfs_proc_setattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_sattr *sattr, struct tcfs_fattr *fattr);
extern int tcfs_proc_lookup(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, struct tcfs_fh *fhandle,
			struct tcfs_fattr *fattr);
extern int tcfs_proc_readlink(struct tcfs_server *server, struct tcfs_fh *fhandle,
			void **p0, char **string, unsigned int *len,
			unsigned int maxlen);
extern int tcfs_proc_read(struct tcfs_server *server, struct tcfs_fh *fhandle,
			int swap, unsigned long offset, unsigned int count,
			void *buffer, struct tcfs_fattr *fattr);
extern int tcfs_proc_write(struct tcfs_server *server, struct tcfs_fh *fhandle,
			int swap, unsigned long offset, unsigned int count,
			const void *buffer, struct tcfs_fattr *fattr);
extern int tcfs_proc_create(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, struct tcfs_sattr *sattr,
			struct tcfs_fh *fhandle, struct tcfs_fattr *fattr);
extern int tcfs_proc_remove(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name);
extern int tcfs_proc_rename(struct tcfs_server *server,
			struct tcfs_fh *old_dir, const char *old_name,
			struct tcfs_fh *new_dir, const char *new_name);
extern int tcfs_proc_link(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fh *dir, const char *name);
extern int tcfs_proc_symlink(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, const char *path,
			struct tcfs_sattr *sattr);
extern int tcfs_proc_mkdir(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, struct tcfs_sattr *sattr,
			struct tcfs_fh *fhandle, struct tcfs_fattr *fattr);
extern int tcfs_proc_rmdir(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name);
extern int tcfs_proc_readdir(struct tcfs_server *server, struct tcfs_fh *fhandle,
			u32 cookie, unsigned int size, __u32 *entry);
extern int tcfs_proc_statfs(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fsinfo *res);


/*
 * linux/fs/tcfs/inode.c
 */
extern struct super_block *tcfs_read_super(struct super_block *, void *, int);
extern int init_tcfs_fs(void);
extern struct inode *tcfs_fhget(struct dentry *, struct tcfs_fh *,
				struct tcfs_fattr *);
extern int tcfs_refresh_inode(struct inode *, struct tcfs_fattr *);
extern int tcfs_revalidate(struct dentry *);
extern int tcfs_open(struct inode *, struct file *);
extern int tcfs_release(struct inode *, struct file *);
extern int _tcfs_revalidate_inode(struct tcfs_server *, struct dentry *);

/*
 * linux/fs/tcfs/file.c
 */
extern struct inode_operations tcfs_file_inode_operations;

/*
 * linux/fs/tcfs/dir.c
 */
extern struct inode_operations tcfs_dir_inode_operations;
extern struct dentry_operations tcfs_dentry_operations;
extern void tcfs_free_dircache(void);
extern void tcfs_invalidate_dircache(struct inode *);
extern void tcfs_invalidate_dircache_sb(struct super_block *);

/*
 * linux/fs/tcfs/symlink.c
 */
extern struct inode_operations tcfs_symlink_inode_operations;

/*
 * linux/fs/tcfs/locks.c
 */
extern int tcfs_lock(struct file *, int, struct file_lock *);

/*
 * linux/fs/tcfs/write.c
 */
extern int  tcfs_writepage(struct file *, struct page *);
extern int  tcfs_check_failed_request(struct inode *);

/*
 * Try to write back everything synchronously (but check the
 * return value!)
 */
extern int  tcfs_wb_all(struct inode *);
extern int  tcfs_wb_page(struct inode *, struct page *);
extern int  tcfs_wb_file(struct inode *, struct file *);

/*
 * Invalidate write-backs, possibly trying to write them
 * back first..
 */
extern void tcfs_inval(struct inode *);
extern int  tcfs_updatepage(struct file *, struct page *, unsigned long, unsigned int, int);

/*
 * linux/fs/tcfs/read.c
 */
extern int  tcfs_readpage(struct file *, struct page *);

/*
 * linux/fs/mount_clnt.c
 * (Used only by tcfsroot module)
 */
extern int  tcfs_mount(struct sockaddr_in *, char *, struct tcfs_fh *);

/*
 * inline functions
 */
static inline int
tcfs_revalidate_inode(struct tcfs_server *server, struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	if (jiffies - TCFS_READTIME(inode) < TCFS_ATTRTIMEO(inode))
		return 0;
	return _tcfs_revalidate_inode(server, dentry);
}

/* TCFS root */

extern int tcfs_root_mount(struct super_block *sb);

#endif /* __KERNEL__ */

/*
 * TCFS debug flags
 */
#define TCFSDBG_VFS		0x0001
#define TCFSDBG_DIRCACHE		0x0002
#define TCFSDBG_LOOKUPCACHE	0x0004
#define TCFSDBG_PAGECACHE	0x0008
#define TCFSDBG_PROC		0x0010
#define TCFSDBG_XDR		0x0020
#define TCFSDBG_FILE		0x0040
#define TCFSDBG_ROOT		0x0080
#define TCFSDBG_ALL		0xFFFF

#ifdef __KERNEL__
# undef ifdebug
# ifdef TCFS_DEBUG
#  define ifdebug(fac)		if (tcfs_debug & TCFSDBG_##fac)
# else
#  define ifdebug(fac)		if (0)
# endif
#endif /* __KERNEL */

#endif
