/* TCFS library of functions */


/* Includes */
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/time.h>
#include <linux/random.h>

/* TCFS includes */
#include <linux/tcfs_fs.h>
#include <linux/tcfs.h>
#include <linux/tcfs/cipherinterface.h>
#include <linux/tcfs/hash.h>

/* MD5 includes */
#include <linux/tcfs/global.h>
#include <linux/tcfs/md5.h>

//#define TCFS_DEBUG	(TCFS_DBG_ALL)
#define TCFS_DEBUG	(TCFS_DBG_NONE)

extern u_int tcfs_numreadflags, tcfs_numreadget, tcfs_numofgetheader;
extern u_int tcfs_numofputheader;
extern u_int tcfs_numputwrite, tcfs_numputflags;

static
void tcfs_printk (unsigned int facility, char *msg)
{
	switch (facility&TCFS_DEBUG) {
		case TCFS_DBG_NONE:
			return ;
		case TCFS_DBG_FUNC_IN:
			printk ("TCFS: in -");
			break;
		case TCFS_DBG_FUNC_OUT:
			printk ("TCFS: out -");
			break;
		case TCFS_DBG_GEN:
			printk ("TCFS:");
			break;
		case TCFS_DBG_ERR:
			printk ("TCFS: error ->");
			break;
	}
	printk (" %s\n", msg);
}

extern
int tcfs_instantiate(struct dentry *, struct tcfs_fh *, struct tcfs_fattr *);

static unsigned int
tcfs_ldiv (long long opa, long long opb)
{
	unsigned int res=0;

	while (( opa -= opb) >= 0) {
		res++;
	}

	return res;
}

static int
tcfs_hash_block (const char *buffer, int size, char *out)
{
	MD5_CTX context;
	unsigned char digest[TCFS_HASH_SIZE];

	tcfs_printk (TCFS_DBG_FUNC_IN, "hash_block");

	memset (digest, 0, TCFS_HASH_SIZE);

	MD5Init (&context);
	MD5Update (&context, buffer, size);
	MD5Final (digest, &context);

        memcpy (out, digest, TCFS_HASH_SIZE);
                
	tcfs_printk (TCFS_DBG_FUNC_OUT, "hash_block");
        return -TCFS_ERROK;
}               
        
static void 
tcfs_get_fk (void *fk)
{
	get_random_bytes (fk, KEYSIZE);
}

int
tcfs_check_hash (const char *buffer, const char *bk, char *hash, int size)
{ 
	char *newhash=NULL;
	char buf[size+KEYSIZE];

	tcfs_printk (TCFS_DBG_FUNC_IN, "check_hash");

	newhash=kmalloc (TCFS_HASH_SIZE, GFP_KERNEL);
	if (!newhash)
		return -ENOMEM;
	
	memcpy (buf, buffer, size);
	memcpy (buf+size, bk, KEYSIZE);

	tcfs_hash_block (buf, size+KEYSIZE, newhash);

	if (!memcmp (hash, newhash, TCFS_HASH_SIZE)) {
		kfree (newhash);

		tcfs_printk (TCFS_DBG_FUNC_OUT, "check_hash");
		return -TCFS_ERROK;
	} else {
		kfree (newhash);

		tcfs_printk (TCFS_DBG_FUNC_OUT, "check_hash");
		return -TCFS_ERRGEN;
	}
}               

int
tcfs_get_header (struct inode *inode, struct tcfs_header *tcfsheader, void *ks)
{
        int res;
        char *buffer=NULL;
        struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;

	tcfs_printk (TCFS_DBG_FUNC_IN, "get_header");
	tcfs_numofgetheader++;

	if (!inode->u.tcfs_i.fhandle) {
		tcfs_printk (TCFS_DBG_GEN, "no file handler");
		tcfs_printk (TCFS_DBG_FUNC_OUT, "get_header");
		return -2;
	}

	if (S_ISDIR(inode->i_mode)) {
		int error=-TCFS_ERROK;

		tcfs_printk (TCFS_DBG_GEN, "looking up for '.tcfs'");

		error=tcfs_proc_lookup (TCFS_SERVER(inode),
			TCFS_INODE_FH(inode), ".tcfs", &fhandle, &fattr);
		if (error) {
			tcfs_printk (TCFS_DBG_GEN, "the dir is clean");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "get_header");

			return -2;
		}
	} else {
		fhandle = *(struct tcfs_fh *)TCFS_INODE_FH(inode);
	}
	
	if (!ks && inode->u.tcfs_i.cached!=2) {
		tcfs_printk (TCFS_DBG_GEN, "reading only the flags");
		tcfs_numreadget++;
		tcfs_numreadflags++;
		res=tcfs_proc_read (TCFS_SERVER(inode), &fhandle, 
			0, 0,
			4+sizeof(tcfsheader->flags), tcfsheader, &fattr);

		if (res<0) {
			tcfs_printk (TCFS_DBG_ERR, "error reading flags");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return res;
		}
	
        	if (strncmp (tcfsheader->start, "TCFS", 4)) {
			tcfs_printk (TCFS_DBG_ERR, "this is not a TCFS file");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return -2;
		}
		
		inode->u.tcfs_i.tcfsheader.flags = tcfsheader->flags;
		inode->u.tcfs_i.cached=2;

		tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
		return res;
	} else if (!ks) {
		tcfsheader->flags = inode->u.tcfs_i.tcfsheader.flags;
		strcpy (tcfsheader->start, "TCFS");

		return -TCFS_ERROK;
	}
		
	if (inode->u.tcfs_i.cached!=1) { 
		int error=-TCFS_ERROK;

		buffer=kmalloc (TCFS_HEADER_SIZE+TCFS_HASH_SIZE, GFP_KERNEL);
		if (!buffer) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			
			return -TCFS_ERRNOMEM;
		}

		memset (buffer, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE);

		tcfs_printk (TCFS_DBG_GEN, "reading file header");

		tcfs_numreadget++;
		res=tcfs_proc_read (TCFS_SERVER(inode), &fhandle, 0, 0,
                	TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);

		if (res<0) {
			tcfs_printk (TCFS_DBG_ERR, "error reading header");
			kfree (buffer);

			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return res;
		}

		if (res==0) {
			tcfs_printk (TCFS_DBG_ERR, "header not present");
			kfree (buffer);

			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return -TCFS_ERRGEN;
		}

		if (res!=TCFS_HEADER_SIZE+TCFS_HASH_SIZE) {
			tcfs_printk (TCFS_DBG_ERR, "header not present");
			kfree (buffer);

			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return -TCFS_ERRGEN;
		}
	
		tcfs_printk (TCFS_DBG_GEN, "checking header");
/* 
        	tcfs_decrypt (buffer+4+sizeof(tcfsheader->flags),
			TCFS_HEADER_SIZE+TCFS_HASH_SIZE-(4+sizeof(tcfsheader->flags)),
			ks, "default");
*/
		error=tcfs_check_hash (buffer, ks, buffer+TCFS_HEADER_SIZE,
			 TCFS_HEADER_SIZE);
		if (error<0) {
			printk ("TCFS Warning: header was modified on the server\n");
			kfree (buffer);

			tcfs_printk(TCFS_DBG_FUNC_OUT, "getheader");
			return -EIO;
		}

        	if (strncmp (((struct tcfs_header *)buffer)->start,"TCFS",4)) {
			tcfs_printk (TCFS_DBG_ERR, "this is not a TCFS file");
			kfree (buffer);

			tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
			return -TCFS_ERRGEN;
		}

		tcfs_printk (TCFS_DBG_GEN, "putting header info into inode");
		memcpy (tcfsheader, buffer, sizeof(struct tcfs_header));
		strcpy (inode->u.tcfs_i.tcfsheader.engine, tcfsheader->engine);

		tcfs_decrypt (tcfsheader->file_key, KEYSIZE, ks,
			tcfsheader->engine);

		memcpy (inode->u.tcfs_i.tcfsheader.file_key,
			tcfsheader->file_key, KEYSIZE);

		inode->u.tcfs_i.tcfsheader.size = tcfsheader->size;
		inode->u.tcfs_i.tcfsheader.flags = tcfsheader->flags;
		inode->u.tcfs_i.cached=1;

		kfree (buffer);
	} else {
		tcfs_printk (TCFS_DBG_GEN, "infos yet loaded");
		strcpy (tcfsheader->start, "TCFS");
		strcpy (tcfsheader->engine, inode->u.tcfs_i.tcfsheader.engine);
		memcpy (tcfsheader->file_key,
			inode->u.tcfs_i.tcfsheader.file_key, KEYSIZE);
		tcfsheader->size = inode->u.tcfs_i.tcfsheader.size;
		tcfsheader->flags = inode->u.tcfs_i.tcfsheader.flags;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "getheader");
	return -TCFS_ERROK;
}

static int
tcfs_get_flag (struct inode *inode)
{
	struct tcfs_header tcfsheader;

	tcfs_printk (TCFS_DBG_FUNC_IN, "get_flag");

	if (!inode->u.tcfs_i.cached) {
		tcfs_get_header (inode, &tcfsheader, NULL);
	}

	/* Remember: a file cannot be both SECURE and SHARED */
	if (TCFS_IS_SECURE(inode)) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "get_flag (secure)");
		return TCFS_SECURE_FL;
	}

	if (TCFS_IS_SHARED(inode)) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "get_flag (shared)");
		return TCFS_SHARED_FL;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "get_flag (clean)");

	return TCFS_CLEAR_FILE;
}

void *
tcfs_get_key (struct inode *inode)
{
	struct hash_entry *htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;

	tcfs_printk (TCFS_DBG_FUNC_IN, "get_key");

	switch (tcfs_get_flag (inode)) {
		case TCFS_SECURE_FL:
			htmp=hash_lookup (current->uid);
			if (!htmp) {
				tcfs_printk (TCFS_DBG_FUNC_OUT, "get_key");
				return NULL;
			}

			tcfs_printk (TCFS_DBG_FUNC_OUT, "get_key");
			if (inode->i_uid==current->uid)
				return htmp->ks;
	
			return NULL;

		case TCFS_SHARED_FL:
			ghtmp=gid_hash_lookup (inode->i_gid);
			if (!ghtmp || !ghtmp->ks) {
				tcfs_printk (TCFS_DBG_FUNC_OUT, "get_key");
				return NULL;
			}

			tcfs_printk (TCFS_DBG_FUNC_OUT, "get_key");
			if (inode->i_gid == current->gid)
				return ghtmp->ks;

			return NULL;
	}


	tcfs_printk (TCFS_DBG_FUNC_OUT, "get_key");
	return TCFS_NO_SECURE;
}

int
tcfs_put_header (struct file *filp, struct dentry *dentry, struct tcfs_header *tcfsheader, void *ks)
{
        char *buffer=NULL, *hres=NULL;
	struct tcfs_fh *fhandle=NULL;
	struct inode *inode=NULL, *inode1=NULL;
	struct dentry *dentry1=NULL;
	struct tcfs_fh fhandle1;
        struct tcfs_fattr fattr;
        int error=0;
	char *engine=NULL;

	tcfs_printk (TCFS_DBG_FUNC_IN, "put_header");
	tcfs_numofputheader++;

	engine=tcfsheader->engine;

	if (!dentry && filp) {
		dentry=filp->f_dentry;
		inode=dentry->d_inode;
	} else {
		inode=dentry->d_inode;
	}

	fhandle=(struct tcfs_fh *)dentry->d_fsdata;

	if (S_ISDIR(inode->i_mode)) {
		struct tcfs_sattr sattr;
		struct qstr qname;
		const char *name=".tcfs";

		tcfs_printk (TCFS_DBG_GEN, "is a dir");

		sattr.mode = S_IRUSR|S_IWUSR;
		sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
		sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;

		tcfs_invalidate_dircache(dentry->d_parent->d_inode);

		qname.name=name;
		qname.len=5;
		
		dentry1=d_alloc (dentry, &qname);
		if (!dentry1) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
			return -TCFS_ERRNOMEM;
		}
		
		dentry1->d_fsdata=kmalloc(sizeof(struct tcfs_fh), GFP_KERNEL);
		if (!dentry1->d_fsdata) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
			return -TCFS_ERRNOMEM;
		}

		error=tcfs_proc_create (TCFS_SERVER(inode), fhandle, name,
			&sattr, &fhandle1, &fattr);

		if (!error) {
			tcfs_printk (TCFS_DBG_GEN,"instantiating inode");

			error=tcfs_instantiate(dentry1, &fhandle1, &fattr);
		}

		if (error) {
			tcfs_printk (TCFS_DBG_ERR, "creating .tcfs");
			d_drop (dentry1);
			dput (dentry1);
			tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");

			return error;
		}
		
		inode1=dentry1->d_inode;
	} else {
		inode1=inode;
		fhandle1=*fhandle;
	}

        buffer=kmalloc (TCFS_HEADER_SIZE+TCFS_HASH_SIZE, GFP_KERNEL);
	if (!buffer) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
		if (dentry1) {
			d_drop (dentry1);
			dput (dentry1);
		}

		return -TCFS_ERRNOMEM;
	}

	memset (buffer, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE);

	if (!ks) { /* Save only the flags */
		strcpy (tcfsheader->start, "TCFS");
		tcfs_numputwrite++;
		tcfs_numputflags++;
		error = tcfs_proc_write (TCFS_SERVER(inode),
				&fhandle1, 0, 0,
				4+sizeof (tcfsheader->flags), tcfsheader,
				&fattr);
	
		tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
		if (dentry1) {
			d_drop (dentry1);
			dput (dentry1);
		}

		return -TCFS_ERROK;
	}

	hres=kmalloc (TCFS_HASH_SIZE, GFP_KERNEL);
	if (!hres) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
		kfree (buffer);
		if (dentry1) {
			d_drop (dentry1);
			dput (dentry1);
		}

		return -TCFS_ERRNOMEM;
	}
	
	tcfs_encrypt (tcfsheader->file_key, KEYSIZE, ks, engine);

        memcpy (buffer, tcfsheader, TCFS_HEADER_SIZE);
	memcpy (buffer+TCFS_HEADER_SIZE, ks, KEYSIZE);
	tcfs_hash_block (buffer, TCFS_HEADER_SIZE+KEYSIZE, hres);
	memcpy (buffer+TCFS_HEADER_SIZE, hres, TCFS_HASH_SIZE);
/*
        tcfs_encrypt (buffer+4+sizeof (tcfsheader->flags),
		TCFS_HEADER_SIZE+TCFS_HASH_SIZE-(4+sizeof(tcfsheader->flags)),
		ks, engine);
*/
	tcfs_numputwrite++;
        error=tcfs_proc_write (TCFS_SERVER(inode), &fhandle1, 0, 0,
		TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);

        kfree (buffer);
	kfree (hres);

	if (dentry1) {
		d_drop (dentry1);
		dput (dentry1);
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "put_header");
        return -TCFS_ERROK;
}

static int
tcfs_decrypt_file(struct dentry *dentry, struct tcfs_header *tcfsheader)
{
	int error=-TCFS_ERROK;
	char buffer[TCFS_HBLOCK_SIZE];
	char *engine=NULL;
	off_t size;
	unsigned int towrite, blocks, block_number;
	loff_t offset, newoffset;
	struct tcfs_fattr fattr;
	struct tcfs_sattr sattr;
	struct inode *inode=NULL;
	char bkey[TCFS_HASH_SIZE], buf[KEYSIZE+sizeof(unsigned int)];
	void *fk=tcfsheader->file_key;
	void *bk=NULL;

	tcfs_printk (TCFS_DBG_FUNC_IN, "decrypt_file");

	memset (bkey, 0, TCFS_HASH_SIZE);
	memset (buf, 0, KEYSIZE+sizeof(unsigned int));
	memcpy (buf, fk, KEYSIZE);

	inode=dentry->d_inode;
	engine=tcfsheader->engine;
	size=tcfsheader->size;

	blocks=size/TCFS_BLOCK_SIZE;
	offset=0;
	towrite=TCFS_BLOCK_SIZE;
	newoffset=TCFS_HEADER_SIZE+TCFS_HASH_SIZE;
	blocks++;
	block_number=1;

	do {
		memcpy (buf+KEYSIZE,(char *)&block_number,sizeof(unsigned int));
		tcfs_hash_block (buf, KEYSIZE+sizeof(unsigned int), bkey);
		bk=tcfs_init_key(bkey,
			inode->u.tcfs_i.tcfsheader.engine);

		error=tcfs_proc_read (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, newoffset, TCFS_HBLOCK_SIZE, buffer, &fattr);
		if (error<0) {
			tcfs_printk (TCFS_DBG_ERR, "reading file");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_file");

			return error;
		}

		tcfs_decrypt (buffer, TCFS_HBLOCK_SIZE, bk, engine);
		error=tcfs_check_hash (buffer, bk, buffer+TCFS_BLOCK_SIZE,
			TCFS_BLOCK_SIZE);
		if (error<0) {
			tcfs_printk (TCFS_DBG_ERR, "file is corrupted!");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_file");
			return -TCFS_ERRGEN;
		}

		error=tcfs_proc_write (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, offset, towrite, buffer, &fattr);

		if (error<0) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_file");
			return error;
		}

		block_number++;
		offset+=TCFS_BLOCK_SIZE;
		newoffset+=TCFS_HBLOCK_SIZE;
		towrite=TCFS_BLOCK_SIZE;
	} while (block_number<=blocks);

	tcfs_printk (TCFS_DBG_GEN, "truncating file");

	sattr.mode=inode->i_mode;
	sattr.size=inode->i_size;
	sattr.uid=inode->i_uid;
	sattr.gid=inode->i_gid;

	error=tcfs_proc_setattr(TCFS_DSERVER(dentry), TCFS_FH(dentry),
		&sattr, &fattr);

	if (error) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_file");
		return error;
	}


	tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_file");

	return -TCFS_ERROK;
}

static int
tcfs_encrypt_file(struct inode *inode, struct tcfs_header *tcfsheader)
{
	int error=-TCFS_ERROK;
	char buffer[TCFS_HBLOCK_SIZE], hash[TCFS_HASH_SIZE];
	char *engine=NULL;
	struct tcfs_fattr fattr;
	off_t size;
	unsigned int toread, blocks;
	loff_t offset, newoffset;
	char bkey[TCFS_HASH_SIZE], buf[KEYSIZE+sizeof(unsigned int)];
	void *fk=tcfsheader->file_key;
	void *bk=NULL;

	tcfs_printk (TCFS_DBG_FUNC_IN, "encrypt_file");

	memset (bkey, 0, TCFS_HASH_SIZE);
	memset (buf, 0, KEYSIZE+sizeof(unsigned int));
	memset (buffer, 0, TCFS_HBLOCK_SIZE);
	memcpy (buf, fk, KEYSIZE);

	engine=tcfsheader->engine;
	size=tcfsheader->size;
	blocks=size/TCFS_BLOCK_SIZE;
	offset=blocks*TCFS_BLOCK_SIZE;
	toread=size-offset; /* last block */
	newoffset=blocks*TCFS_HBLOCK_SIZE+TCFS_HEADER_SIZE+TCFS_HASH_SIZE;
	blocks++;

	do {
		memcpy (buf+KEYSIZE, (char *)&blocks, sizeof(unsigned int));
		tcfs_hash_block (buf, KEYSIZE+sizeof(unsigned int), bkey);
		bk=tcfs_init_key (bkey,
			inode->u.tcfs_i.tcfsheader.engine);

		error=tcfs_proc_read (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, offset, toread, buffer, &fattr);
		if (error<0) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_file");

			return error;
		}

		memcpy (buffer+TCFS_BLOCK_SIZE, bk, KEYSIZE);
		tcfs_hash_block (buffer, TCFS_BLOCK_SIZE+KEYSIZE, hash);
		memcpy (buffer+TCFS_BLOCK_SIZE, hash, TCFS_HASH_SIZE);
		tcfs_encrypt (buffer, TCFS_HBLOCK_SIZE, bk, engine);
		error=tcfs_proc_write (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, newoffset, TCFS_HBLOCK_SIZE, buffer, &fattr);
		if (error<0) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_file");

			return error;
		}

		blocks--;
		offset-=TCFS_BLOCK_SIZE;
		newoffset-=TCFS_HBLOCK_SIZE;
		toread=TCFS_BLOCK_SIZE;
	} while (blocks);

	tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_file");
	return -TCFS_ERROK;
}

static int
tcfs_reencrypt_file (struct inode *inode, char *newcipher, struct tcfs_header *tcfsheader)
{
	int error=-TCFS_ERROK;
	char buffer[TCFS_HBLOCK_SIZE], hash[TCFS_HASH_SIZE];
	char *oldcipher=NULL;
	struct tcfs_fattr fattr;
	off_t size;
	unsigned int blocks, offset;
	char buf[KEYSIZE+sizeof (unsigned int)];
	char bkey[TCFS_HASH_SIZE];
	void *fk=tcfsheader->file_key;
	void *bk=NULL;

	tcfs_printk (TCFS_DBG_FUNC_IN, "reencrypt_file");

	memset (bkey, 0, TCFS_HASH_SIZE);
	memset (buf, 0, KEYSIZE+sizeof(unsigned int));
	memcpy (buf, fk, KEYSIZE);

	oldcipher=tcfsheader->engine;
	size=tcfsheader->size;
	blocks=size/TCFS_BLOCK_SIZE;
	offset=TCFS_HEADER_SIZE+TCFS_HASH_SIZE; /* Start of the file */

	blocks++;

	do {
		memcpy (buffer+KEYSIZE, (char *)&blocks, sizeof(unsigned int));
		tcfs_hash_block (buffer, KEYSIZE+sizeof(unsigned int), bkey);
		bk=tcfs_init_key (bkey,
			inode->u.tcfs_i.tcfsheader.engine);

		error=tcfs_proc_read (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, offset, TCFS_HBLOCK_SIZE, buffer, &fattr);
		if (error<0) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_file");
	
			return error;
		}

		tcfs_decrypt (buffer, TCFS_HBLOCK_SIZE, bk, oldcipher);
		error=tcfs_check_hash (buffer, bk, buffer+TCFS_BLOCK_SIZE,
				TCFS_BLOCK_SIZE);
		if (error<0) {
			tcfs_printk (TCFS_DBG_ERR, "file is corrupted!");
			tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_file");
			return -TCFS_ERRGEN;
		}

		memcpy (buffer+TCFS_BLOCK_SIZE, bk, KEYSIZE);
		tcfs_hash_block (buffer, TCFS_BLOCK_SIZE+KEYSIZE, hash);
		memcpy (buffer+TCFS_BLOCK_SIZE, hash, TCFS_HASH_SIZE);
		tcfs_encrypt (buffer, TCFS_HBLOCK_SIZE, bk, newcipher);
		error=tcfs_proc_write (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
			0, offset, TCFS_HBLOCK_SIZE, buffer, &fattr);

		if (error<0) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_file");

			return error;
		}

		offset+=TCFS_HBLOCK_SIZE;
		blocks--;
	} while (blocks);

	tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_file");
	return -TCFS_ERROK;
}

static int
tcfs_decrypt_dir (struct file *filp, struct tcfs_header *tcfsheader, void *ks)
{
	__u32 *entry=NULL;
	char *engine=NULL, *start;
	struct dentry *dentry=filp->f_dentry;
	struct inode *dir=dentry->d_inode;
	unsigned int size, error=0, index=0;
	u32 cookie;

	tcfs_printk (TCFS_DBG_FUNC_IN, "decrypt_dir");

	entry=(__u32 *)get_free_page(GFP_KERNEL);
	if (!entry) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_dir");

		return -TCFS_ERRNOMEM;
	}

	cookie=filp->f_pos;
	engine=tcfsheader->engine;

	error=tcfs_proc_readdir (TCFS_SERVER(dir), TCFS_FH(dentry),
		cookie, PAGE_SIZE, entry);

	if (error<0) {
		free_page ((unsigned long)entry);
		tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_dir");

		return -TCFS_ERRGEN;
	}

	size=error;
	start= (char *)entry;
	entry++;

	while (index<size) {
		__u32	nextpos=*entry++;
		__u32	length=*entry++;
		unsigned int len=length&0x7FFF;
		char	*name=NULL;
		char	tcfsname[TCFS_MAXNAMLEN];
		char	oldname[TCFS_MAXNAMLEN];

		name=start+((length>>16)&0x7FFF);

		memset (tcfsname, 0, TCFS_MAXNAMLEN);
		memset (oldname, 0, TCFS_MAXNAMLEN);

		memcpy (tcfsname, name, len);
		memcpy (oldname, name, len);
		tcfsname[len]=0;
		oldname[len]=0;

		if (strcmp (tcfsname, ".") && strcmp (tcfsname, "..") &&
			strcmp (tcfsname, ".tcfs")) {
			len=tcfsdecode (tcfsname, len+1);
			len=strlen(tcfsname);
			tcfs_decrypt (tcfsname, (len&0xFFF8)+8, ks, engine);
			
			tcfs_invalidate_dircache (dir);

			error=tcfs_proc_rename (TCFS_DSERVER(dentry), 
				TCFS_FH(dentry), oldname,
				TCFS_FH(dentry), tcfsname);
			if (error) {
				tcfs_printk (TCFS_DBG_ERR, "rename error");
				tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_dir");

				return error;
			}
		}
		cookie = nextpos;
		index++;
		entry++;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_dir");
	return -TCFS_ERROK;
}

static int
tcfs_encrypt_dir (struct file *filp, struct tcfs_header *tcfsheader, void *ks)
{
	__u32 *entry=NULL;
	char *engine=NULL, *start=NULL;
	struct dentry *dentry=filp->f_dentry;
	struct inode *dir=dentry->d_inode;
	unsigned int size, error=0, index=0;
	u32 cookie;

	tcfs_printk (TCFS_DBG_FUNC_IN, "encrypt_dir");

	entry=(__u32 *)get_free_page(GFP_KERNEL);
	if (!entry) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_dir");

		return -TCFS_ERRNOMEM;
	}

	cookie=filp->f_pos;
	engine=tcfsheader->engine;

	error=tcfs_proc_readdir (TCFS_SERVER(dir), TCFS_FH(dentry),
		cookie, PAGE_SIZE, entry);

	if (error<0) {
		free_page ((unsigned long)entry);

		tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_dir");
		return -TCFS_ERRGEN;
	}

	size=error;
	start=(char *)entry;
	entry++;

	while (index<size) {
		__u32	nextpos=*entry++;
		__u32	length=*entry++;
		unsigned int len=length&0x7FFF;
		char	*name=NULL;
		char	tcfsname[TCFS_MAXNAMLEN];
		char	oldname[TCFS_MAXNAMLEN];

		name=start+((length>>16)&0x7FFF);

		memset (tcfsname, 0, TCFS_MAXNAMLEN);
		memset (oldname, 0, TCFS_MAXNAMLEN);
		memcpy (tcfsname, name, len);
		memcpy (oldname, name, len);
		tcfsname[len]=0;
		oldname[len]=0;

		if (strcmp (tcfsname, ".") && strcmp (tcfsname, "..")) {
			tcfs_encrypt (tcfsname, ((len+1)&0xFFF8)+8, ks, engine);
			tcfsencode (tcfsname, ((len+1)&0xFFF8)+8);
			tcfs_invalidate_dircache (dir);
			error=tcfs_proc_rename (TCFS_DSERVER(dentry), 
				TCFS_FH(dentry), oldname,
				TCFS_FH(dentry), tcfsname);

			if (error) {
				tcfs_printk (TCFS_DBG_ERR, "rename error");
				tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_dir");

				return error;
			}
		}
		cookie = nextpos;
		index++;
		entry++;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_dir");

	return -TCFS_ERROK;
}

static int
tcfs_reencrypt_dir (struct file *filp, char *newcipher, struct tcfs_header *tcfsheader, void *ks)
{
	__u32 *entry=NULL;
	char *engine=NULL, *start=NULL;
	struct dentry *dentry=filp->f_dentry;
	struct inode *dir=dentry->d_inode;
	unsigned int size, error=0, index=0;
	u32 cookie;

	tcfs_printk (TCFS_DBG_FUNC_IN, "reencrypt_dir");

	entry=(__u32 *)get_free_page(GFP_KERNEL);
	if (!entry) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_dir");

		return -TCFS_ERRNOMEM;
	}

	cookie=filp->f_pos;
	engine=tcfsheader->engine;

	error=tcfs_proc_readdir (TCFS_SERVER(dir), TCFS_FH(dentry),
		cookie, PAGE_SIZE, entry);

	if (error<0) {
		free_page ((unsigned long)entry);
		tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_dir");

		return -TCFS_ERRGEN;
	}

	size=error;
	start=(char *)entry;
	entry++;

	while (index<size) {
		__u32	nextpos=*entry++;
		__u32	length=*entry++;
		unsigned int len=length&0x7FFF;
		char	*name=NULL;
		char	tcfsname[TCFS_MAXNAMLEN];
		char	oldname[TCFS_MAXNAMLEN];

		name=start+((length>>16)&0x7FFF);

		memset (tcfsname, 0, TCFS_MAXNAMLEN);
		memset (oldname, 0, TCFS_MAXNAMLEN);
		memcpy (tcfsname, name, len);
		memcpy (tcfsname, name, len);
		tcfsname[len]=0;
		oldname[len]=0;

		if (strcmp (tcfsname, ".") && strcmp (tcfsname, "..")
			&& strcmp (tcfsname, ".tcfs")) {
			len=tcfsdecode (tcfsname, len+1);
			len=strlen(tcfsname);
			tcfs_decrypt (tcfsname, ((len+1)&0xFFF8)+8, ks, engine);
			tcfs_encrypt (tcfsname, ((len+1)&0xFFF8)+8, ks,
					newcipher);
			tcfsencode (tcfsname, ((len+1)&0xFFF8)+8);
			tcfs_proc_rename (TCFS_DSERVER(dentry), 
				TCFS_FH(dentry), oldname,
				TCFS_FH(dentry), tcfsname);
		}
		cookie = nextpos;
		index++;
		entry++;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "reencrypt_dir");

	return -TCFS_ERROK;
}

static int
tcfs_decrypt_it (struct file *filp, void *ks, unsigned int flag)
{
	int error=-TCFS_ERROK;
	struct inode *inode=filp->f_dentry->d_inode;
	struct tcfs_header tcfsheader;

	tcfs_printk (TCFS_DBG_FUNC_IN, "decrypt_it");

	if (flag==TCFS_SECURE_FL) {
		struct hash_entry *htmp=NULL;

		htmp=hash_lookup (current->uid);
		if (!htmp) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_it");

			return -TCFS_ERRACCES;
		}
		ks=htmp->ks;
	}
	if (flag==TCFS_SHARED_FL) {
		struct gid_hash_entry *ghtmp=NULL;

		ghtmp=gid_hash_lookup (filp->f_dentry->d_inode->i_gid);
		if (!ghtmp || !ghtmp->ks) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_it");

			return -TCFS_ERRACCES;
		}
		ks=ghtmp->ks;
	}

	error=tcfs_get_header (inode, &tcfsheader, ks);
	if (error<0) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_it");

		return error;
	}

	if (S_ISDIR(inode->i_mode)) {
		error=tcfs_decrypt_dir (filp, &tcfsheader, ks);
		if (error) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_it");

			return error;
		}

		tcfs_printk (TCFS_DBG_GEN, "trying to remove .tcfs");
		
		error=tcfs_proc_remove (TCFS_SERVER(inode),
			(struct tcfs_fh *)filp->f_dentry->d_fsdata,
			".tcfs");
		tcfs_invalidate_dircache (inode);
	} else {
		error=tcfs_decrypt_file (filp->f_dentry, &tcfsheader);
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "decrypt_it");

	return error;
}

static int
tcfs_encrypt_it (struct file *filp, void *ks, unsigned int flag)
{
	int error=-TCFS_ERROK;
	struct inode *inode=filp->f_dentry->d_inode;
	struct tcfs_header tcfsheader;
	
	tcfs_printk (TCFS_DBG_FUNC_IN, "encrypt_it");

	if (flag==TCFS_SECURE_FL) {
		struct hash_entry *htmp=NULL;

		htmp=hash_lookup (current->uid);
		if (!htmp) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_it");
			
			return -TCFS_ERRACCES;
		}
		ks=htmp->ks;
	}

	if (flag==TCFS_SHARED_FL) {
		struct gid_hash_entry *ghtmp=NULL;

		ghtmp=gid_hash_lookup (filp->f_dentry->d_inode->i_gid);
		if (!ghtmp || !ghtmp->ks) {
			tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_it");

			return -TCFS_ERRACCES;
		}
		ks=ghtmp->ks;
	}

	memset (&tcfsheader, 0, sizeof (tcfsheader));

	tcfsheader.size=inode->i_size;
	strcpy (tcfsheader.engine, "default");
	strcpy (tcfsheader.start, "TCFS");
	tcfsheader.flags=inode->u.tcfs_i.tcfsheader.flags|flag;

	tcfs_get_fk (tcfsheader.file_key);

	if (S_ISDIR(inode->i_mode)) {
		error=tcfs_encrypt_dir (filp, &tcfsheader, ks);
	} else {
		error=tcfs_encrypt_file (inode, &tcfsheader);
	}

	if (error<0) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_it");

		return error;
	}

	error=tcfs_put_header (filp, NULL, &tcfsheader, ks);

	tcfs_printk (TCFS_DBG_FUNC_OUT, "encrypt_it");

	return error;
}

static int
tcfs_flags_changed (struct file *filp, unsigned int newflags, void *ks)
{
	int error=0;
	unsigned int oldflags;
	unsigned int file_is_secure, file_is_shared, want_secure, want_shared;

	tcfs_printk (TCFS_DBG_FUNC_IN, "flags_changed");

	if (!ks) {
		ks=tcfs_get_key (filp->f_dentry->d_inode);
	}

	oldflags=filp->f_dentry->d_inode->u.tcfs_i.tcfsheader.flags;

	file_is_secure=oldflags&TCFS_SECURE_FL;
	file_is_shared=oldflags&TCFS_SHARED_FL;
	want_secure=newflags&TCFS_SECURE_FL;
	want_shared=newflags&TCFS_SHARED_FL;

	if (file_is_secure && file_is_shared) {
		printk ("TCFS: inode %ld has both group and secure flag!\n",
			filp->f_dentry->d_inode->i_ino);

		tcfs_printk (TCFS_DBG_FUNC_OUT, "flags_changed");

		return -TCFS_ERRINVAL;
	}

	if (want_secure && want_shared) {
		printk ("TCFS: cannot set both group and secure flag!\n");

		tcfs_printk (TCFS_DBG_FUNC_OUT, "flags_changed");

		return -TCFS_ERRINVAL;
	}

	if (!file_is_secure && !file_is_shared) {
		tcfs_printk (TCFS_DBG_GEN, "file is clear");

		if (want_secure) {
			tcfs_printk (TCFS_DBG_GEN, "encrypting file");
			error=tcfs_encrypt_it (filp, ks, TCFS_SECURE_FL);
		}
		
		if (want_shared) {
			tcfs_printk (TCFS_DBG_GEN, "encrypting group file");
			error=tcfs_encrypt_it (filp, ks, TCFS_SHARED_FL);
		}
	}

	if (file_is_shared && !want_shared) {
		tcfs_printk (TCFS_DBG_GEN, "decrypting group file");
		error=tcfs_decrypt_it (filp, ks, TCFS_SHARED_FL);
	}

	if (file_is_secure && !want_secure) {
		tcfs_printk (TCFS_DBG_GEN, "decrypting file");
		error=tcfs_decrypt_it (filp, ks, TCFS_SECURE_FL);
	}

	if (error<0) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "flags_changed");
		return error;
	}

	filp->f_dentry->d_inode->u.tcfs_i.tcfsheader.flags=newflags;

	tcfs_printk (TCFS_DBG_FUNC_OUT, "flags_changed");
	return -TCFS_ERROK;
}

static int
tcfs_engine_changed (struct file *filp, char *newcipher, void *ks)
{
	int error=-TCFS_ERROK;
	char *buffer=NULL, *hash=NULL;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	struct tcfs_header *tcfsheader=NULL;
	struct inode *inode=filp->f_dentry->d_inode;

	buffer=kmalloc (TCFS_HEADER_SIZE+TCFS_HASH_SIZE, GFP_KERNEL);
	hash=kmalloc (TCFS_HASH_SIZE, GFP_KERNEL);

	if (!hash && !buffer) {
		return -TCFS_ERRNOMEM;
	}

	if (S_ISDIR(inode->i_mode)) {
		error=tcfs_proc_lookup (TCFS_SERVER(inode),
			TCFS_INODE_FH(inode), ".tcfs", &fhandle, &fattr);
		if (error) {
			fhandle=*(struct tcfs_fh *)TCFS_INODE_FH(inode);
		} else {
			kfree (buffer);
			kfree (hash);

			return error;
		}
	} else {
		fhandle=*(struct tcfs_fh *)TCFS_INODE_FH(inode);
	}

	error=tcfs_proc_read (TCFS_SERVER(inode), &fhandle,
		0, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);

	if (error<0) {
		kfree (buffer);
		kfree (hash);

		return error;
	}

/*
	tcfs_decrypt (buffer, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, ks, "default");
*/
	error=tcfs_check_hash (buffer, ks, buffer+TCFS_HEADER_SIZE,
		TCFS_HEADER_SIZE);
	if (error<0) {
		printk ("TCFS warning: not a TCFS file\n");
		kfree (buffer);
		kfree (hash);

		return -TCFS_ERRACCES;
	}

	tcfsheader=(struct tcfs_header *)buffer;

	if (S_ISDIR(inode->i_mode)) {
		error=tcfs_reencrypt_dir (filp, newcipher, tcfsheader, ks);
	} else {
		error=tcfs_reencrypt_file (inode, newcipher, tcfsheader);
	}

	if (error<0) {
		kfree (buffer);
		kfree (hash);

		return error;
	}

	strcpy (tcfsheader->engine, newcipher);
	strcpy (inode->u.tcfs_i.tcfsheader.engine, newcipher);

	memcpy (buffer+TCFS_HEADER_SIZE, ks, KEYSIZE);
	tcfs_hash_block (buffer, TCFS_HEADER_SIZE+KEYSIZE, hash);
	memcpy (buffer+TCFS_HEADER_SIZE, hash, TCFS_HASH_SIZE);
/*
	tcfs_encrypt (buffer, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, ks, "default");
*/
	error=tcfs_proc_write (TCFS_SERVER(inode), &fhandle,
		0, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);
	kfree (buffer);
	kfree (hash);

	if (error<0) {
		return error;
	}

	return -TCFS_ERROK;
}

static int
tcfs_size_changed (struct inode *inode, off_t newsize, void *ks)
{
	int error=-TCFS_ERROK;
	char *buffer=NULL, *hash=NULL;
	struct tcfs_fattr fattr;
	struct tcfs_header *tcfsheader=NULL;

	buffer=kmalloc(TCFS_HEADER_SIZE+TCFS_HASH_SIZE, GFP_KERNEL);
	hash=kmalloc(TCFS_HASH_SIZE, GFP_KERNEL);
	if (!buffer && !hash) {
		return -TCFS_ERRNOMEM;
	}

	error=tcfs_proc_read (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
		0, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);

	if (error<0) {
		kfree (buffer);
		kfree (hash);

		return error;
	}

/*
	tcfs_decrypt (buffer+4+sizeof(unsigned int),
		TCFS_HEADER_SIZE+TCFS_HASH_SIZE-(4+sizeof(unsigned int)),
		ks, "default");
*/

	error=tcfs_check_hash (buffer, ks, buffer+TCFS_HEADER_SIZE,
		TCFS_HEADER_SIZE);
	if (error<0) {
		printk ("TCFS warning: not a TCFS file\n");
		kfree (buffer);
		kfree (hash);

		return -TCFS_ERRACCES;
	}

	tcfsheader=(struct tcfs_header *)buffer;

	tcfsheader->size=newsize;
	inode->u.tcfs_i.tcfsheader.size=newsize;

	memcpy (buffer+TCFS_HEADER_SIZE, ks, KEYSIZE);
	tcfs_hash_block (buffer, TCFS_HEADER_SIZE+KEYSIZE, hash);
	memcpy (buffer+TCFS_HEADER_SIZE, hash, TCFS_HASH_SIZE);
/*
	tcfs_encrypt (buffer+4+sizeof(tcfsheader->flags),
		TCFS_HEADER_SIZE+TCFS_HASH_SIZE-(4+sizeof(tcfsheader->flags)),
		 ks, "default");
*/

	error=tcfs_proc_write (TCFS_SERVER(inode), TCFS_INODE_FH(inode),
		0, 0, TCFS_HEADER_SIZE+TCFS_HASH_SIZE, buffer, &fattr);

	kfree (buffer);
	kfree (hash);

	if (error<0) {
		return error;
	}

	return -TCFS_ERROK;
}
	

/*
 * This is called from the tcfs_ioctl (to set the flags or the ciphername) and
 * from the write function (for the file size).
 * It will encrypt/decrypt a file or a dir (i.e. the filenames) according to
 * the flags, if called from tcfs_ioctl.
 */
int
tcfs_modify_header (struct file *filp, struct inode *inode, unsigned int what, void *val, void *ks)
{
	unsigned int flags=(unsigned int)*(unsigned int *)val;
	unsigned int error=-TCFS_ERROK;

	tcfs_printk (TCFS_DBG_FUNC_IN, "modify_header");

	if (!inode) {
		inode=filp->f_dentry->d_inode;
	}

	if (!ks) {
		ks=tcfs_get_key (inode);

		if (what==TCFS_FIELD_FLAGS) {
			if (ks==TCFS_NO_SECURE && flags&TCFS_SECURE_FL) {
				struct hash_entry *htmp=NULL;

				htmp=hash_lookup(current->uid);
				if (!htmp) {
					tcfs_printk (TCFS_DBG_FUNC_OUT, "modify_header");
					return -TCFS_ERRACCES;
				}

				ks=htmp->ks;
			}

			if (ks==TCFS_NO_SECURE && flags&TCFS_SHARED_FL) {
				struct gid_hash_entry *ghtmp=NULL;

				ghtmp=gid_hash_lookup(inode->i_gid);
				if (!ghtmp || !ghtmp->ks) {
					tcfs_printk (TCFS_DBG_FUNC_OUT, "modify_header");
					return -TCFS_ERRACCES;
				}

				if (!ghtmp->ks) {
					tcfs_printk (TCFS_DBG_FUNC_OUT, "modify_header");
					return -TCFS_ERRACCES;
				}

				ks=ghtmp->ks;
			}
		} 
	}

	/* Ok, if ks is still NULL the no access is permitted */

	if (!ks) {
		tcfs_printk (TCFS_DBG_FUNC_OUT, "modify_header");

		return -TCFS_ERRACCES;
	}
	
	switch (what) {
		case TCFS_FIELD_FLAGS:
			tcfs_printk (TCFS_DBG_GEN, "changing flags");
			error=tcfs_flags_changed (filp, flags, ks);
			break;

		case TCFS_FIELD_ENGINE:
			tcfs_printk (TCFS_DBG_GEN, "changing engine");
			error=tcfs_engine_changed (filp, (char *)val, ks);
			break;

		case TCFS_FIELD_SIZE:
			tcfs_printk (TCFS_DBG_GEN, "changing size");
			error=tcfs_size_changed (inode, *(off_t *)val, ks);
			break;
	}

	tcfs_printk (TCFS_DBG_FUNC_OUT, "modify_header");

	return error;
}

/* We assume TCFS_HEADER_SIZE <= TCFS_BLOCK_SIZE */
loff_t
tcfs_new_offset(loff_t old_offset, loff_t * blockstart, loff_t * blockoff)
{
        static loff_t blocks;
        static loff_t result;

        blocks = tcfs_ldiv(old_offset+TCFS_HEADER_SIZE,
                         TCFS_BLOCK_SIZE);

        result = TCFS_HEADER_SIZE+old_offset+blocks*TCFS_HASH_SIZE;

        *blockstart=(result>>10)<<10;
        *blockoff=result-*blockstart;

        return result;
}
