/*
 *  linux/fs/tcfs/symlink.c
 *
 *  Copyright (C) 1992  Rick Sladkey
 *
 *  Optimization changes Copyright (C) 1994 Florian La Roche
 *
 *  tcfs symlink handling code
 */

#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/tcfs_fs.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/string.h>

#include <asm/uaccess.h>

extern int tcfs_debug;

static int tcfs_readlink(struct dentry *, char *, int);
static struct dentry *tcfs_follow_link(struct dentry *, struct dentry *, unsigned int);

/*
 * symlinks can't do much...
 */
struct inode_operations tcfs_symlink_inode_operations = {
	NULL,			/* no file-operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	tcfs_readlink,		/* readlink */
	tcfs_follow_link,	/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};

static int tcfs_readlink(struct dentry *dentry, char *buffer, int buflen)
{
	int error;
	unsigned int len;
	char *res;
	void *mem;

	dfprintk(VFS, "tcfs: readlink(%s/%s)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);

	error = tcfs_proc_readlink(TCFS_DSERVER(dentry), TCFS_FH(dentry),
					&mem, &res, &len, TCFS_MAXPATHLEN);
	if (! error) {
		if (len > buflen)
			len = buflen;
		copy_to_user(buffer, res, len);
		error = len;
		kfree(mem);
	}
	return error;
}

static struct dentry *
tcfs_follow_link(struct dentry * dentry, struct dentry *base, unsigned int follow)
{
	int error;
	unsigned int len;
	char *res;
	void *mem;
	char *path;
	struct dentry *result;

	dfprintk(VFS, "tcfs: follow_link(%s/%s)\n",
		dentry->d_parent->d_name.name, dentry->d_name.name);

	error = tcfs_proc_readlink(TCFS_DSERVER(dentry), TCFS_FH(dentry),
				 &mem, &res, &len, TCFS_MAXPATHLEN);
	result = ERR_PTR(error);
	if (error)
		goto out_dput;

	result = ERR_PTR(-ENOMEM);
	path = kmalloc(len + 1, GFP_KERNEL);
	if (!path)
		goto out_mem;
	memcpy(path, res, len);
	path[len] = 0;
	kfree(mem);

	result = lookup_dentry(path, base, follow);
	kfree(path);
out:
	return result;

out_mem:
	kfree(mem);
out_dput:
	dput(base);
	goto out;
}
