/*
 * linux/fs/tcfs/read.c
 *
 * Block I/O for TCFS
 *
 * Partial copy of Linus' read cache modifications to fs/tcfs/file.c
 * modified for async RPC by okir@monad.swb.de
 *
 * We do an ugly hack here in order to return proper error codes to the
 * user program when a read request failed: since generic_file_read
 * only checks the return value of inode->i_op->readpage() which is always 0
 * for async RPC, we set the error bit of the page to 1 when an error occurs,
 * and make tcfs_readpage transmit requests synchronously when encountering this.
 * This is only a small problem, though, since we now retry all operations
 * within the RPC code when root squashing is suspected.
 */

#define TCFS_NEED_XDR_TYPES
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/pagemap.h>
#include <linux/sunrpc/clnt.h>

#include <linux/tcfs_fs.h>
#include <linux/tcfs/cipherinterface.h>
#include <linux/tcfs/tcfs_library.h>

#include <asm/segment.h>
#include <asm/system.h>
#include <asm/pgtable.h>

extern int tcfs_debug;
extern unsigned int tcfs_numreadpage;

#define TCFSDBG_FACILITY		TCFSDBG_PAGECACHE

struct tcfs_rreq {
	struct inode *		ra_inode;	/* inode from which to read */
	struct page *		ra_page;	/* page to be read */
	struct tcfs_readargs	ra_args;	/* XDR argument struct */
	struct tcfs_readres	ra_res;		/* ... and result struct */
	struct tcfs_fattr	ra_fattr;	/* fattr storage */
};

/* Hack for future TCFS swap support */
#ifndef IS_SWAPFILE
# define IS_SWAPFILE(inode)	(0)
#endif


/*
 * Set up the TCFS read request struct
 */
static inline void
tcfs_readreq_setup(struct tcfs_rreq *req, struct tcfs_fh *fh,
		  unsigned long offset, void *buffer, unsigned int rsize)
{
	req->ra_args.fh     = fh;
	req->ra_args.offset = offset;
	req->ra_args.count  = rsize;
	req->ra_args.buffer = buffer;
	req->ra_res.fattr   = &req->ra_fattr;
	req->ra_res.count   = rsize;
}


/*
 * Read a page synchronously.
static int
tcfs_readpage_sync(struct dentry *dentry, struct inode *inode, struct page *page)
{
	struct tcfs_rreq	rqst;
	unsigned long	offset = page->offset;
	char		*buffer = (char *) page_address(page);
	p
	int		rsize = TCFS_SERVER(inode)->rsize;
	int		result, refresh = 0;
	int		count = PAGE_SIZE;
	int		flags = IS_SWAPFILE(inode)? TCFS_RPC_SWAPFLAGS : 0;

	dprintk("TCFS: tcfs_readpage_sync(%p)\n", page);
	clear_bit(PG_error, &page->flags);

	do {
		if (count < rsize)
			rsize = count;

		dprintk("TCFS: tcfs_proc_read(%s, (%s/%s), %ld, %d, %p)\n",
			TCFS_SERVER(inode)->hostname,
			dentry->d_parent->d_name.name, dentry->d_name.name,
			offset, rsize, buffer);

		tcfs_readreq_setup(&rqst, TCFS_FH(dentry), offset, buffer, rsize);
		result = rpc_call(TCFS_CLIENT(inode), TCFSPROC_READ,
					&rqst.ra_args, &rqst.ra_res, flags);

		if (result < 0) {
			if (result == -EISDIR)
				result = -EINVAL;
			goto io_error;
		}
		refresh = 1;
		count  -= result;
		offset += result;
		buffer += result;
		if (result < rsize)
			break;
	} while (count);

	memset(buffer, 0, count);
	set_bit(PG_uptodate, &page->flags);
	result = 0;

io_error:
	if (refresh && result >= 0)
		tcfs_refresh_inode(inode, &rqst.ra_fattr);
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);
	return result;
}
*/

static int
tcfs_readpage_sync(struct dentry *dentry, struct inode *inode, struct page *page)
{
	struct tcfs_rreq	rqst;
	unsigned long	offset = page->offset;
	char		*buffer = (char *) page_address(page);
	int		rsize = TCFS_SERVER(inode)->rsize;
	int		result, refresh = 0;
	int		count = PAGE_SIZE;
	int		flags = IS_SWAPFILE(inode)? TCFS_RPC_SWAPFLAGS : 0;
	char		*newbuf=NULL;
	void		*ks=NULL, *bk=NULL;
	int		howmanycopy = 0, toread=0;
	unsigned long long	blockstart = 0, blockoff = 0, newpos = 0;
	unsigned int	block_number=0;
	char 		buf[KEYSIZE+sizeof(unsigned int)];
	char		bkey[TCFS_HASH_SIZE];

	tcfs_numreadpage++;
	dprintk("TCFS: tcfs_readpage_sync(%p)\n", page);
	clear_bit(PG_error, &page->flags);
	memset (buf, 0, KEYSIZE+sizeof(unsigned int));
	memset (bkey, 0, TCFS_HASH_SIZE);

	if (!(ks=tcfs_get_key (inode))) {
		return -EACCES;
	}

	toread=rsize;
	blockstart=offset;
	blockoff=0;

	if (ks!=TCFS_NO_SECURE) {
		int error=-TCFS_ERROK;
		struct tcfs_header tcfsheader;

		error=tcfs_get_header (inode, &tcfsheader, ks);
		if (error<0)
			return error;

		memcpy (buf, tcfsheader.file_key, KEYSIZE);

		newbuf=kmalloc (TCFS_HBLOCK_SIZE, GFP_KERNEL);
		if (!newbuf) {
			return -ENOMEM;
		}

		memset (newbuf, 0, TCFS_HBLOCK_SIZE);
		toread=TCFS_HBLOCK_SIZE;
	} else {
		newbuf=buffer;
	}

	do {
		unsigned int quit=0;

		if (ks!=TCFS_NO_SECURE) {
			newpos=tcfs_new_offset (offset,
				(loff_t *)&blockstart,
				(loff_t *)&blockoff);
			block_number=(blockstart/TCFS_HBLOCK_SIZE);
			memcpy (buf+KEYSIZE, (char *)&block_number,
				sizeof(unsigned int));
			tcfs_hash_block (buf, KEYSIZE+sizeof(unsigned int),
				bkey);
			bk=tcfs_init_key(bkey,
				inode->u.tcfs_i.tcfsheader.engine);
		}

		/* Set up arguments and perform rpc call */
		tcfs_readreq_setup(&rqst, TCFS_FH(dentry), blockstart,
				newbuf, toread);
		result = rpc_call(TCFS_CLIENT(inode), TCFSPROC_READ,
				&rqst.ra_args, &rqst.ra_res, flags);

		/*
		 * Even if we had a partial success we can't mark the page
		 * cache valid.
		 */
		if (result < 0) {
			if (result == -EISDIR)
				result = -EINVAL;
			goto io_error;
		}

		if (ks!=TCFS_NO_SECURE) {
			char *checkbuf=newbuf;

			while (*checkbuf==0 && (checkbuf-newbuf)<result) {
				checkbuf++;
			}

			if ((checkbuf-newbuf) != result) {
				int error=0;

				tcfs_decrypt (newbuf, TCFS_HBLOCK_SIZE, bk,
					inode->u.tcfs_i.tcfsheader.engine);
				error=tcfs_check_hash (newbuf,
						bk,
						newbuf+TCFS_BLOCK_SIZE,
						TCFS_BLOCK_SIZE);
				if (error<0) {
					printk ("TCFS Warning: file was modified on the server\n");
					result=-EIO;

					goto io_error;
				}
			}

			howmanycopy = result - blockoff - TCFS_HASH_SIZE;

			if ((offset+howmanycopy) > inode->i_size) {
				howmanycopy=inode->i_size-offset;
				quit=1;
			}

			if ((count-howmanycopy)<0) /* out of page bound */
				howmanycopy=count; /* fill the page */

			memcpy (buffer, newbuf+blockoff, howmanycopy);
		} else {
			howmanycopy=result;

			if (howmanycopy<toread)
				quit=1;
		}

		refresh = 1;
		count  -= howmanycopy;
		offset += howmanycopy;
		buffer += howmanycopy;

		if (quit)
			break;
	} while (count>0);

	if (count<0)
		count=0;

	memset(buffer, 0, count);
	set_bit(PG_uptodate, &page->flags);
	result = 0;

io_error:
	/* Note: we don't refresh if the call returned error */
	if (refresh && result >= 0) {
		tcfs_refresh_inode(inode, &rqst.ra_fattr);
	}

	if (ks!=TCFS_NO_SECURE && newbuf)
		kfree (newbuf);

	/* N.B. Use tcfs_unlock_page here? */
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);

	return result;
}
/*
 * This is the callback from RPC telling us whether a reply was
 * received or some error occurred (timeout or socket shutdown).
 */
static void
tcfs_readpage_result(struct rpc_task *task)
{
	struct tcfs_rreq	*req = (struct tcfs_rreq *) task->tk_calldata;
	struct page	*page = req->ra_page;
	unsigned long	address = page_address(page);
	int		result = task->tk_status;
	static int	succ = 0, fail = 0;

	dprintk("TCFS: %4d received callback for page %lx, result %d\n",
			task->tk_pid, address, result);

	if (result >= 0) {
		result = req->ra_res.count;
		if (result < PAGE_SIZE) {
			memset((char *) address + result, 0, PAGE_SIZE - result);
		}
		tcfs_refresh_inode(req->ra_inode, &req->ra_fattr);
		set_bit(PG_uptodate, &page->flags);
		succ++;
	} else {
		set_bit(PG_error, &page->flags);
		fail++;
		dprintk("TCFS: %d successful reads, %d failures\n", succ, fail);
	}
	/* N.B. Use tcfs_unlock_page here? */
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);

	free_page(address);

	rpc_release_task(task);
	kfree(req);
}

static inline int
tcfs_readpage_async(struct dentry *dentry, struct inode *inode,
			struct page *page)
{
	unsigned long address = page_address(page);
	struct tcfs_rreq	*req;
	int		result = -1, flags;

	dprintk("TCFS: tcfs_readpage_async(%p)\n", page);
	if (TCFS_CONGESTED(inode))
		goto out_defer;

	/* N.B. Do we need to test? Never called for swapfile inode */
	flags = RPC_TASK_ASYNC | (IS_SWAPFILE(inode)? TCFS_RPC_SWAPFLAGS : 0);
	req = (struct tcfs_rreq *) rpc_allocate(flags, sizeof(*req));
	if (!req)
		goto out_defer;

	/* Initialize request */
	/* N.B. Will the dentry remain valid for life of request? */
	tcfs_readreq_setup(req, TCFS_FH(dentry), page->offset,
				(void *) address, PAGE_SIZE);
	req->ra_inode = inode;
	req->ra_page = page; /* count has been incremented by caller */

	/* Start the async call */
	dprintk("TCFS: executing async READ request.\n");
	result = rpc_do_call(TCFS_CLIENT(inode), TCFSPROC_READ,
				&req->ra_args, &req->ra_res, flags,
				tcfs_readpage_result, req);
	if (result < 0)
		goto out_free;
	result = 0;
out:
	return result;

out_defer:
	dprintk("TCFS: deferring async READ request.\n");
	goto out;
out_free:
	dprintk("TCFS: failed to enqueue async READ request.\n");
	kfree(req);
	goto out;
}

/*
 * Read a page over TCFS.
 * We read the page synchronously in the following cases:
 *  -	The file is a swap file. Swap-ins are always sync operations,
 *	so there's no need bothering to make async reads 100% fail-safe.
 *  -	The TCFS rsize is smaller than PAGE_SIZE. We could kludge our way
 *	around this by creating several consecutive read requests, but
 *	that's hardly worth it.
 *  -	The error flag is set for this page. This happens only when a
 *	previous async read operation failed.
 *  -	The server is congested.
 */
int
tcfs_readpage(struct file *file, struct page *page)
{
	struct dentry *dentry = file->f_dentry;
	struct inode *inode = dentry->d_inode;
	int		error;

	dprintk("TCFS: tcfs_readpage (%p %ld@%ld)\n",
		page, PAGE_SIZE, page->offset);
	atomic_inc(&page->count);
	set_bit(PG_locked, &page->flags);

	/*
	 * Try to flush any pending writes to the file..
	 *
	 * NOTE! Because we own the page lock, there cannot
	 * be any new pending writes generated at this point
	 * for this page (other pages can be written to).
	 */
	error = tcfs_wb_page(inode, page);
	if (error)
		goto out_error;

#if 0 /* TCFS does not yet support async read (and write) */
	error = -1;
	if (!IS_SWAPFILE(inode) && !PageError(page) &&
	    TCFS_SERVER(inode)->rsize >= PAGE_SIZE)
		error = tcfs_readpage_async(dentry, inode, page);
	if (error >= 0)
		goto out;
#endif

	error = tcfs_readpage_sync(dentry, inode, page);
	if (error < 0 && IS_SWAPFILE(inode))
		printk("Aiee.. tcfs swap-in of page failed!\n");
	goto out_free;

out_error:
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);
out_free:
	free_page(page_address(page));
out:
	return error;
}
