/*
 *  linux/fs/tcfs/proc.c
 *
 *  Copyright (C) 1992, 1993, 1994  Rick Sladkey
 *
 *  OS-independent tcfs remote procedure call functions
 *
 *  Tuned by Alan Cox <A.Cox@swansea.ac.uk> for >3K buffers
 *  so at last we can have decent(ish) throughput off a 
 *  Sun server.
 *
 *  Coding optimized and cleaned up by Florian La Roche.
 *  Note: Error returns are optimized for TCFS_OK, which isn't translated via
 *  tcfs_stat_to_errno(), but happens to be already the right return code.
 *
 *  FixMe: We ought to define a sensible small max size for
 *  things like getattr that are tiny packets and use the
 *  old get_free_page stuff with it.
 *
 *  Also, the code currently doesn't check the size of the packet, when
 *  it decodes the packet.
 *
 *  Feel free to fix it and mail me the diffs if it worries you.
 *
 *  Completely rewritten to support the new RPC call interface;
 *  rewrote and moved the entire XDR stuff to xdr.c
 *  --Olaf Kirch June 1996
 */

#define TCFS_NEED_XDR_TYPES

#include <linux/param.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/utsname.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/in.h>
#include <linux/pagemap.h>
#include <linux/sunrpc/clnt.h>

#include <linux/tcfs_fs.h>
#include <asm/segment.h>

extern int tcfs_debug;
extern u_int tcfs_numread, tcfs_numwrite;

#define RPCDBG_FACILITY	RPCDBG_CALL

#ifdef TCFS_DEBUG
# define TCFSDBG_FACILITY	TCFSDBG_PROC
#endif


/*
 * One function for each procedure in the TCFS protocol.
 */
int
tcfs_proc_getattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fattr *fattr)
{
	int	status;

	dprintk("TCFS call  getattr\n");
	status = rpc_call(server->client, TCFSPROC_GETATTR, fhandle, fattr, 0);
	dprintk("TCFS reply getattr\n");
	return status;
}

int
tcfs_proc_setattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_sattr *sattr, struct tcfs_fattr *fattr)
{
	struct tcfs_sattrargs	arg = { fhandle, sattr };
	int	status;

	dprintk("TCFS call  setattr\n");
	status = rpc_call(server->client, TCFSPROC_SETATTR, &arg, fattr, 0);
	dprintk("TCFS reply setattr\n");
	return status;
}

int
tcfs_proc_lookup(struct tcfs_server *server, struct tcfs_fh *dir, const char *name,
		    struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	struct tcfs_diropargs	arg = { dir, name };
	struct tcfs_diropok	res = { fhandle, fattr };
	int			status;

	dprintk("TCFS call  lookup %s\n", name);
	status = rpc_call(server->client, TCFSPROC_LOOKUP, &arg, &res, 0);
	dprintk("TCFS reply lookup: %d\n", status);
	return status;
}

int
tcfs_proc_readlink(struct tcfs_server *server, struct tcfs_fh *fhandle,
			void **p0, char **string, unsigned int *len,
			unsigned int maxlen)
{
	struct tcfs_readlinkres	res = { string, len, maxlen, NULL };
	int			status;

	dprintk("TCFS call  readlink\n");
	status = rpc_call(server->client, TCFSPROC_READLINK, fhandle, &res, 0);
	dprintk("TCFS reply readlink: %d\n", status);
	if (!status)
		*p0 = res.buffer;
	else if (res.buffer)
		kfree(res.buffer);
	return status;
}

int
tcfs_proc_read(struct tcfs_server *server, struct tcfs_fh *fhandle, int swap,
			  unsigned long offset, unsigned int count,
			  void *buffer, struct tcfs_fattr *fattr)
{
	struct tcfs_readargs	arg = { fhandle, offset, count, buffer };
	struct tcfs_readres	res = { fattr, count };
	int			status;

	tcfs_numread++;
	dprintk("TCFS call  read %d @ %ld\n", count, offset);
	status = rpc_call(server->client, TCFSPROC_READ, &arg, &res,
			swap? TCFS_RPC_SWAPFLAGS : 0);
	dprintk("TCFS reply read: %d\n", status);
	return status;
}

int
tcfs_proc_write(struct tcfs_server *server, struct tcfs_fh *fhandle, int swap,
			unsigned long offset, unsigned int count,
			const void *buffer, struct tcfs_fattr *fattr)
{
	struct tcfs_writeargs	arg = { fhandle, offset, count, buffer };
	int			status;

	tcfs_numwrite++;
	dprintk("TCFS call  write %d @ %ld\n", count, offset);
	status = rpc_call(server->client, TCFSPROC_WRITE, &arg, fattr,
			swap? (RPC_TASK_SWAPPER|RPC_TASK_ROOTCREDS) : 0);
	dprintk("TCFS reply read: %d\n", status);
	return status < 0? status : count;
}

int
tcfs_proc_create(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, struct tcfs_sattr *sattr,
			struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	struct tcfs_createargs	arg = { dir, name, sattr };
	struct tcfs_diropok	res = { fhandle, fattr };
	int			status;

	dprintk("TCFS call  create %s\n", name);
	status = rpc_call(server->client, TCFSPROC_CREATE, &arg, &res, 0);
	dprintk("TCFS reply create: %d\n", status);
	return status;
}

int
tcfs_proc_remove(struct tcfs_server *server, struct tcfs_fh *dir, const char *name)
{
	struct tcfs_diropargs	arg = { dir, name };
	int			status;

	dprintk("TCFS call  remove %s\n", name);
	status = rpc_call(server->client, TCFSPROC_REMOVE, &arg, NULL, 0);
	dprintk("TCFS reply remove: %d\n", status);
	return status;
}

int
tcfs_proc_rename(struct tcfs_server *server,
		struct tcfs_fh *old_dir, const char *old_name,
		struct tcfs_fh *new_dir, const char *new_name)
{
	struct tcfs_renameargs	arg = { old_dir, old_name, new_dir, new_name };
	int			status;

	dprintk("TCFS call  rename %s -> %s\n", old_name, new_name);
	status = rpc_call(server->client, TCFSPROC_RENAME, &arg, NULL, 0);
	dprintk("TCFS reply rename: %d\n", status);
	return status;
}

int
tcfs_proc_link(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fh *dir, const char *name)
{
	struct tcfs_linkargs	arg = { fhandle, dir, name };
	int			status;

	dprintk("TCFS call  link %s\n", name);
	status = rpc_call(server->client, TCFSPROC_LINK, &arg, NULL, 0);
	dprintk("TCFS reply link: %d\n", status);
	return status;
}

int
tcfs_proc_symlink(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, const char *path,
			struct tcfs_sattr *sattr)
{
	struct tcfs_symlinkargs	arg = { dir, name, path, sattr };
	int			status;

	dprintk("TCFS call  symlink %s -> %s\n", name, path);
	status = rpc_call(server->client, TCFSPROC_SYMLINK, &arg, NULL, 0);
	dprintk("TCFS reply symlink: %d\n", status);
	return status;
}

int
tcfs_proc_mkdir(struct tcfs_server *server, struct tcfs_fh *dir,
			const char *name, struct tcfs_sattr *sattr,
			struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	struct tcfs_createargs	arg = { dir, name, sattr };
	struct tcfs_diropok	res = { fhandle, fattr };
	int			status;

	dprintk("TCFS call  mkdir %s\n", name);
	status = rpc_call(server->client, TCFSPROC_MKDIR, &arg, &res, 0);
	dprintk("TCFS reply mkdir: %d\n", status);
	return status;
}

int
tcfs_proc_rmdir(struct tcfs_server *server, struct tcfs_fh *dir, const char *name)
{
	struct tcfs_diropargs	arg = { dir, name };
	int			status;

	dprintk("TCFS call  rmdir %s\n", name);
	status = rpc_call(server->client, TCFSPROC_RMDIR, &arg, NULL, 0);
	dprintk("TCFS reply rmdir: %d\n", status);
	return status;
}

/*
 * The READDIR implementation is somewhat hackish - we pass a temporary
 * buffer to the encode function, which installs it in the receive
 * iovec. The dirent buffer itself is passed in the result struct.
 */
int
tcfs_proc_readdir(struct tcfs_server *server, struct tcfs_fh *fhandle,
			u32 cookie, unsigned int size, __u32 *entry)
{
	struct tcfs_readdirargs	arg;
	struct tcfs_readdirres	res;
	void *			buffer;
	unsigned int		buf_size = PAGE_SIZE;
	int			status;

	/* First get a temp buffer for the readdir reply */
	/* N.B. does this really need to be cleared? */
	status = -ENOMEM;
	buffer = (void *) get_free_page(GFP_KERNEL);
	if (!buffer)
		goto out;

	/*
	 * Calculate the effective size the buffer.  To make sure
	 * that the returned data will fit into the user's buffer,
	 * we decrease the buffer size as necessary.
	 *
	 * Note: TCFS returns three __u32 values for each entry,
	 * and we assume that the data is packed into the user
	 * buffer with the same efficiency. 
	 */
	if (size < buf_size)
		buf_size = size;
	if (server->rsize < buf_size)
		buf_size = server->rsize;
#if 0
printk("tcfs_proc_readdir: user size=%d, rsize=%d, buf_size=%d\n",
size, server->rsize, buf_size);
#endif

	arg.fh = fhandle;
	arg.cookie = cookie;
	arg.buffer = buffer;
	arg.bufsiz = buf_size;
	res.buffer = entry;
	res.bufsiz = size;

	dprintk("TCFS call  readdir %d\n", cookie);

	status = rpc_call(server->client, TCFSPROC_READDIR, &arg, &res, 0);

	dprintk("TCFS reply readdir: %d\n", status);
	free_page((unsigned long) buffer);
out:
	return status;
}

int
tcfs_proc_statfs(struct tcfs_server *server, struct tcfs_fh *fhandle,
			struct tcfs_fsinfo *info)
{
	int	status;

	dprintk("TCFS call  statfs\n");
	status = rpc_call(server->client, TCFSPROC_STATFS, fhandle, info, 0);
	dprintk("TCFS reply statfs: %d\n", status);
	return status;
}
