/*
 * linux/fs/tcfs/xdr.c
 *
 * XDR functions to encode/decode TCFS RPC arguments and results.
 *
 * Copyright (C) 1992, 1993, 1994  Rick Sladkey
 * Copyright (C) 1996 Olaf Kirch
 */

#define TCFS_NEED_XDR_TYPES

#include <linux/param.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/utsname.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/in.h>
#include <linux/pagemap.h>
#include <linux/tcfs_fs.h>
#include <linux/proc_fs.h>
#include <linux/sunrpc/clnt.h>

extern int tcfs_debug;
#define RPCDBG_FACILITY	RPCDBG_CALL
/* Uncomment this to support servers requiring longword lengths */
#define TCFS_PAD_WRITES 1

#define TCFSDBG_FACILITY		TCFSDBG_XDR
/* #define TCFS_PARANOIA 1 */

#define QUADLEN(len)		(((len) + 3) >> 2)
static int			tcfs_stat_to_errno(int stat);

/* Mapping from TCFS error code to "errno" error code. */
#define errno_TCFSERR_IO		EIO

/*
 * Declare the space requirements for TCFS arguments and replies as
 * number of 32bit-words
 */
#define TCFS_fhandle_sz		8
#define TCFS_sattr_sz		8
#define TCFS_filename_sz		1+(TCFS_MAXNAMLEN>>2)
#define TCFS_path_sz		1+(TCFS_MAXPATHLEN>>2)
#define TCFS_fattr_sz		17
#define TCFS_info_sz		5
#define TCFS_entry_sz		TCFS_filename_sz+3

#define TCFS_enc_void_sz		0
#define TCFS_diropargs_sz	TCFS_fhandle_sz+TCFS_filename_sz
#define TCFS_sattrargs_sz	TCFS_fhandle_sz+TCFS_sattr_sz
#define TCFS_readargs_sz		TCFS_fhandle_sz+3
#define TCFS_writeargs_sz	TCFS_fhandle_sz+4
#define TCFS_createargs_sz	TCFS_diropargs_sz+TCFS_sattr_sz
#define TCFS_renameargs_sz	TCFS_diropargs_sz+TCFS_diropargs_sz
#define TCFS_linkargs_sz		TCFS_fhandle_sz+TCFS_diropargs_sz
#define TCFS_symlinkargs_sz	TCFS_diropargs_sz+TCFS_path_sz+TCFS_sattr_sz
#define TCFS_readdirargs_sz	TCFS_fhandle_sz+2

#define TCFS_dec_void_sz		0
#define TCFS_attrstat_sz		1+TCFS_fattr_sz
#define TCFS_diropres_sz		1+TCFS_fhandle_sz+TCFS_fattr_sz
#define TCFS_readlinkres_sz	1+TCFS_path_sz
#define TCFS_readres_sz		1+TCFS_fattr_sz+1
#define TCFS_stat_sz		1
#define TCFS_readdirres_sz	1
#define TCFS_statfsres_sz	1+TCFS_info_sz

/*
 * Common TCFS XDR functions as inlines
 */
static inline u32 *
xdr_encode_fhandle(u32 *p, struct tcfs_fh *fhandle)
{
	*((struct tcfs_fh *) p) = *fhandle;
	return p + QUADLEN(sizeof(*fhandle));
}

static inline u32 *
xdr_decode_fhandle(u32 *p, struct tcfs_fh *fhandle)
{
	*fhandle = *((struct tcfs_fh *) p);
	return p + QUADLEN(sizeof(*fhandle));
}

static inline u32 *
xdr_decode_string2(u32 *p, char **string, unsigned int *len,
			unsigned int maxlen)
{
	*len = ntohl(*p++);
	if (*len > maxlen)
		return NULL;
	*string = (char *) p;
	return p + QUADLEN(*len);
}

static inline u32 *
xdr_decode_fattr(u32 *p, struct tcfs_fattr *fattr)
{
	fattr->type = (enum tcfs_ftype) ntohl(*p++);
	fattr->mode = ntohl(*p++);
	fattr->nlink = ntohl(*p++);
	fattr->uid = ntohl(*p++);
	fattr->gid = ntohl(*p++);
	fattr->size = ntohl(*p++);
	fattr->blocksize = ntohl(*p++);
	fattr->rdev = ntohl(*p++);
	fattr->blocks = ntohl(*p++);
	fattr->fsid = ntohl(*p++);
	fattr->fileid = ntohl(*p++);
	fattr->atime.seconds = ntohl(*p++);
	fattr->atime.useconds = ntohl(*p++);
	fattr->mtime.seconds = ntohl(*p++);
	fattr->mtime.useconds = ntohl(*p++);
	fattr->ctime.seconds = ntohl(*p++);
	fattr->ctime.useconds = ntohl(*p++);
	return p;
}

static inline u32 *
xdr_encode_sattr(u32 *p, struct tcfs_sattr *sattr)
{
	*p++ = htonl(sattr->mode);
	*p++ = htonl(sattr->uid);
	*p++ = htonl(sattr->gid);
	*p++ = htonl(sattr->size);
	*p++ = htonl(sattr->atime.seconds);
	*p++ = htonl(sattr->atime.useconds);
	*p++ = htonl(sattr->mtime.seconds);
	*p++ = htonl(sattr->mtime.useconds);
	return p;
}

/*
 * TCFS encode functions
 */
/*
 * Encode void argument
 */
static int
tcfs_xdr_enc_void(struct rpc_rqst *req, u32 *p, void *dummy)
{
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode file handle argument
 * GETATTR, READLINK, STATFS
 */
static int
tcfs_xdr_fhandle(struct rpc_rqst *req, u32 *p, struct tcfs_fh *fh)
{
	p = xdr_encode_fhandle(p, fh);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode SETATTR arguments
 */
static int
tcfs_xdr_sattrargs(struct rpc_rqst *req, u32 *p, struct tcfs_sattrargs *args)
{
	p = xdr_encode_fhandle(p, args->fh);
	p = xdr_encode_sattr(p, args->sattr);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode directory ops argument
 * LOOKUP, REMOVE, RMDIR
 */
static int
tcfs_xdr_diropargs(struct rpc_rqst *req, u32 *p, struct tcfs_diropargs *args)
{
	p = xdr_encode_fhandle(p, args->fh);
	p = xdr_encode_string(p, args->name);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Arguments to a READ call. Since we read data directly into the page
 * cache, we also set up the reply iovec here so that iov[1] points
 * exactly to the page we want to fetch.
 */
static int
tcfs_xdr_readargs(struct rpc_rqst *req, u32 *p, struct tcfs_readargs *args)
{
	struct rpc_auth	*auth = req->rq_task->tk_auth;
	int		replen, buflen;

	p = xdr_encode_fhandle(p, args->fh);
	*p++ = htonl(args->offset);
	*p++ = htonl(args->count);
	*p++ = htonl(args->count);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);

#if 1
	/* set up reply iovec */
	replen = (RPC_REPHDRSIZE + auth->au_rslack + TCFS_readres_sz) << 2;
	buflen = req->rq_rvec[0].iov_len;
	req->rq_rvec[0].iov_len  = replen;
	req->rq_rvec[1].iov_base = args->buffer;
	req->rq_rvec[1].iov_len  = args->count;
	req->rq_rvec[2].iov_base = (u8 *) req->rq_rvec[0].iov_base + replen;
	req->rq_rvec[2].iov_len  = buflen - replen;
	req->rq_rlen = args->count + buflen;
	req->rq_rnr = 3;
#else
	replen = (RPC_REPHDRSIZE + auth->au_rslack + TCFS_readres_sz) << 2;
	req->rq_rvec[0].iov_len  = replen;
#endif

	return 0;
}

/*
 * Decode READ reply
 */
static int
tcfs_xdr_readres(struct rpc_rqst *req, u32 *p, struct tcfs_readres *res)
{
	struct iovec *iov = req->rq_rvec;
	int	status, count, recvd, hdrlen;

	dprintk("RPC:      readres OK status %lx\n", (long)ntohl(*p));
	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	p = xdr_decode_fattr(p, res->fattr);

	count = ntohl(*p++);
	hdrlen = (u8 *) p - (u8 *) iov->iov_base;
	recvd = req->rq_rlen - hdrlen;
	if (p != iov[2].iov_base) {
		/* Unexpected reply header size. Punt.
		 * XXX: Move iovec contents to align data on page
		 * boundary and adjust RPC header size guess */
		printk("TCFS: Odd RPC header size in read reply: %d\n", hdrlen);
		return -errno_TCFSERR_IO;
	}
	if (count > recvd) {
		printk("TCFS: server cheating in read reply: "
			"count %d > recvd %d\n", count, recvd);
		count = recvd;
	}

	dprintk("RPC:      readres OK count %d\n", count);
	if (count < res->count)
		memset((u8 *)(iov[1].iov_base+count), 0, res->count-count);

	return count;
}


/*
 * Write arguments. Splice the buffer to be written into the iovec.
 */
static int
tcfs_xdr_writeargs(struct rpc_rqst *req, u32 *p, struct tcfs_writeargs *args)
{
	u32 count = args->count;

	p = xdr_encode_fhandle(p, args->fh);
	*p++ = htonl(args->offset);
	*p++ = htonl(args->offset);
	*p++ = htonl(count);
	*p++ = htonl(count);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);

	req->rq_svec[1].iov_base = (void *) args->buffer;
	req->rq_svec[1].iov_len = count;
	req->rq_slen += count;
	req->rq_snr = 2;

#ifdef TCFS_PAD_WRITES
	/*
	 * Some old servers require that the message length
	 * be a multiple of 4, so we pad it here if needed.
	 */
	count = ((count + 3) & ~3) - count;
	if (count) {
#if 0
printk("tcfs_writeargs: padding write, len=%d, slen=%d, pad=%d\n",
req->rq_svec[1].iov_len, req->rq_slen, count);
#endif
		req->rq_svec[2].iov_base = (void *) "\0\0\0";
		req->rq_svec[2].iov_len  = count;
		req->rq_slen += count;
		req->rq_snr = 3;
	}
#endif

	return 0;
}

/*
 * Encode create arguments
 * CREATE, MKDIR
 */
static int
tcfs_xdr_createargs(struct rpc_rqst *req, u32 *p, struct tcfs_createargs *args)
{
	p = xdr_encode_fhandle(p, args->fh);
	p = xdr_encode_string(p, args->name);
	p = xdr_encode_sattr(p, args->sattr);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode RENAME arguments
 */
static int
tcfs_xdr_renameargs(struct rpc_rqst *req, u32 *p, struct tcfs_renameargs *args)
{
	p = xdr_encode_fhandle(p, args->fromfh);
	p = xdr_encode_string(p, args->fromname);
	p = xdr_encode_fhandle(p, args->tofh);
	p = xdr_encode_string(p, args->toname);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode LINK arguments
 */
static int
tcfs_xdr_linkargs(struct rpc_rqst *req, u32 *p, struct tcfs_linkargs *args)
{
	p = xdr_encode_fhandle(p, args->fromfh);
	p = xdr_encode_fhandle(p, args->tofh);
	p = xdr_encode_string(p, args->toname);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode SYMLINK arguments
 */
static int
tcfs_xdr_symlinkargs(struct rpc_rqst *req, u32 *p, struct tcfs_symlinkargs *args)
{
	p = xdr_encode_fhandle(p, args->fromfh);
	p = xdr_encode_string(p, args->fromname);
	p = xdr_encode_string(p, args->topath);
	p = xdr_encode_sattr(p, args->sattr);
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);
	return 0;
}

/*
 * Encode arguments to readdir call
 */
static int
tcfs_xdr_readdirargs(struct rpc_rqst *req, u32 *p, struct tcfs_readdirargs *args)
{
	struct rpc_task	*task = req->rq_task;
	struct rpc_auth	*auth = task->tk_auth;
	u32		bufsiz = args->bufsiz;
	int		replen;

	/*
	 * Some servers (e.g. HP OS 9.5) seem to expect the buffer size
	 * to be in longwords ... check whether to convert the size.
	 */
	if (task->tk_client->cl_flags & TCFS_CLNTF_BUFSIZE)
		bufsiz = bufsiz >> 2;

	p = xdr_encode_fhandle(p, args->fh);
	*p++ = htonl(args->cookie);
	*p++ = htonl(bufsiz); /* see above */
	req->rq_slen = xdr_adjust_iovec(req->rq_svec, p);

	/* set up reply iovec */
	replen = (RPC_REPHDRSIZE + auth->au_rslack + TCFS_readdirres_sz) << 2;
	/*
	dprintk("RPC: readdirargs: slack is 4 * (%d + %d + %d) = %d\n",
		RPC_REPHDRSIZE, auth->au_rslack, TCFS_readdirres_sz, replen);
	 */
	req->rq_rvec[0].iov_len  = replen;
	req->rq_rvec[1].iov_base = args->buffer;
	req->rq_rvec[1].iov_len  = args->bufsiz;
	req->rq_rlen = replen + args->bufsiz;
	req->rq_rnr = 2;

	/*
	dprintk("RPC:      readdirargs set up reply vec:\n");
	dprintk("          rvec[0] = %p/%d\n",
			req->rq_rvec[0].iov_base,
			req->rq_rvec[0].iov_len);
	dprintk("          rvec[1] = %p/%d\n",
			req->rq_rvec[1].iov_base,
			req->rq_rvec[1].iov_len);
	 */

	return 0;
}

/*
 * Decode the result of a readdir call. We decode the result in place
 * to avoid a malloc of TCFS_MAXNAMLEN+1 for each file name.
 * After decoding, the layout in memory looks like this:
 *	entry1 entry2 ... entryN <space> stringN ... string2 string1
 * Each entry consists of three __u32 values, the same space as TCFS uses.
 * Note that the strings are not null-terminated so that the entire number
 * of entries returned by the server should fit into the buffer.
 */
static int
tcfs_xdr_readdirres(struct rpc_rqst *req, u32 *p, struct tcfs_readdirres *res)
{
	struct iovec		*iov = req->rq_rvec;
	int			 status, nr;
	char			*string, *start;
	u32			*end, *entry, len, fileid, cookie;

	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	if ((void *) p != ((u8 *) iov->iov_base+iov->iov_len)) {
		/* Unexpected reply header size. Punt. */
		printk("TCFS: Odd RPC header size in readdirres reply\n");
		return -errno_TCFSERR_IO;
	}

	/* Get start and end address of XDR data */
	p   = (u32 *) iov[1].iov_base;
	end = (u32 *) ((u8 *) p + iov[1].iov_len);

	/* Get start and end of dirent buffer */
	entry  = (u32 *) res->buffer;
	start  = (char *) res->buffer;
	string = (char *) res->buffer + res->bufsiz;
	for (nr = 0; *p++; nr++) {
		fileid = ntohl(*p++);

		len = ntohl(*p++);
		/*
		 * Check whether the server has exceeded our reply buffer,
		 * and set a flag to convert the size to longwords.
		 */
		if ((p + QUADLEN(len) + 3) > end) {
			struct rpc_clnt *clnt = req->rq_task->tk_client;
			printk(KERN_WARNING
				"TCFS: server %s, readdir reply truncated\n",
				clnt->cl_server);
			printk(KERN_WARNING "TCFS: nr=%d, slots=%d, len=%d\n",
				nr, (end - p), len);
			clnt->cl_flags |= TCFS_CLNTF_BUFSIZE;
			break;
		}
		if (len > TCFS_MAXNAMLEN) {
			printk("TCFS: giant filename in readdir (len %x)!\n",
						len);
			return -errno_TCFSERR_IO;
		}
		string -= len;
		if ((void *) (entry+3) > (void *) string) {
			/* 
			 * This error is impossible as long as the temp
			 * buffer is no larger than the user buffer. The 
			 * current packing algorithm uses the same amount
			 * of space in the user buffer as in the XDR data,
			 * so it's guaranteed to fit.
			 */
			printk("TCFS: incorrect buffer size in %s!\n",
				__FUNCTION__);
			break;
		}

		memmove(string, p, len);
		p += QUADLEN(len);
		cookie = ntohl(*p++);
		/*
		 * To make everything fit, we encode the length, offset,
		 * and eof flag into 32 bits. This works for filenames
		 * up to 32K and PAGE_SIZE up to 64K.
		 */
		status = !p[0] && p[1] ? (1 << 15) : 0; /* eof flag */
		*entry++ = fileid;
		*entry++ = cookie;
		*entry++ = ((string - start) << 16) | status | (len & 0x7FFF);
	}
#ifdef TCFS_PARANOIA
printk("tcfs_xdr_readdirres: %d entries, ent sp=%d, str sp=%d\n",
nr, ((char *) entry - start), (start + res->bufsiz - string));
#endif
	return nr;
}

/*
 * TCFS XDR decode functions
 */
/*
 * Decode void reply
 */
static int
tcfs_xdr_dec_void(struct rpc_rqst *req, u32 *p, void *dummy)
{
	return 0;
}

/*
 * Decode simple status reply
 */
static int
tcfs_xdr_stat(struct rpc_rqst *req, u32 *p, void *dummy)
{
	int	status;

	if ((status = ntohl(*p++)) != 0)
		status = -tcfs_stat_to_errno(status);
	return status;
}

/*
 * Decode attrstat reply
 * GETATTR, SETATTR, WRITE
 */
static int
tcfs_xdr_attrstat(struct rpc_rqst *req, u32 *p, struct tcfs_fattr *fattr)
{
	int	status;

	dprintk("RPC:      attrstat status %lx\n", (long)ntohl(*p));
	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	xdr_decode_fattr(p, fattr);
	dprintk("RPC:      attrstat OK type %d mode %o dev %x ino %x\n",
		fattr->type, fattr->mode, fattr->fsid, fattr->fileid);
	return 0;
}

/*
 * Decode diropres reply
 * LOOKUP, CREATE, MKDIR
 */
static int
tcfs_xdr_diropres(struct rpc_rqst *req, u32 *p, struct tcfs_diropok *res)
{
	int	status;

	dprintk("RPC:      diropres status %lx\n", (long)ntohl(*p));
	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	p = xdr_decode_fhandle(p, res->fh);
	xdr_decode_fattr(p, res->fattr);
	dprintk("RPC:      diropres OK type %x mode %o dev %x ino %x\n",
		res->fattr->type, res->fattr->mode,
		res->fattr->fsid, res->fattr->fileid);
	return 0;
}

/*
 * Decode READLINK reply
 */
static int
tcfs_xdr_readlinkres(struct rpc_rqst *req, u32 *p, struct tcfs_readlinkres *res)
{
	int	status;

	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	xdr_decode_string2(p, res->string, res->lenp, res->maxlen);

	/* Caller takes over the buffer here to avoid extra copy */
	res->buffer = req->rq_task->tk_buffer;
	req->rq_task->tk_buffer = NULL;
	return 0;
}

/*
 * Decode STATFS reply
 */
static int
tcfs_xdr_statfsres(struct rpc_rqst *req, u32 *p, struct tcfs_fsinfo *res)
{
	int	status;

	if ((status = ntohl(*p++)))
		return -tcfs_stat_to_errno(status);
	res->tsize = ntohl(*p++);
	res->bsize = ntohl(*p++);
	res->blocks = ntohl(*p++);
	res->bfree = ntohl(*p++);
	res->bavail = ntohl(*p++);
	return 0;
}

/*
 * We need to translate between tcfs status return values and
 * the local errno values which may not be the same.
 */
static struct {
	int stat;
	int errno;
} tcfs_errtbl[] = {
	{ TCFS_OK,		0		},
	{ TCFSERR_PERM,		EPERM		},
	{ TCFSERR_NOENT,		ENOENT		},
	{ TCFSERR_IO,		errno_TCFSERR_IO	},
	{ TCFSERR_NXIO,		ENXIO		},
	{ TCFSERR_EAGAIN,	EAGAIN		},
	{ TCFSERR_ACCES,		EACCES		},
	{ TCFSERR_EXIST,		EEXIST		},
	{ TCFSERR_XDEV,		EXDEV		},
	{ TCFSERR_NODEV,		ENODEV		},
	{ TCFSERR_NOTDIR,	ENOTDIR		},
	{ TCFSERR_ISDIR,		EISDIR		},
	{ TCFSERR_INVAL,		EINVAL		},
	{ TCFSERR_FBIG,		EFBIG		},
	{ TCFSERR_NOSPC,		ENOSPC		},
	{ TCFSERR_ROFS,		EROFS		},
	{ TCFSERR_OPNOTSUPP,	EOPNOTSUPP	},
	{ TCFSERR_NAMETOOLONG,	ENAMETOOLONG	},
	{ TCFSERR_NOTEMPTY,	ENOTEMPTY	},
	{ TCFSERR_DQUOT,		EDQUOT		},
	{ TCFSERR_STALE,		ESTALE		},
#ifdef EWFLUSH
	{ TCFSERR_WFLUSH,	EWFLUSH		},
#endif
	{ -1,			EIO		}
};

static int
tcfs_stat_to_errno(int stat)
{
	int i;

	for (i = 0; tcfs_errtbl[i].stat != -1; i++) {
		if (tcfs_errtbl[i].stat == stat)
			return tcfs_errtbl[i].errno;
	}
	printk("tcfs_stat_to_errno: bad tcfs status return value: %d\n", stat);
	return tcfs_errtbl[i].errno;
}

#ifndef MAX
# define MAX(a, b)	(((a) > (b))? (a) : (b))
#endif

#define PROC(proc, argtype, restype)	\
    { "tcfs_" #proc,					\
      (kxdrproc_t) tcfs_xdr_##argtype,			\
      (kxdrproc_t) tcfs_xdr_##restype,			\
      MAX(TCFS_##argtype##_sz,TCFS_##restype##_sz) << 2	\
    }

static struct rpc_procinfo	tcfs_procedures[18] = {
    PROC(null,		enc_void,	dec_void),
    PROC(getattr,	fhandle,	attrstat),
    PROC(setattr,	sattrargs,	attrstat),
    PROC(root,		enc_void,	dec_void),
    PROC(lookup,	diropargs,	diropres),
    PROC(readlink,	fhandle,	readlinkres),
    PROC(read,		readargs,	readres),
    PROC(writecache,	enc_void,	dec_void),
    PROC(write,		writeargs,	attrstat),
    PROC(create,	createargs,	diropres),
    PROC(remove,	diropargs,	stat),
    PROC(rename,	renameargs,	stat),
    PROC(link,		linkargs,	stat),
    PROC(symlink,	symlinkargs,	stat),
    PROC(mkdir,		createargs,	diropres),
    PROC(rmdir,		diropargs,	stat),
    PROC(readdir,	readdirargs,	readdirres),
    PROC(statfs,	fhandle,	statfsres),
};

static struct rpc_version	tcfs_version2 = {
	2,
	sizeof(tcfs_procedures)/sizeof(tcfs_procedures[0]),
	tcfs_procedures
};

static struct rpc_version *	tcfs_version[] = {
	NULL,
	NULL,
	&tcfs_version2
};

struct rpc_program	tcfs_program = {
	"tcfs",
	TCFS_PROGRAM,
	sizeof(tcfs_version) / sizeof(tcfs_version[0]),
	tcfs_version,
	&tcfs_rpcstat,
};
